package com.icetech.park.service.report.itc.impl;

import cn.hutool.core.thread.ThreadUtil;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.ItcVersionEnum;
import com.icetech.cloudcenter.domain.request.itc.PropertyInfoRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.DeviceUpgradeHandle;
import com.icetech.park.service.handle.ItcCacheHandle;
import com.icetech.park.service.report.CallService;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

@Service("itcPropertyInfoServiceImpl")
@Slf4j
public class PropertyInfoServiceImpl extends AbstractService implements CallService<PropertyInfoRequest, Map<String, Object>> {

    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private DeviceUpgradeHandle deviceUpgradeHandle;
    @Autowired
    private ItcCacheHandle cacheHandle;
    @Autowired
    private RedisUtils redisUtils;

    @Override
    public P2cBaseResponse<Map<String, Object>> execute(TokenDeviceVo deviceInfo, P2cBaseRequest<PropertyInfoRequest> baseRequest) {
        PropertyInfoRequest propertyInfoRequest = baseRequest.getBizContent();

        //参数校验
        verifyParams(propertyInfoRequest);

        //版本号不存在
        String reportVersion = propertyInfoRequest.getVersion().trim();
        if (ItcVersionEnum.getIndex(reportVersion) == 0){
            return P2cBaseResponse.instance(baseRequest, CodeEnum.非法参数.getCode(), "版本号不存在");
        }
        CompletableFuture.runAsync(() -> setCache(deviceInfo, propertyInfoRequest, reportVersion));
        Map<String, Object> dataMap = new HashMap<>();
        dataMap.put("enexType", deviceInfo.getInandoutType());
        //设备属性发生变化时处理
        propertyModify(propertyInfoRequest, deviceInfo.getDeviceNo(), deviceInfo.getSource());

        return P2cBaseResponse.success(baseRequest, dataMap);

    }

    private void setCache(TokenDeviceVo deviceInfo, PropertyInfoRequest propertyInfoRequest, String reportVersion) {
        //如果和缓存的版本号不一致时，更新缓存版本号
        if (!reportVersion.equals(deviceInfo.getVersion())){
            deviceInfo.setVersion(reportVersion);
            cacheHandle.updateDeviceInfo(deviceInfo.getDeviceNo(), deviceInfo);
        }
        if (redisUtils.exists(RedisConstants.PROPERTY_GET_FLAG_PROFILE + deviceInfo.getDeviceNo())){
            redisUtils.set(RedisConstants.PROPERTY_GET_INFO_PROFILE + deviceInfo.getDeviceNo(),
                    propertyInfoRequest, 8L);
        }
    }

    /**
     * 相机属性发生变化时处理
     * @param propertyInfoRequest
     * @param serialNumber
     */
    private void propertyModify(PropertyInfoRequest propertyInfoRequest, String serialNumber, Integer protocolType) {
        ThreadUtil.execute(() -> {
            ObjectResponse<ParkDevice> deviceBySerialNumber = parkDeviceService.getDeviceBySerialNumber(serialNumber);
            ParkDevice parkDevice = deviceBySerialNumber.getData();
            //根据条件更新协议版本号和协议类型
            String protocolVer = propertyInfoRequest.getVersion();
            String productModel = propertyInfoRequest.getProductModel();
            String firmwareVersion = propertyInfoRequest.getFirmwareVersion();
            Integer connectionMode = propertyInfoRequest.getConnectionMode();
            Integer wirelessCarrier = propertyInfoRequest.getWirelessCarrier();
            String network = propertyInfoRequest.getNetwork();

            if (firmwareVersion != null){
                deviceUpgradeHandle.versionModify(serialNumber, firmwareVersion);
            }
            if (!protocolVer.equals(parkDevice.getProtocolVer())
                    || !protocolType.equals(parkDevice.getProtocolType())
                    || (productModel != null && !productModel.equals(parkDevice.getProductModel()))
                    || (firmwareVersion != null && !firmwareVersion.equals(parkDevice.getFirmwareVersion()))
                    || (connectionMode != null && !connectionMode.equals(parkDevice.getCallMethods()))
                    || (wirelessCarrier != null && !wirelessCarrier.equals(parkDevice.getWirelessCarrier()))
                    || (network != null && !network.equals(parkDevice.getNetwork()))
            ) {
                parkDevice.setProtocolVer(protocolVer);
                parkDevice.setProtocolType(protocolType);
                parkDevice.setProductModel(productModel);
                parkDevice.setFirmwareVersion(firmwareVersion);
                parkDevice.setCallMethods(connectionMode);
                parkDevice.setWirelessCarrier(wirelessCarrier);
                parkDevice.setNetwork(network);
                parkDeviceService.updateDevice(parkDevice);
                log.info("[设备属性上报接口] 更新版本号和协议类型完成, sn[{}], protocolType[{}], protocolVer[{}], productModel[{}], firmwareVersion[{}]",
                        serialNumber, protocolType, protocolVer, productModel, firmwareVersion);
            }
        });
    }
}
