package com.icetech.park.service.report.full.impl;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.request.OpenBrakeRequest;
import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.cloudcenter.domain.response.p2c.CarEnexResponse;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.Base64Tools;
import com.icetech.park.domain.dto.full.IvsResultDTO;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.down.full.impl.FullCloudGpIoOutServiceImpl;
import com.icetech.park.service.down.full.impl.FullCloudSerialDataServiceImpl;
import com.icetech.park.service.factory.SendServiceFactory;
import com.icetech.park.service.report.CallService;
import com.icetech.park.service.report.full.FullEventCarService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * 出入场事件实现
 *
 * @author wgq
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class FullEventCarServiceImpl implements FullEventCarService {

    private final CacheHandle cacheHandle;

    private final ObjectMapper objectMapper;

    private final FullCloudGpIoOutServiceImpl fullDownGpIoOutService;

    private final FullCloudSerialDataServiceImpl fullCloudSerialDataService;

    /**
     * 处理出入事件
     *
     * @param resultDTO 事件
     * @return status
     */
    @Override
    public P2cBaseResponse<?> eventCar(IvsResultDTO resultDTO, String messageId, String sn) {
        String channelCode = String.valueOf(resultDTO.getAlarmInfoPlate().getChannel());
        TokenDeviceVo deviceToken = cacheHandle.getDeviceInfo(sn);
        if (deviceToken == null) {
            log.info("获取设备通道信息失败！,channelCode:{},deviceToken:{}", channelCode, deviceToken);
            P2cBaseResponse<?> p2cBaseResponse = new P2cBaseResponse<>();
            p2cBaseResponse.setCode(CodeEnum.未关联通道信息.getCode());
            p2cBaseResponse.setMsg("获取设备通道信息失败!");
            return p2cBaseResponse;
        }
        P2cBaseRequest<Object> content = assembleContent(resultDTO, messageId, deviceToken);
        String cmd = content.getCmd();
        //找carEvent Bean
        CallService<Object, Object> callService = SendServiceFactory.getP2cBean(cmd, CallService.class);
        //出入场处理
        P2cBaseResponse<Object> response = callService.execute(deviceToken, content);
        log.info("出入场返回参数：{}", response);
        if (response.getCode() != 200) {
            P2cBaseResponse<Object> resp = new P2cBaseResponse<>();
            resp.setCode(CodeEnum.服务器异常.getCode());
            return resp;
        }
        CompletableFuture.runAsync(ThreadUtils.wrapTrace(() -> {
            try {
                //异步处理 开闸
                brakeIo(response, deviceToken);
                //异步处理 屏显 语音
                brakeIoExecute(response, deviceToken);
            } catch (Exception e) {
                log.error("开闸或屏显语音信息失败", e);
            }
        }));
        return response;
    }

    /**
     * 下发开闸信息
     *
     * @param deviceToken 通道信息
     */
    private void brakeIo(P2cBaseResponse<Object> response, TokenDeviceVo deviceToken) {
        CarEnexResponse carEnexResponse = (CarEnexResponse) response.getData();
        if (carEnexResponse.getOpenFlag() != 1) {
            log.info("不下发开闸信息！");
            return;
        }
        log.info("下发开闸信息");
        OpenBrakeRequest open = new OpenBrakeRequest();
        open.setParkCode(deviceToken.getParkCode());
        open.setAisleCode(deviceToken.getDeviceNo());
        fullDownGpIoOutService.requestOpenBrake(open, deviceToken.getParkId());
    }

    /**
     * 下发语音屏显
     *
     * @param deviceToken 通道信息
     */
    private void brakeIoExecute(P2cBaseResponse<Object> response, TokenDeviceVo deviceToken) {
        CarEnexResponse carEnexResponse = (CarEnexResponse) response.getData();
        log.info("下发语音：{}，屏显：{}", carEnexResponse.getSay(), carEnexResponse.getShow());
        HintRequest hintRequest = new HintRequest();
        hintRequest.setShow(carEnexResponse.getShow());
        hintRequest.setShowType(carEnexResponse.getShowType());
        hintRequest.setSay(carEnexResponse.getSay());
        hintRequest.setExtendShow(carEnexResponse.getExtendShow());
        fullCloudSerialDataService.execute(deviceToken.getParkId(), deviceToken.getParkCode(), deviceToken.getDeviceNo(), hintRequest);
    }


    /**
     * 组装Content信息
     *
     * @param ivsResultDTO content
     * @param channel      通道信息
     * @return map
     */
    public P2cBaseRequest<Object> assembleContent(IvsResultDTO ivsResultDTO, String messageId, TokenDeviceVo channel) {
        P2cBaseRequest<Object> request = new P2cBaseRequest<>();
        // 出入口类型，1入口 2出口
        if (channel.getInandoutType() == 1) {
            request.setCmd("car_enter");
            request.setBizContent(objectMapper.convertValue(assembleBizContent(ivsResultDTO, channel), new TypeReference<CarEnterRequest>() {
            }));
        } else {
            request.setCmd("car_exit");
            request.setBizContent(objectMapper.convertValue(assembleBizContent(ivsResultDTO, channel), new TypeReference<CarExitRequest>() {
            }));
        }
        request.setMessageId(messageId);
        return request;
    }

    /**
     * 组装BizContent
     *
     * @param payload     出入场参数
     * @param deviceToken 通道信息
     * @return map
     */
    public static Map<String, Object> assembleBizContent(IvsResultDTO payload, TokenDeviceVo deviceToken) {
        Map<String, Object> bizContent = new HashMap<>(16);
        bizContent.put("reportTime", System.currentTimeMillis());
        bizContent.put("openFlag", 0);
        bizContent.put("type",1);
        //取 payload.getAlarmInfoPlate().getResult().getPlateResult().getIsOffline()   但是只支持在线就默认1
        bizContent.put("property", 1);
        bizContent.put("plateNum", getPlateNum(Base64Tools.decodeBase64String(payload.getAlarmInfoPlate().getResult().getPlateResult().getLicense())));
        bizContent.put("triggerType", 2);
        bizContent.put("plateColor", getPlateColor(payload.getAlarmInfoPlate().getResult().getPlateResult().getColorType()));
        bizContent.put("shamFlag", payload.getAlarmInfoPlate().getResult().getPlateResult().getIsFakePlate() == 0 ? 1 : 0);
        bizContent.put("carType", 1);
        bizContent.put("reliability", 28);
        bizContent.put("carColor", getCarColor(payload.getAlarmInfoPlate().getResult().getPlateResult().getCarColor()));
        bizContent.put("maxImage", ExtractContentAfterDomain.extractContentAfterDomain(Base64Tools.decodeBase64String(payload.getAlarmInfoPlate().getResult().getPlateResult().getImagePath()), "aliyuncs.com/"));
        bizContent.put("smallImage", ExtractContentAfterDomain.extractContentAfterDomain(Base64Tools.decodeBase64String(payload.getAlarmInfoPlate().getResult().getPlateResult().getImagePath()), "aliyuncs.com/"));
        if (deviceToken.getInandoutType() == 1) {
            bizContent.put("enterTime", payload.getAlarmInfoPlate().getResult().getPlateResult().getTimeStamp().getTimeval().getSec());
        } else {
            bizContent.put("exitTime", payload.getAlarmInfoPlate().getResult().getPlateResult().getTimeStamp().getTimeval().getSec());
        }
        return bizContent;
    }

    /**
     * 车牌颜色转换
     *
     * @param colorValue 上报参数
     * @return 车牌颜色
     * 蓝色; 黄色; 白色; 黑色; 绿色; 黄绿
     */
    private static String getPlateColor(int colorValue) {
        switch (colorValue) {
            case 2:
                return "黄色";
            case 3:
                return "白色";
            case 4:
                return "黑色";
            case 5:
                return "绿色";
            case 6:
                return "黄绿";
            default:
                return "蓝色";
        }
    }

    /**
     * 车辆颜色
     *
     * @param carColor int
     * @return string
     */
    private static String getCarColor(int carColor) {
        switch (carColor) {
            case 0:
                return "白色";
            case 1:
                return "银色";
            case 2:
                return "黄色";
            case 3:
                return "粉色";
            case 4:
                return "红色";
            case 5:
                return "绿色";
            case 6:
                return "蓝绿";
            case 7:
                return "棕绿";
            case 8:
                return "黑绿";
            case 9:
                return "灰绿";
            default:
                return "未知";
        }
    }

    /**
     * 车牌
     *
     * @param plateNum 上报
     * @return 返回
     */
    private static String getPlateNum(String plateNum) {
        if ("_无_".equals(plateNum)) {
            return "未识别";
        }
        return plateNum;
    }
}
