package com.icetech.park.service.queryfee.impl;

import com.icetech.cloudcenter.api.QueryExitPlateNumService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.domain.constant.RedisDiscountKeyConstant;
import com.icetech.park.service.queryfee.BaseQueryFeeChain;
import com.icetech.park.service.queryfee.multipleareachain.OutAreaPreFeeHandleChain;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.request.QueryOrderFeeRequest;
import com.icetech.cloudcenter.domain.response.QueryExitPlateNumResponse;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.basics.service.charge.FeeParamHolder;
import com.icetech.park.service.queryfee.MultipleAreaFeeParamHolder;
import com.icetech.park.service.queryfee.chain.PreFeeHandleChain;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.AssertTools;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Slf4j
@Service
public class P2cQueryFeeServiceImpl extends BaseQueryFeeChain {

    @Autowired
    private ParkService parkService;
    @Autowired
    private OrderInfoDao orderInfoDao;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private QueryExitPlateNumService queryExitPlateNumService;
    @Autowired
    private PreFeeHandleChain preFeeChainHandleChain;
    @Autowired
    private OutAreaPreFeeHandleChain outAreaPreFeeHandleChain;

    /**
     * 端云查询费用
     *
     * @param queryOrderFeeRequest 请求参数
     * @param orderInfo 订单信息
     * @param park 车场信息
     * @param parkConfig 高级配置
     * @return 费用信息
     */
    public ObjectResponse<QueryOrderFeeResponse> queryFee(QueryOrderFeeRequest queryOrderFeeRequest,
                                                          OrderInfo orderInfo, Park park, ParkConfig parkConfig) {
        QueryOrderFeeResponse queryOrderFeeResponse;
        boolean lock = orderInfo != null && orderInfo.getOrderNum() != null && orderInfo.getOrderNum().length() > 10;
        try {
            if (lock) {
                redisUtils.set(RedisConstants.QUERY_FEE_LOCK + orderInfo.getOrderNum(), "1", 3000L);
            }
            if (park.getIsInterior() == null || park.getIsInterior() == 0 || (orderInfo != null && orderInfo.getRegionId() == null)){
                //普通场库
                queryOrderFeeResponse = queryFeeObject(queryOrderFeeRequest, orderInfo, park, parkConfig);
            }else {
                //端云场中场
                queryOrderFeeResponse = queryMultiFeeObject(queryOrderFeeRequest, orderInfo, park, parkConfig);
            }
        } finally {
            if (lock) {
                redisUtils.remove(RedisConstants.QUERY_FEE_LOCK + orderInfo.getOrderNum());
            }
        }
        return ObjectResponse.success(queryOrderFeeResponse);

    }

    public QueryOrderFeeResponse queryMultiFeeObject(QueryOrderFeeRequest queryOrderFeeRequest,
                                                     OrderInfo orderInfo, Park park, ParkConfig parkConfig) {
        String channelCode = queryOrderFeeRequest.getChannelId();
        Boolean isMasterChannel = null;
        if (channelCode != null) {
            ParkInoutdevice parkChannel = parkService.getInoutDeviceByCode(channelCode).getData();
            isMasterChannel = Integer.valueOf(1).equals(parkChannel.getIsMaster());
        }
        if (isMasterChannel != null && !isMasterChannel) {
            if (orderInfo.getId() == null) {
                //出口支付，获取车辆信息
                ObjectResponse<QueryExitPlateNumResponse> queryResponse =
                        queryExitPlateNumService.query(queryOrderFeeRequest.getParkCode(), queryOrderFeeRequest.getChannelId());
                ObjectResponse.notError(queryResponse, "当前通道未找到车辆");
                QueryExitPlateNumResponse data = queryResponse.getData();
                QueryOrderFeeResponse cacheFee = data.getCacheFee();
                if (cacheFee != null) {
                    log.info("直接从出口缓存获取费用，cacheFee：{}", cacheFee);
                    return cacheFee;
                }
                String orderNum = data.getOrderNum();
                Long exitTime = data.getExitTime();
                orderInfo = new OrderInfo();
                orderInfo.setOrderNum(orderNum);
                orderInfo.setParkId(park.getId());
                orderInfo = orderInfoDao.selectLimitOneOrderByEnterDesc(orderInfo);
                AssertTools.notNull(orderInfo, CodeConstants.ERROR_3001, "未查询到订单信息");
                orderInfo.setExitTime(exitTime);
            }
            //计费结束时间
            Long endTime = orderInfo.getExitTime() == null ? DateTools.unixTimestamp() : orderInfo.getExitTime();

            //停车时长，计费结束时间 - 入场时间
            long parkTime = endTime - orderInfo.getEnterTime();

            //缴费后的免费时长
            int freeTimeAfterPay = parkConfig.getIsfreeAfterpay(15);
            log.info("预留免费时长为[{}]秒, 子通道无需缴费返回, 车牌号[{}]", freeTimeAfterPay * 60, orderInfo.getPlateNum());
            return getFreeRetWithPaidPrice(orderInfo, endTime, parkTime, freeTimeAfterPay, park.getParkName());
        }

        //场内支付
        if (orderInfo.getId() != null && NumberUtils.toPrimitive(orderInfo.getHasSon()) == 0) {
            return queryFeeObject(queryOrderFeeRequest, orderInfo, park,parkConfig);
        }
        //计算大场费用和小场费用
        return multipleAreaQueryOrderFee(queryOrderFeeRequest, orderInfo, parkConfig, park);
    }

    public QueryOrderFeeResponse queryFeeObject(QueryOrderFeeRequest queryOrderFeeRequest,
                                                OrderInfo orderInfo, Park park, ParkConfig parkConfig) {
        Long parkId = park.getId();
        if (orderInfo.getId() == null){
            //出口支付，获取车辆信息
            ObjectResponse<QueryExitPlateNumResponse> queryResponse =
                    queryExitPlateNumService.query(queryOrderFeeRequest.getParkCode(), queryOrderFeeRequest.getChannelId());
            ObjectResponse.notError(queryResponse, "当前通道未找到车辆");
            QueryExitPlateNumResponse data = queryResponse.getData();
            QueryOrderFeeResponse cacheFee = data.getCacheFee();
            if (cacheFee != null){
                log.info("直接从出口缓存获取费用，cacheFee[{}]", cacheFee);
                return cacheFee;
            }
            String orderNum = data.getOrderNum();
            Long exitTime = data.getExitTime();
            orderInfo = new OrderInfo();
            orderInfo.setOrderNum(orderNum);
            orderInfo.setParkId(parkId);
            orderInfo = orderInfoDao.selectLimitOneOrderByEnterDesc(orderInfo);
            orderInfo.setExitTime(exitTime);
        }else{
            /*
             * 异常离场的订单，再次请求计费时，计费结束时间不应该按异常离场的时间，还应该按当前时间，
             * 所以先设置为空，；
             */
            if (!orderInfo.getServiceStatus().equals(OrderStatusConstants.IN_PARK)){
                orderInfo.setExitTime(DateTools.unixTimestamp());
            }
            if (Integer.valueOf(1).equals(orderInfo.getNoneEnterFlag())) {
                return fixedFeeHandle(queryOrderFeeRequest, orderInfo, park, parkConfig);
            }
        }
        if (StringUtils.isNotBlank(orderInfo.getOrderNum())) {
            redisUtils.remove(RedisDiscountKeyConstant.DISCOUNT_USE_PRE + parkId + ":" + orderInfo.getOrderNum());
            redisUtils.remove(RedisDiscountKeyConstant.PAY_DISCOUNT_PRE + parkId + ":" + orderInfo.getOrderNum());
        }
        return preFeeChainHandleChain.queryFee(new FeeParamHolder(queryOrderFeeRequest, orderInfo, parkConfig, park));
    }

    public QueryOrderFeeResponse multipleAreaQueryOrderFee(QueryOrderFeeRequest queryOrderFeeRequest, OrderInfo orderInfo,
                                                           ParkConfig parkConfig, Park park){
        log.info("[场中场计费] 请求参数[{}]", queryOrderFeeRequest);
        if (orderInfo.getId() == null){
            /*
             * 出口支付，获取车辆信息
             */
            ObjectResponse<QueryExitPlateNumResponse> queryResponse =
                    queryExitPlateNumService.query(queryOrderFeeRequest.getParkCode(), queryOrderFeeRequest.getChannelId());
            ObjectResponse.notError(queryResponse, "当前通道未找到车辆");
            QueryExitPlateNumResponse data = queryResponse.getData();
            QueryOrderFeeResponse cacheFee = data.getCacheFee();
            if (cacheFee != null){
                log.info("直接从出口缓存获取费用，cacheFee：{}",cacheFee);
                return cacheFee;
            }
            String orderNum = data.getOrderNum();
            Long exitTime = data.getExitTime();
            orderInfo = new OrderInfo();
            orderInfo.setOrderNum(orderNum);
            orderInfo.setParkId(park.getId());
            orderInfo = orderInfoDao.selectLimitOneOrderByEnterDesc(orderInfo);
            orderInfo.setExitTime(exitTime);
        } else {
            /*
             * 异常离场的订单，再次请求计费时，计费结束时间不应该按异常离场的时间，还应该按当前时间，
             * 所以先设置为空，；
             */
            if (!orderInfo.getServiceStatus().equals(OrderStatusConstants.IN_PARK)) {
                orderInfo.setExitTime(DateTools.unixTimestamp());
            }
        }
        if (StringUtils.isNotBlank(orderInfo.getOrderNum())) {
            redisUtils.remove(RedisDiscountKeyConstant.DISCOUNT_USE_PRE + park.getId() + ":" + orderInfo.getOrderNum());
            redisUtils.remove(RedisDiscountKeyConstant.PAY_DISCOUNT_PRE + park.getId() + ":" + orderInfo.getOrderNum());
        }
        return outAreaPreFeeHandleChain.queryFee(new MultipleAreaFeeParamHolder(queryOrderFeeRequest, orderInfo, parkConfig, park));

    }

    private QueryOrderFeeResponse fixedFeeHandle(QueryOrderFeeRequest queryOrderFeeRequest,
                                                 OrderInfo orderInfo, Park park, ParkConfig parkConfig) {
        QueryOrderFeeResponse queryOrderFeeResponse = new QueryOrderFeeResponse();
        queryOrderFeeResponse.setOrderNum(orderInfo.getOrderNum());
        queryOrderFeeResponse.setParkName(park.getParkName());
        queryOrderFeeResponse.setPlateNum(orderInfo.getPlateNum());
        queryOrderFeeResponse.setEnterTime(orderInfo.getEnterTime());
        queryOrderFeeResponse.setCarType(orderInfo.getCarType());
        queryOrderFeeResponse.setQueryTime(
                queryOrderFeeRequest.getExitTime() == null ? DateTools.unixTimestamp() : queryOrderFeeRequest.getExitTime());
        queryOrderFeeResponse.setParkTime(0L);
        String fixedFee = String.valueOf(fixedFee(parkConfig, orderInfo.getCarType()));
        queryOrderFeeResponse.setTotalAmount(fixedFee);
        queryOrderFeeResponse.setUnpayPrice(fixedFee);
        queryOrderFeeResponse.setPaidAmount(String.valueOf(0));
        queryOrderFeeResponse.setDiscountAmount(String.valueOf(0));
        queryOrderFeeResponse.setDiscountPrice(String.valueOf(0));
        queryOrderFeeResponse.setStatus(2);
        queryOrderFeeResponse.setFreeTime((long) parkConfig.getIsfreeAfterpay(15));
        queryOrderFeeResponse.setIsFixedFee(true);
        return queryOrderFeeResponse;
    }

    /**
     * 判断是否固定收费
     * @return 固定收费金额
     */
    private float fixedFee(ParkConfig parkConfig,Integer carType){
        if (parkConfig == null) {
            return -1;
        }
        if (Integer.valueOf(3).equals(parkConfig.getNoenterHandleType())
                || (Integer.valueOf(2).equals(parkConfig.getNoenterHandleType())
                    && Integer.valueOf(1).equals(parkConfig.getIsfixedfees()))){
            return carType == 1 ? parkConfig.getFixedfeevalue() : parkConfig.getFixedFeeValueBig();
        }
        return -1;
    }
}
