package com.icetech.park.service.queryfee.chain;

import cn.hutool.core.date.DateUtil;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.service.charge.FeeParamHolder;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.charge.dto.OrderSumFeeDto;
import com.icetech.cloudcenter.domain.request.QueryOrderFeeRequest;
import com.icetech.cloudcenter.domain.response.PlateTypeDto;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.queryfee.MonthFeeDto;
import com.icetech.park.service.queryfee.MonthFeeHandle;
import com.icetech.park.service.queryfee.QueryFeeChainAbstract;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Slf4j
@Service
public class PreFeeHandleChain extends QueryFeeChainAbstract {

    @Autowired
    private OrderInfoDao orderInfoDao;
    @Autowired
    private OrderCarInfoDao orderCarInfoDao;
    @Autowired
    private MonthFeeHandle monthFeeHandle;
    @Autowired
    private ParkService parkService;

    //下一个责任类
    private final ComputeFeeHandleChain nextChain;
    @Autowired
    public PreFeeHandleChain(ComputeFeeHandleChain nextChain) {
        this.nextChain = nextChain;
    }

    @Override
    public QueryOrderFeeResponse queryFee(FeeParamHolder feeParamHolder) {
        QueryOrderFeeResponse queryOrderFeeResponse = queryFeeObject(feeParamHolder);
        feeParamHolder.setQueryOrderFeeResponse(queryOrderFeeResponse);
        if (!feeParamHolder.isBreak()) {
            if (nextChain != null) {
                return nextChain.queryFee(feeParamHolder);
            }
        }
        return queryOrderFeeResponse;
    }

    public QueryOrderFeeResponse queryFeeObject(FeeParamHolder feeParamHolder) {

        QueryOrderFeeRequest queryOrderFeeRequest = feeParamHolder.getQueryOrderFeeRequest();
        OrderInfo orderInfo = feeParamHolder.getOrderInfo();
        Park park = feeParamHolder.getPark();
        ParkConfig parkConfig = feeParamHolder.getParkConfig();

        Long parkId = park.getId();

        //用参数中的车型来查询费用
        Integer carType = queryOrderFeeRequest.getCarType();
        if (carType != null) {
            orderInfo.setCarType(carType);
        }

        Long endTime = feeParamHolder.getEndTime();
        //计费结束时间
        if (endTime == null) {
            endTime = DateTools.unixTimestamp();
        }
        Long startTime = orderInfo.getEnterTime();

        //查询已缴费汇总信息
        OrderSumFeeDto feeDto = getPaidFee(orderInfo.getOrderNum(), parkId);

        //判断新能源车 首次免费时长
        try {
            Integer batteryCarFreeTime = getBatteryCarForFree(orderInfo, feeDto, parkConfig, parkId, endTime - startTime);
            startTime = startTime + batteryCarFreeTime;
            orderInfo.setEnterTime(startTime);
        } catch (Exception e) {
            log.error("处理失败: {}. queryOrderFeeRequest[{}], orderInfo[{}], park[{}], parkConfig[{}]",
                    e.getMessage(), queryOrderFeeRequest, orderInfo, park, parkConfig, e);
        }

        //停车时长，计费结束时间 - 入场时间
        long parkTime = endTime - startTime;

        //缴费后的免费时长
        int freeTimeAfterPay = NumberUtils.toPrimitive(parkConfig.getIsfreeAfterpay(15));

        //公务车以G结尾
        if (NumberUtils.toPrimitive(parkConfig.getFreeOfficialCars()) == 1
                && PlateTypeDto.officialCar(orderInfo.getPlateNum())){
            QueryOrderFeeResponse freeRet = getFreeRet(orderInfo, feeParamHolder.getQueryTime(), parkTime,
                    freeTimeAfterPay, park.getParkName());
            log.info("公务车无需缴费返回，车牌号：{}", orderInfo.getPlateNum());
            feeParamHolder.setBreak(true);
            return freeRet;
        }
        //判断月卡类型
        MonthFeeDto monthFeeDto = monthFeeHandle.getMonthFeeParam(feeParamHolder, parkId, orderInfo,
                startTime, endTime, parkConfig, 0);
        if (monthFeeDto != null) {
            if (monthFeeDto.isMonthRet()) {
                QueryOrderFeeResponse monthCarRet = getFreeRet(orderInfo, feeParamHolder.getQueryTime(), parkTime,
                        freeTimeAfterPay, park.getParkName());
                feeParamHolder.setBreak(true);
                return monthCarRet;
            } else {
                if (monthFeeDto.getNewStartTime() != null) {
                    startTime = monthFeeDto.getNewStartTime();
                }
                if (monthFeeDto.getNewEndTime() != null) {
                    endTime = monthFeeDto.getNewEndTime();
                }
                feeParamHolder.setCsFeeType(monthFeeDto.getCsFeeType());
                feeParamHolder.setCsMonthCarFee(monthFeeDto.isCsMonthCarFee());
                feeParamHolder.setCsStartTime(monthFeeDto.getCsStartTime());
                feeParamHolder.setCsEndTime(monthFeeDto.getCsEndTime());
                feeParamHolder.setCsSwitchTime(monthFeeDto.getCsSwitchTime());
                feeParamHolder.setBillIdList(monthFeeDto.getBillIdList());

            }
        }

        //预约车判断
        Integer type = orderInfo.getType();
        if (PlateTypeEnum.访客车辆.getType().equals(type)) {
            //判断是否收费
            Integer visitIscharge = parkConfig.getVisitIscharge();
            if (visitIscharge != null && visitIscharge == 0) {
                QueryOrderFeeResponse monthCarRet = getFreeRet(orderInfo, feeParamHolder.getQueryTime(), parkTime,
                        freeTimeAfterPay, park.getParkName());
                log.info("预约车无需缴费返回，车牌号：{}", orderInfo.getPlateNum());
                feeParamHolder.setBreak(true);
                return monthCarRet;
            }
        }
        feeParamHolder.setParkTime(parkTime);
        feeParamHolder.setStartTime(startTime);
        feeParamHolder.setEndTime(endTime);
        feeParamHolder.setOrderInfo(orderInfo);
        feeParamHolder.setOrderSumFeeDto(feeDto);

        return new QueryOrderFeeResponse();
    }

    /**
     * 新能源车 减时长
     *
     * @param orderInfo 订单信息
     * @param parkId 车场ID
     * @return 减时长
     */
    private Integer getBatteryCarForFree(OrderInfo orderInfo, OrderSumFeeDto feeDto,
                                         ParkConfig parkConfig, Long parkId,Long parkTime) {
        boolean isOpenFlag = Integer.valueOf(1).equals(parkConfig.getIsNewenergyCharge());
        if (!isOpenFlag || feeDto != null) {
            return 0;
        }
        OrderCarInfo carInfo = orderCarInfoDao.selectByOrderNum(orderInfo.getOrderNum());
        if (carInfo == null) {
            return 0;
        }
        //车牌颜色
        String plateColor = carInfo.getPlateColor();
        if (StringUtils.isEmpty(plateColor) || !plateColor.contains("绿")) {
            return 0;
        }


        //入场当天 出场记录
        Long startTimeForMatch = DateUtil.beginOfDay(DateUtil.date()).getTime() / 1000;

        Long endTimeForMatch = DateUtil.endOfDay(DateUtil.date()).getTime() / 1000;
        Integer count = orderInfoDao.countRecentExitByEnterTime(parkId, orderInfo.getPlateNum(), startTimeForMatch, endTimeForMatch);
        //走配置项
        Integer newEnergyDiscountMinutes = parkConfig.getNewEnergyDiscountMinutes();
        //查询计费规则的免费时长
        ObjectResponse<Integer> parkFreeTime = parkService.getFreeTime(parkId);
        if (!ObjectResponse.isSuccess(parkFreeTime)&&parkFreeTime.getData()==null) {
            return 0;
        }
        int BATTERY_FREEDOM_SECONDS = Math.max(newEnergyDiscountMinutes*60 - parkFreeTime.getData()*60,0);
        Integer TWO_HOUR_SECONDS = newEnergyDiscountMinutes * 60;
        Long newFreeParkTime = count<=0 ? (parkTime > TWO_HOUR_SECONDS) ? BATTERY_FREEDOM_SECONDS : parkTime - 1 : 0;
        log.info("运算新能源首免结果, platenum[{}], parkTime[{}],  freeParkTime[{}]",orderInfo.getPlateNum(), parkTime, newFreeParkTime);
        return newFreeParkTime.intValue();
    }
}
