package com.icetech.park.service.queryfee;

import cn.hutool.core.collection.CollectionUtil;
import com.google.common.collect.Lists;
import com.icetech.basics.domain.entity.park.ParkChargeconfig;
import com.icetech.basics.service.charge.BaseFeeParamHolder;
import com.icetech.cloudcenter.domain.charge.dto.OrderSumFeeDto;
import com.icetech.cloudcenter.domain.discount.ParkDiscountUsercharge;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.common.constants.DiscountTypeEnum;
import com.icetech.common.utils.JsonUtils;
import com.icetech.fee.dao.merchant.DiscountDayDao;
import com.icetech.fee.domain.entity.merchant.DiscountDay;
import com.icetech.order.domain.entity.OrderDiscount;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPayDiscount;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.ToDoubleFunction;
import java.util.function.ToIntFunction;
import java.util.stream.Collectors;

/**
 * 优惠计算工具类
 * @author wangzhiwei
 *
 */
@Component
@Slf4j
public class DiscountOverHandle extends QueryFeeChainAbstract {


    @Autowired
    private DiscountDayDao discountDayDao;
    /**
     * 获取优惠信息
     * @param feeParamHolder
     * @param parkChargeconfig
     * @param orderInfo
     * @param orderSumFeeDto
     * @param parkDiscountUsercharge
     * @param discountList
     * @param parkingTime
     * @param unPayPrice
     * @param totalPrice
     * @param multi 是否场中场
     * @return
     */
    public DiscountInfo getDiscount(BaseFeeParamHolder feeParamHolder, ParkChargeconfig parkChargeconfig, OrderInfo orderInfo,
                                    OrderSumFeeDto orderSumFeeDto, ParkDiscountUsercharge parkDiscountUsercharge, List<OrderDiscount> discountList,
                                    Long parkingTime, BigDecimal unPayPrice, BigDecimal totalPrice, Boolean multi){
        //有第三方优惠券走第三方
        List<OrderDiscount> thirdOrderDiscount = discountList.stream()
                .filter(orderDiscount -> orderDiscount.getFrom() == 2)
                .collect(Collectors.toList());
        List<OrderDiscount> cloudOrderDiscount = discountList.stream()
                .filter(orderDiscount -> orderDiscount.getFrom() == 1)
                .collect(Collectors.toList());
        Boolean isThird = (parkDiscountUsercharge.getThirdpartyStackeUsage() == 1);
        //判断是否叠加
        boolean isPriceOver;
        boolean isTimeOver;
        if(StringUtils.isNotBlank(parkDiscountUsercharge.getSametypeTogetherUser())){
            String[] discountTypes = parkDiscountUsercharge.getSametypeTogetherUser().split(",");
            List<Integer> discountTypeList = Lists.newArrayList();
            for (String discountType : discountTypes) {
                discountTypeList.add(Integer.valueOf(discountType));
            }
            isPriceOver = (parkDiscountUsercharge.getSupportStackeUsage() == 1 && discountTypeList.contains(DiscountTypeEnum.减免金额.getType()));
            isTimeOver = (parkDiscountUsercharge.getSupportStackeUsage() == 1 && discountTypeList.contains(DiscountTypeEnum.减免时间.getType()));
        }else {
            isPriceOver = false;
            isTimeOver = false;
        }
        DiscountInfo thirdDiscountInfo = new DiscountInfo(Lists.newArrayList(),BigDecimal.ZERO,BigDecimal.ZERO);
        //第三方
        if (CollectionUtils.isNotEmpty(thirdOrderDiscount)){
            Map<Integer,List<OrderDiscount>> discountsMap = thirdOrderDiscount.stream().collect(Collectors.groupingBy(OrderDiscount::getType));
            List<OrderDiscount> orderDiscounts4Time = discountsMap.get(DiscountTypeEnum.减免时间.getType());
            List<OrderDiscount> orderDiscounts4Price = discountsMap.get(DiscountTypeEnum.减免金额.getType());
            List<OrderDiscount> orderDiscounts4Rate = discountsMap.get(DiscountTypeEnum.折扣.getType());
            List<OrderDiscount> orderDiscounts4Free = discountsMap.get(DiscountTypeEnum.全免.getType());
            List<OrderDiscount> orderDiscounts4Day = discountsMap.get(DiscountTypeEnum.按天优免.getType());
            if (CollectionUtils.isNotEmpty(orderDiscounts4Free)){
                return getDiscountFree(orderDiscounts4Free, unPayPrice);
            }
            if (CollectionUtils.isNotEmpty(orderDiscounts4Day)){
                return getDiscountDay(feeParamHolder, parkChargeconfig, orderInfo, orderDiscounts4Day, unPayPrice);
            }
            log.info("[叠加优惠-三方开始计算] orderNum {} thirdOrderDiscount{} isPriceOver {} isTimeOver {} isThird {}",orderInfo.getOrderNum(),thirdOrderDiscount, isPriceOver,isTimeOver,isThird);
            DiscountInfo discountInfoPrice = getDiscountPrice(orderDiscounts4Price, unPayPrice,true,true);
            DiscountInfo discountInfoTime = getDiscountTime(feeParamHolder,parkChargeconfig, orderInfo, orderSumFeeDto,orderDiscounts4Time, unPayPrice,parkingTime,true,multi,true);
            DiscountInfo discountInfoRate = getDiscountRate(orderDiscounts4Rate, totalPrice);
            //比较最大优惠进行返回
            List<DiscountInfo> discountInfos = Lists.newArrayList(discountInfoPrice, discountInfoTime, discountInfoRate);
            discountInfos = discountInfos.stream()
                    .sorted(Comparator.comparing(DiscountInfo::getMaxDiscountPrice).reversed())
                    .collect(Collectors.toList());
            thirdDiscountInfo = discountInfos.stream().max(Comparator.comparing(DiscountInfo::getDiscountPrice)).get();
            log.info("[叠加优惠-三方最终] orderNum {} discountList {} discountInfo {} ", orderInfo.getOrderNum(),JsonUtils.toString(discountList),thirdDiscountInfo);
            if (!isThird) {
                return thirdDiscountInfo;
            }
            if (thirdDiscountInfo.getDiscountPrice().compareTo(unPayPrice) >= 0) {
                return thirdDiscountInfo;
            }
        }
        DiscountInfo cloudDiscountInfo = new DiscountInfo(Lists.newArrayList(),BigDecimal.ZERO,BigDecimal.ZERO);
        //平台
        if (CollectionUtils.isNotEmpty(cloudOrderDiscount)){
            Map<Integer,List<OrderDiscount>> discountsMap = cloudOrderDiscount.stream().collect(Collectors.groupingBy(OrderDiscount::getType));
            List<OrderDiscount> orderDiscounts4Time = discountsMap.get(DiscountTypeEnum.减免时间.getType());
            List<OrderDiscount> orderDiscounts4Price = discountsMap.get(DiscountTypeEnum.减免金额.getType());
            List<OrderDiscount> orderDiscounts4Rate = discountsMap.get(DiscountTypeEnum.折扣.getType());
            List<OrderDiscount> orderDiscounts4Free = discountsMap.get(DiscountTypeEnum.全免.getType());
            List<OrderDiscount> orderDiscounts4Day = discountsMap.get(DiscountTypeEnum.按天优免.getType());
            if (CollectionUtils.isNotEmpty(orderDiscounts4Free)){
                return getDiscountFree(orderDiscounts4Free,unPayPrice);
            }
            if (CollectionUtils.isNotEmpty(orderDiscounts4Day)){
                return getDiscountDay(feeParamHolder,parkChargeconfig,orderInfo,orderDiscounts4Day,unPayPrice);
            }
            log.info("[叠加优惠-平台开始计算] orderNum {} cloudOrderDiscount {} isPriceOver {} isTimeOver {}",orderInfo.getOrderNum(),cloudOrderDiscount, isPriceOver,isTimeOver);
            DiscountInfo discountInfoPrice = getDiscountPrice(orderDiscounts4Price, unPayPrice,isPriceOver,false);
            DiscountInfo discountInfoTime = getDiscountTime(feeParamHolder,parkChargeconfig, orderInfo, orderSumFeeDto,orderDiscounts4Time, unPayPrice,parkingTime,isTimeOver,multi,false);
            DiscountInfo discountInfoRate = getDiscountRate(orderDiscounts4Rate, totalPrice.subtract(thirdDiscountInfo.getDiscountPrice()));
            //比较最大优惠进行返回
            List<DiscountInfo> discountInfos = Lists.newArrayList(discountInfoPrice, discountInfoTime, discountInfoRate);
            discountInfos = discountInfos.stream()
                    .sorted(Comparator.comparing(DiscountInfo::getMaxDiscountPrice).reversed())
                    .collect(Collectors.toList());
            cloudDiscountInfo = discountInfos.stream().max(Comparator.comparing(DiscountInfo::getDiscountPrice)).get();
            log.info("[叠加优惠-平台最终] orderNum {} discountList {} discountInfo {} ", orderInfo.getOrderNum(),JsonUtils.toString(discountList),cloudDiscountInfo);
        }
        if (cloudDiscountInfo.getDiscountPrice().add(thirdDiscountInfo.getDiscountPrice()).compareTo(unPayPrice) >= 0) {
            BigDecimal subtract = totalPrice.subtract(thirdDiscountInfo.getDiscountPrice()).setScale(2, BigDecimal.ROUND_HALF_UP);
            cloudDiscountInfo.setDiscountPrice(subtract);
            cloudDiscountInfo.setMaxDiscountPrice(subtract);
            cloudDiscountInfo.setMerchantDiscountPrice(subtract);
            OrderPayDiscount.MerchantCouponsInfo merchantCouponsInfo = cloudDiscountInfo.getMerchantCouponsInfo();
            merchantCouponsInfo.setDiscountPrice(subtract);
        }
        DiscountInfo discountInfo = new DiscountInfo(
                CollectionUtil.addAllIfNotContains(thirdDiscountInfo.getDiscountNos(), cloudDiscountInfo.getDiscountNos()),
                thirdDiscountInfo.getDiscountPrice().add(cloudDiscountInfo.getDiscountPrice()),
                thirdDiscountInfo.getMaxDiscountPrice().add(cloudDiscountInfo.getMaxDiscountPrice()));
        discountInfo.setThirdCouponsInfo(thirdDiscountInfo.getThirdCouponsInfo());
        discountInfo.setThirdDiscountPrice(thirdDiscountInfo.getThirdDiscountPrice());
        discountInfo.setMerchantCouponsInfo(cloudDiscountInfo.getMerchantCouponsInfo());
        discountInfo.setMerchantDiscountPrice(cloudDiscountInfo.getMerchantDiscountPrice());
        return discountInfo;
    }


    /**
     * 计算金额优惠信息
     * @param discountList
     * @param unpaidPrice
     * @param isOver
     * @return
     */
    private DiscountInfo getDiscountPrice(List<OrderDiscount> discountList,BigDecimal unpaidPrice,Boolean isOver,Boolean isThird){
        if (CollectionUtils.isEmpty(discountList)){
            return new DiscountInfo(Lists.newArrayList(),BigDecimal.ZERO,BigDecimal.ZERO);
        }
        discountList = discountList.stream()
                .sorted(Comparator.comparingDouble((ToDoubleFunction<OrderDiscount>) value -> Double.parseDouble(value.getAmount())).reversed())
                .collect(Collectors.toList());
        if (isOver){
            return getPriceOverDiscount(discountList,unpaidPrice,isThird);
        }else {
            //使用最大的优惠券
            return getPriceOverDiscount(Lists.newArrayList(discountList.get(0)), unpaidPrice,isThird);
        }
    }


    /**
     * 计算时长优惠信息
     * @param feeParamHolder
     * @param parkChargeconfig
     * @param orderInfo
     * @param orderSumFeeDto
     * @param discountList
     * @param unpaidPrice
     * @param parkingTime
     * @param isThird
     * @param isOver
     * @return
     */
    private DiscountInfo getDiscountTime(BaseFeeParamHolder feeParamHolder, ParkChargeconfig parkChargeconfig, OrderInfo orderInfo, OrderSumFeeDto orderSumFeeDto, List<OrderDiscount> discountList, BigDecimal unpaidPrice, Long parkingTime, Boolean isOver,Boolean multi,Boolean isThird) {
        if (CollectionUtils.isEmpty(discountList)) {
            return new DiscountInfo(Lists.newArrayList(), BigDecimal.ZERO, BigDecimal.ZERO);
        }
        discountList = discountList.stream()
                .sorted(Comparator.comparingInt((ToIntFunction<OrderDiscount>) value -> Integer.parseInt(value.getAmount())).reversed())
                .collect(Collectors.toList());
        if (isOver) {
            if (multi){
                return getTimeOverDiscountMultiple((MultipleAreaFeeParamHolder) feeParamHolder,orderInfo,discountList,unpaidPrice,isThird);
            }else {
                return getTimeOverDiscount(feeParamHolder, parkChargeconfig, orderInfo, orderSumFeeDto, discountList, unpaidPrice, parkingTime,isThird);
            }
        } else {
            if (multi){
                return getTimeOverDiscountMultiple((MultipleAreaFeeParamHolder) feeParamHolder,orderInfo,Lists.newArrayList(discountList.get(0)),unpaidPrice,isThird);
            }else {
                return getTimeOverDiscount(feeParamHolder, parkChargeconfig, orderInfo, orderSumFeeDto, Lists.newArrayList(discountList.get(0)), unpaidPrice, parkingTime,isThird);
            }
        }
    }

    /**
     * 计算折扣优惠信息
     * @param discountList
     * @param totalPrice
     * @return
     */
    private DiscountInfo getDiscountRate(List<OrderDiscount> discountList,BigDecimal totalPrice){
        if (CollectionUtils.isEmpty(discountList)){
            return new DiscountInfo(Lists.newArrayList(),BigDecimal.ZERO,BigDecimal.ZERO);
        }
        discountList = discountList.stream().sorted(Comparator.comparing(OrderDiscount::getAmount)).collect(Collectors.toList());
        OrderDiscount orderDiscount = discountList.get(0);
        DiscountInfo rateOverDiscount = getRateOverDiscount(orderDiscount, totalPrice);
        log.info("[叠加优惠-折扣4] discountList {} discountInfo {} ", JsonUtils.toString(discountList),rateOverDiscount);
        return rateOverDiscount;
    }

    /**
     * 计算按天优免优惠
     * @param discountList
     * @param unpaidPrice
     * @return
     */
    private DiscountInfo getDiscountFree(List<OrderDiscount> discountList,BigDecimal unpaidPrice){
        OrderDiscount orderDiscount = discountList.stream().max(Comparator.comparing(OrderDiscount::getSendTime)).get();
        DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(orderDiscount.getDiscountNo()), unpaidPrice, unpaidPrice);
        discountInfo.setCouponsDiscountInfo(Lists.newArrayList(orderDiscount), orderDiscount.getThirdId() != null, unpaidPrice);

        log.info("[叠加优惠-全免] discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
        return discountInfo;
    }


    /**
     * 计算按天优免优惠
     * @param feeParamHolder
     * @param parkChargeconfig
     * @param orderInfo
     * @param discountList
     * @param unpaidPrice
     * @return
     */
    private DiscountInfo getDiscountDay(BaseFeeParamHolder feeParamHolder, ParkChargeconfig parkChargeconfig, OrderInfo orderInfo,
                                        List<OrderDiscount> discountList, BigDecimal unpaidPrice) {
        //是否有按天优免
        DiscountDay discountDay = discountDayDao.selectLastByPlateNum(orderInfo.getPlateNum(), orderInfo.getParkId());
        if (discountDay != null) {
            float discountDayAfterFee = unpaidPrice.floatValue();
            Date validStart = discountDay.getStartTime();
            Date validEnd = discountDay.getEndTime();
            Date startDate = new Date(orderInfo.getEnterTime() * 1000);
            Date endDate = new Date(feeParamHolder.getEndTime() * 1000);
            //计费结束时间如果在优惠有效期内，则全免
            if (endDate.compareTo(validStart) > 0 && validEnd.compareTo(endDate) > 0) {
                discountDayAfterFee = 0;
            } else {
                //计费结束时间如果在优惠有效期之后，并且车辆是在有效期结束之前进的场，则计算有效期结束后的费用，即计费开始时间为有效期结束时间
                if (endDate.compareTo(validEnd) > 0 && validEnd.compareTo(startDate) > 0) {
                    int startTime = (int) (validEnd.getTime() / 1000);
                    discountDayAfterFee = chargeFee(feeParamHolder, orderInfo.getParkId(), parkChargeconfig.getId(), startTime,
                            feeParamHolder.getEndTime().intValue(), 0, false, orderInfo.getCarType(),
                            feeParamHolder.isCsMonthCarFee(), feeParamHolder.getCsFeeType(),
                            feeParamHolder.getCsStartTime(), feeParamHolder.getCsEndTime(),
                            feeParamHolder.getCsSwitchTime(), feeParamHolder.isUseFreetime());
                }
            }
            List<OrderDiscount> orderDiscounts = discountList.stream().filter(od -> od.getDiscountNo().equals(discountDay.getDiscountNo()))
                    .collect(Collectors.toList());
            if (discountDayAfterFee > unpaidPrice.floatValue()) {
                DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(discountDay.getDiscountNo()), unpaidPrice, BigDecimal.valueOf(discountDayAfterFee));

                discountInfo.setCouponsDiscountInfo(orderDiscounts,orderDiscounts.get(0).getThirdId() != null, unpaidPrice);
                log.info("[叠加优惠-按天优免1] discountList {} discountInfo {} ", JsonUtils.toString(discountDay),discountInfo);
                return discountInfo;
            }
            DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(discountDay.getDiscountNo()),
                    unpaidPrice.subtract(BigDecimal.valueOf(discountDayAfterFee)).setScale(2, BigDecimal.ROUND_HALF_UP),
                    unpaidPrice.subtract(BigDecimal.valueOf(discountDayAfterFee)).setScale(2, BigDecimal.ROUND_HALF_UP));
            discountInfo.setCouponsDiscountInfo(orderDiscounts,orderDiscounts.get(0).getThirdId() != null,
                    unpaidPrice.subtract(BigDecimal.valueOf(discountDayAfterFee)).setScale(2, BigDecimal.ROUND_HALF_UP));
            log.info("[叠加优惠-按天优免2] discountList {} discountInfo {} ", JsonUtils.toString(discountDay),discountInfo);
            return discountInfo;
        }
        DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(), BigDecimal.ZERO, BigDecimal.ZERO);
        log.info("[叠加优惠-按天优免3] discountList {} discountInfo {} ", JsonUtils.toString(discountDay),discountInfo);
        return discountInfo;
    }


    private DiscountInfo getRateOverDiscount(OrderDiscount orderDiscount, BigDecimal totalPrice){
        BigDecimal rate = new BigDecimal(orderDiscount.getAmount());
        //计算折扣
        BigDecimal discountPrice = totalPrice.subtract(totalPrice.multiply(rate.multiply(BigDecimal.valueOf(0.1f)))
                .setScale(2, BigDecimal.ROUND_HALF_UP));
        DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(orderDiscount.getDiscountNo()), discountPrice, discountPrice);
        discountInfo.setCouponsDiscountInfo(Lists.newArrayList(orderDiscount),orderDiscount.getThirdId() != null, discountPrice);
        return discountInfo;
    }

    /**
     * 场中场
     * @param feeParamHolder
     * @param orderInfo
     * @param discountList
     * @param unpaidPrice
     * @return
     */
    private DiscountInfo getTimeOverDiscountMultiple(MultipleAreaFeeParamHolder feeParamHolder,OrderInfo orderInfo, List<OrderDiscount> discountList, BigDecimal unpaidPrice,Boolean isThird){
        int discountMinutesSum = discountList.stream().mapToInt(orderDiscount -> Integer.parseInt(orderDiscount.getAmount())).sum();
        if (discountMinutesSum <= 0 && isThird){
            DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(), BigDecimal.ZERO, BigDecimal.ZERO);
            log.info("[叠加优惠-优惠时长小于0],discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
            return discountInfo;
        }
        Long outParkingTime = feeParamHolder.getOutParkTime();
        Long innerParkingTime = feeParamHolder.getInnerParkTime();
        ParkChargeconfig parkChargeconfig = feeParamHolder.getParkChargeConfig();
        ParkChargeconfig innerParkChargeconfig = feeParamHolder.getInnerParkChargeconfig();
        OrderSumFeeDto orderSumFeeDto = feeParamHolder.getOrderSumFeeDto();
        BigDecimal outUnPaidPrice = unpaidPrice;
        BigDecimal innerUnPayPrice = unpaidPrice;
        QueryOrderFeeResponse innerQueryOrderFeeResponse = feeParamHolder.getInnerQueryOrderFeeResponse();
        if (innerQueryOrderFeeResponse != null){
            innerUnPayPrice = new BigDecimal(innerQueryOrderFeeResponse.getUnpayPrice());
            outUnPaidPrice = unpaidPrice.subtract(innerUnPayPrice).setScale(2, BigDecimal.ROUND_HALF_UP);
            log.info("[叠加优惠场中场] innerunPaidPrice {}, outUnPaidPrice {} ", innerUnPayPrice, outUnPaidPrice);
        }

        if (outUnPaidPrice.compareTo(BigDecimal.ZERO) > 0 && outParkingTime >= discountMinutesSum * 60L){
            //全部优惠大场
            float maxDiscountPrice = chargeFee(feeParamHolder, orderInfo.getParkId(), parkChargeconfig.getId(),
                    feeParamHolder.getStartTime().intValue(), Math.toIntExact(feeParamHolder.getStartTime().intValue() + outParkingTime),
                    discountMinutesSum, orderSumFeeDto != null,
                    orderInfo.getCarType(), feeParamHolder.isCsMonthCarFee(), feeParamHolder.getCsFeeType(),
                    feeParamHolder.getCsStartTime(), feeParamHolder.getCsEndTime(), feeParamHolder.getCsSwitchTime(),
                    feeParamHolder.isUseFreetime());

            BigDecimal discountPrice = outUnPaidPrice.subtract(BigDecimal.valueOf(maxDiscountPrice));
            discountPrice = discountPrice.compareTo(BigDecimal.ZERO) > 0 ? discountPrice : BigDecimal.ZERO;
            discountPrice = discountPrice.compareTo(unpaidPrice) > 0 ? unpaidPrice : discountPrice;

            List<String> discountNos = Lists.newArrayList();
            List<OrderDiscount> usedDiscountList = Lists.newArrayList();
            int discountTime = 0;
            for (OrderDiscount orderDiscount : discountList) {
                discountNos.add(orderDiscount.getDiscountNo());
                usedDiscountList.add(orderDiscount);
                discountTime += Integer.parseInt(orderDiscount.getAmount());
                if (discountTime * 60L >= outParkingTime) {
                    break;
                }
            }
            if (discountTime * 60L > outParkingTime) {
                DiscountInfo discountInfo = new DiscountInfo(discountNos, outUnPaidPrice, discountPrice);
                discountInfo.setCouponsDiscountInfo(usedDiscountList, isThird, outUnPaidPrice);
                log.info("[叠加优惠场中场-时间1] discountList {} discountInfo {} ", JsonUtils.toString(discountList), discountInfo);
                return discountInfo;
            }
            DiscountInfo discountInfo = new DiscountInfo(discountNos, discountPrice, discountPrice);
            discountInfo.setCouponsDiscountInfo(usedDiscountList, isThird, discountPrice);
            log.info("[叠加优惠场中场-时间2] discountList {} discountInfo {} ", JsonUtils.toString(discountList), discountInfo);
            return discountInfo;
        } else {
            BigDecimal outDiscountPrice = BigDecimal.ZERO;
            //优惠时间分钟
            long innerParkingDiscountMinutes = discountMinutesSum;
            if (outUnPaidPrice.compareTo(BigDecimal.ZERO) > 0) {
                //先大场优惠
                float maxDiscountPrice =  chargeFee(feeParamHolder, orderInfo.getParkId(), parkChargeconfig.getId(),
                        feeParamHolder.getStartTime().intValue(), Math.toIntExact(feeParamHolder.getStartTime().intValue() + outParkingTime),
                        discountMinutesSum, orderSumFeeDto != null,
                        orderInfo.getCarType(), feeParamHolder.isCsMonthCarFee(), feeParamHolder.getCsFeeType(),
                        feeParamHolder.getCsStartTime(), feeParamHolder.getCsEndTime(), feeParamHolder.getCsSwitchTime(),
                        feeParamHolder.isUseFreetime());
                outDiscountPrice = outUnPaidPrice.subtract(BigDecimal.valueOf(maxDiscountPrice));
                outDiscountPrice = outDiscountPrice.compareTo(BigDecimal.ZERO) > 0 ? outDiscountPrice : BigDecimal.ZERO;
                outDiscountPrice = outDiscountPrice.compareTo(outUnPaidPrice) > 0 ? unpaidPrice : outDiscountPrice;
                log.info("[叠加优惠场中场-大场优惠] maxDiscountPrice {} outDiscountPrice {} ", maxDiscountPrice, outDiscountPrice);

                innerParkingDiscountMinutes = discountMinutesSum - outParkingTime / 60;
            }

            BigDecimal innerDiscountPrice = BigDecimal.ZERO;
            if (innerUnPayPrice.compareTo(BigDecimal.ZERO) > 0 && innerParkingDiscountMinutes > 0) {
                //小场优惠
                float innerMaxDiscountPrice =  chargeFee(feeParamHolder, orderInfo.getParkId(), innerParkChargeconfig.getId(),
                        feeParamHolder.getInnerStartTime().intValue(), feeParamHolder.getInnerEndTime().intValue(),
                        (int) innerParkingDiscountMinutes, orderSumFeeDto != null,
                        orderInfo.getCarType(), feeParamHolder.isInnerCsMonthCarFee(), feeParamHolder.getInnerCsFeeType(),
                        feeParamHolder.getInnerCsStartTime(), feeParamHolder.getInnerCsEndTime(), feeParamHolder.getInnerCsSwitchTime(),
                        feeParamHolder.isUseFreetime(), parkChargeconfig, 2);
                innerDiscountPrice = innerUnPayPrice.subtract(BigDecimal.valueOf(innerMaxDiscountPrice));
                innerDiscountPrice = innerDiscountPrice.compareTo(BigDecimal.ZERO) > 0 ? innerDiscountPrice : BigDecimal.ZERO;
                innerDiscountPrice = innerDiscountPrice.compareTo(innerUnPayPrice) > 0 ? innerUnPayPrice : innerDiscountPrice;
                log.info("[叠加优惠场中场-小场优惠] innerMaxDiscountPrice {} innerDiscountPrice {} ", innerMaxDiscountPrice, innerDiscountPrice);
            }
            BigDecimal totalDiscountPrice = innerDiscountPrice.add(outDiscountPrice);
            List<String> discountNos = Lists.newArrayList();
            List<OrderDiscount> usedDiscountList = Lists.newArrayList();
            int discountTime = 0;
            for (OrderDiscount orderDiscount : discountList) {
                discountNos.add(orderDiscount.getDiscountNo());
                usedDiscountList.add(orderDiscount);
                discountTime += Integer.parseInt(orderDiscount.getAmount());
                if (discountTime * 60L > (outParkingTime + innerParkingTime)) {
                    break;
                }
            }
            //统计总优惠
            DiscountInfo discountInfo = new DiscountInfo(discountNos, totalDiscountPrice, totalDiscountPrice);
            discountInfo.setCouponsDiscountInfo(usedDiscountList, isThird, totalDiscountPrice);
            log.info("[叠加优惠场中场-时间3] discountList {} discountInfo {} ", JsonUtils.toString(discountList), discountInfo);
            return discountInfo;
        }
    }
    /**
     * 叠加优惠计算
     * @param feeParamHolder
     * @param parkChargeconfig
     * @param orderInfo
     * @param orderSumFeeDto
     * @param discountList
     * @param unpaidPrice
     * @param parkingTime
     * @return
     */
    private DiscountInfo getTimeOverDiscount(BaseFeeParamHolder feeParamHolder, ParkChargeconfig parkChargeconfig, OrderInfo orderInfo, OrderSumFeeDto orderSumFeeDto, List<OrderDiscount> discountList, BigDecimal unpaidPrice, Long parkingTime,Boolean isThird){
        int discountMinutesSum = discountList.stream().mapToInt(orderDiscount -> Integer.parseInt(orderDiscount.getAmount())).sum();
        if (discountMinutesSum<=0 && isThird){
            DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(), BigDecimal.ZERO, BigDecimal.ZERO);
            log.info("[叠加优惠-优惠时长小于0],discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
            return discountInfo;
        }
        //叠加
        float maxDiscountPrice =  chargeFee(feeParamHolder, orderInfo.getParkId(), parkChargeconfig.getId(),
                feeParamHolder.getStartTime().intValue(), feeParamHolder.getEndTime().intValue(),
                discountMinutesSum, orderSumFeeDto != null,
                orderInfo.getCarType(), feeParamHolder.isCsMonthCarFee(), feeParamHolder.getCsFeeType(),
                feeParamHolder.getCsStartTime(), feeParamHolder.getCsEndTime(), feeParamHolder.getCsSwitchTime(),
                feeParamHolder.isUseFreetime());
        if (isThird){
            BigDecimal discountPrice = unpaidPrice.subtract(BigDecimal.valueOf(maxDiscountPrice));
            discountPrice = discountPrice.compareTo(BigDecimal.ZERO)>0?discountPrice:BigDecimal.ZERO;
            discountPrice = discountPrice.compareTo(unpaidPrice)>0?unpaidPrice:discountPrice;
            DiscountInfo discountInfo = new DiscountInfo(discountList.stream().map(OrderDiscount::getDiscountNo).collect(Collectors.toList()), discountPrice, discountPrice);
            discountInfo.setCouponsDiscountInfo(discountList, isThird, discountPrice);
            return discountInfo;
        }
        BigDecimal discountPrice = unpaidPrice.subtract(BigDecimal.valueOf(maxDiscountPrice));
        discountPrice = discountPrice.compareTo(BigDecimal.ZERO)>0?discountPrice:BigDecimal.ZERO;
        discountPrice = discountPrice.compareTo(unpaidPrice)>0?unpaidPrice:discountPrice;
        List<String> discountNos = Lists.newArrayList();
        List<OrderDiscount> usedList = Lists.newArrayList();
        int discountTime = 0;
        for (OrderDiscount orderDiscount:discountList){
            discountNos.add(orderDiscount.getDiscountNo());
            usedList.add(orderDiscount);
            discountTime+=Integer.parseInt(orderDiscount.getAmount());
            if (discountTime>=parkingTime/60){
                break;
            }
        }
        if (discountTime>=parkingTime/60){
            DiscountInfo discountInfo = new DiscountInfo(discountNos, unpaidPrice, discountPrice);
            discountInfo.setCouponsDiscountInfo(usedList, isThird, unpaidPrice);
            log.info("[叠加优惠-时间1] discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
            return discountInfo;
        }
        DiscountInfo discountInfo = new DiscountInfo(discountNos, discountPrice, discountPrice);
        discountInfo.setCouponsDiscountInfo(usedList, isThird, discountPrice);
        log.info("[叠加优惠-时间2] discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
        return discountInfo;
    }

    /**
     * 叠加优惠计算
     * @param discountList
     * @param unpaidPrice
     * @return
     */
    private DiscountInfo getPriceOverDiscount(List<OrderDiscount> discountList, BigDecimal unpaidPrice,Boolean isThird){
        double discountPriceSum = discountList.stream()
                .mapToDouble(orderDiscount -> Double.parseDouble(orderDiscount.getAmount()))
                .sum();
        if (discountPriceSum <= 0 && isThird){
            DiscountInfo discountInfo = new DiscountInfo(Lists.newArrayList(), BigDecimal.ZERO, BigDecimal.ZERO);
            log.info("[叠加优惠-优惠金额小于0],discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
            return discountInfo;
        }
        if (isThird){
            //全部核销
            if (BigDecimal.valueOf(discountPriceSum).compareTo(unpaidPrice) > 0) {
                DiscountInfo discountInfo = new DiscountInfo(discountList.stream().map(OrderDiscount::getDiscountNo).collect(Collectors.toList()), unpaidPrice, BigDecimal.valueOf(discountPriceSum));
                discountInfo.setCouponsDiscountInfo(discountList, isThird, unpaidPrice);
                return discountInfo;
            }else{
                DiscountInfo discountInfo = new DiscountInfo(discountList.stream().map(OrderDiscount::getDiscountNo).collect(Collectors.toList()), BigDecimal.valueOf(discountPriceSum), BigDecimal.valueOf(discountPriceSum));
                discountInfo.setCouponsDiscountInfo(discountList, isThird, BigDecimal.valueOf(discountPriceSum));
                return discountInfo;
            }
        }
        List<String> discountNos = Lists.newArrayList();
        BigDecimal discountPrice = BigDecimal.ZERO;
        List<OrderDiscount> usedDiscountList = Lists.newArrayList();
        for (OrderDiscount orderDiscount:discountList){
            discountNos.add(orderDiscount.getDiscountNo());
            usedDiscountList.add(orderDiscount);
            discountPrice=discountPrice.add(BigDecimal.valueOf(Double.parseDouble(orderDiscount.getAmount())));
            if (discountPrice.compareTo(unpaidPrice) >= 0){
                break;
            }
        }
        if (discountPrice.compareTo(unpaidPrice) > 0){
            DiscountInfo discountInfo = new DiscountInfo(discountNos, unpaidPrice, BigDecimal.valueOf(discountPriceSum));
            discountInfo.setCouponsDiscountInfo(usedDiscountList, isThird, unpaidPrice);
            log.info("[叠加优惠-金额1] discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
            return discountInfo;
        }
        DiscountInfo discountInfo = new DiscountInfo(discountNos, BigDecimal.valueOf(discountPriceSum), BigDecimal.valueOf(discountPriceSum));
        discountInfo.setCouponsDiscountInfo(usedDiscountList, isThird, BigDecimal.valueOf(discountPriceSum));
        log.info("[叠加优惠-金额2] discountList {} discountInfo {} ", JsonUtils.toString(discountList),discountInfo);
        return discountInfo;
    }

    /**
     * 优惠返回实体类
     * @author wangzhiwei
     */
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class DiscountInfo implements Serializable {
        /**
         * 使用的优惠券
         */
        private List<String> discountNos;
        /**
         * 实际优惠金额
         */
        private BigDecimal discountPrice;
        /**
         * 最大优惠金额
         */
        private BigDecimal maxDiscountPrice;
        /**
         * 第三方实际优惠金额
         */
        private BigDecimal thirdDiscountPrice;
        /**
         * 商户端实际优惠金额
         */
        private BigDecimal merchantDiscountPrice;

        private OrderPayDiscount.ThirdCouponsInfo thirdCouponsInfo;

        private OrderPayDiscount.MerchantCouponsInfo merchantCouponsInfo;

        public DiscountInfo(List<String> discountNos, BigDecimal discountPrice, BigDecimal maxDiscountPrice) {
            this.discountNos = discountNos;
            this.discountPrice = discountPrice;
            this.maxDiscountPrice = maxDiscountPrice;
        }

        public void setThirdCouponsInfo(OrderPayDiscount.ThirdCouponsInfo thirdCouponsInfo) {
            this.thirdCouponsInfo = thirdCouponsInfo;
        }

        public void setMerchantCouponsInfo(OrderPayDiscount.MerchantCouponsInfo merchantCouponsInfo) {
            this.merchantCouponsInfo = merchantCouponsInfo;
        }

        public void setCouponsDiscountInfo(List<OrderDiscount> usedDiscountList, Boolean isThird, BigDecimal discountPrice) {
            for (OrderDiscount orderDiscount : usedDiscountList) {
                if (orderDiscount.getFrom() == 2) {
                    this.thirdCouponsInfo = this.thirdCouponsInfo == null
                            ? new OrderPayDiscount.ThirdCouponsInfo() : this.thirdCouponsInfo;
                    this.thirdCouponsInfo.addThirdId(orderDiscount.getThirdId());
                    this.thirdCouponsInfo.addDiscountType(orderDiscount.getType());
                    this.thirdCouponsInfo.addDiscountName(orderDiscount.getDiscountName());
                    this.thirdCouponsInfo.setDiscountPrice(discountPrice);
                } else if (orderDiscount.getFrom() == 1) {
                    this.merchantCouponsInfo = this.merchantCouponsInfo == null
                            ? new OrderPayDiscount.MerchantCouponsInfo() : this.merchantCouponsInfo;
                    this.merchantCouponsInfo.addMerchantId(orderDiscount.getMerchantId());
                    this.merchantCouponsInfo.addDiscountType(orderDiscount.getType());
                    this.merchantCouponsInfo.addDiscountName(orderDiscount.getDiscountName());
                    this.merchantCouponsInfo.setDiscountPrice(discountPrice);
                }
            }
            if (isThird) {
                this.thirdDiscountPrice = discountPrice;
            } else {
                this.merchantDiscountPrice = discountPrice;
            }
        }
    }
}
