package com.icetech.park.service.queryfee;

import com.icetech.basics.constants.RegionChargePlateTypeConsts;
import com.icetech.basics.domain.entity.NotWorkDay;
import com.icetech.basics.domain.entity.RegionChargeconfig;
import com.icetech.basics.domain.entity.park.ParkChargeconfig;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.basics.service.charge.BaseFeeParamHolder;
import com.icetech.basics.service.charge.impl.ParkChargeConfigServiceImpl;
import com.icetech.basics.service.other.NotWorkDayServiceImpl;
import com.icetech.cloudcenter.domain.charge.dto.OrderSumFeeDto;
import com.icetech.cloudcenter.domain.order.SumPayByOrderNumDto;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.common.constants.PlateColorEnum;
import com.icetech.common.constants.SqlConstant;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.DateUtils;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.order.dao.OrderPayDao;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.order.service.impl.OrderCarInfoServiceImpl;
import com.icetech.park.config.JdbcProperties;
import com.icetech.park.service.park.impl.ParkRegionServiceImpl;
import com.icetech.park.service.park.impl.RegionChargeconfigServiceImpl;
import com.icetech.park.service.runso.Clibrary;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@Slf4j
public abstract class BaseQueryFeeChain {
    //计费开始时间的最小时间点
    private static final Long SYS_START_TIME = 1546272000L;

    @Autowired
    private JdbcProperties jdbcProperties;
    @Autowired
    private OrderPayDao orderPayDao;
    @Autowired
    private ParkRegionServiceImpl parkRegionService;
    @Autowired
    private OrderCarInfoServiceImpl orderCarInfoService;
    @Autowired
    private RegionChargeconfigServiceImpl regionChargeconfigService;
    @Autowired
    private ParkChargeConfigServiceImpl parkChargeConfigService;
    @Autowired
    private NotWorkDayServiceImpl notWorkDayService;

    protected static DecimalFormat FORMAT = new DecimalFormat("###.##");

    protected float chargeFee(BaseFeeParamHolder feeParamHolder, Long parkId, Integer billId, int startTime, int tmExit,
                              int discountMinutes, boolean isPaid, Integer carType,
                              boolean isCsMonthCarFee, int csFeeType, String csStartTime, String csEndTime, Long csSwitchTime,
                              boolean isUseFreetime) {
        return chargeFee(feeParamHolder, parkId, billId, startTime, tmExit, discountMinutes, isPaid, carType,
                isCsMonthCarFee, csFeeType, csStartTime, csEndTime, csSwitchTime, isUseFreetime, feeParamHolder.getParkChargeConfig(billId), 1);
    }
    protected float chargeFee(BaseFeeParamHolder feeParamHolder, Long parkId, Integer billId, int startTime, int tmExit,
                              int discountMinutes, boolean isPaid, Integer carType,
                              boolean isCsMonthCarFee, int csFeeType, String csStartTime, String csEndTime, Long csSwitchTime,
                              boolean isUseFreetime, ParkChargeconfig parkChargeconfig, int areaType) {
        if (startTime < SYS_START_TIME) {
            throw new ResponseBodyException(CodeConstants.ERROR_3001, "计费开始时间非法");
        }
        //调用so查询费用
        Clibrary instance = Clibrary.INSTANTCE;
        int chargeFlag = isPaid ? 1 : 0;
        int useFreeTime = isUseFreetime ? 0 : 1;

        long before = System.currentTimeMillis();
        float fee;
        try{
            String sChargeConfigCfg = feeParamHolder.getSChargeConfigCfg(feeParamHolder.getParkConfig());
            parkChargeconfig = parkChargeconfig == null ? feeParamHolder.getParkChargeConfig(billId) : parkChargeconfig;
            String sChargeRule = feeParamHolder.getSChargeRule(parkChargeconfig);
            log.info("[SO计算费用] startTime:{},tmExit:{}, 计费时长:{}, discountMinutes:{},chargeFlag:{},carType:{},parkId:{},{},{},{},{},{}, {}",
                    startTime, tmExit, tmExit - startTime, discountMinutes, chargeFlag, carType, parkId,
                    csStartTime, csEndTime, csFeeType, csSwitchTime, useFreeTime, parkChargeconfig.getBilltypename());
            if (isCsMonthCarFee){
                fee = instance.chargeinterface_time_json(startTime, tmExit, discountMinutes, chargeFlag, carType,
                        sChargeConfigCfg, sChargeRule, csStartTime, csEndTime, csFeeType,
                        csSwitchTime == null ? 0 : csSwitchTime.intValue(), useFreeTime);
            }else {
                fee = instance.chargeinterface_json(startTime, tmExit, discountMinutes, chargeFlag, carType,
                        sChargeConfigCfg, sChargeRule, useFreeTime);
            }
            log.info("[SO计算费用] [{}]元", fee);
            //如果新版本计费出现问题，再去使用旧版本
            if (fee < 0) {
                String dbserver = jdbcProperties.getHost();
                String username = jdbcProperties.getUsername();
                String password = jdbcProperties.getPassword();
                float old;
                if (isCsMonthCarFee){
                    log.info("[SO计算费用] startTime:{},tmExit:{}, 计费时长:{},discountMinutes:{},chargeFlag:{},carType:{},parkId:{},billId:{},{},{},{},{},{},{},{},{}",
                            startTime, tmExit, tmExit - startTime, discountMinutes, chargeFlag, carType,
                            parkId.intValue(),billId, dbserver, username, password, csStartTime, csEndTime, csFeeType,
                            csSwitchTime == null ? 0 : csSwitchTime.intValue(), useFreeTime);
                    old = instance.chargeinterface_time_db_freetime(startTime, tmExit, discountMinutes, chargeFlag, carType,
                            parkId.intValue(),billId, dbserver, username, password, csStartTime, csEndTime, csFeeType,
                            csSwitchTime == null ? 0 : csSwitchTime.intValue(), useFreeTime);
                }else {
                    old = instance.chargeinterface_db_freetime(startTime, tmExit, discountMinutes, chargeFlag, carType,
                            parkId.intValue(),billId, dbserver, username, password, useFreeTime);
                }
                log.info("[SO旧版计算费用] 旧版本调用so获取费用[{}]元", old);
                if (NumberUtils.parseDecimal(fee).compareTo(NumberUtils.parseDecimal(old)) == 0){
                    log.warn("[监控埋点] 计费SO新旧版本获取费用不一致, new[{}], old[{}]", fee, old);
                    fee = old;
                }
            }
            if (fee < 0){
                log.warn("[SO计算费用] 调用so获取费用失败, sChargeConfigCfg[{}], sChargeRule[{}]", sChargeConfigCfg, sChargeRule);
                throw new ResponseBodyException(CodeConstants.ERROR_3001, "so计费失败");
            }
        } finally {
            long elapsedTime = System.currentTimeMillis() - before;
            if (elapsedTime > 2000){
                log.warn("[SO计算费用] 计费用时[{}]毫秒", elapsedTime);
            }
        }
        BigDecimal bg = new BigDecimal(fee);
        fee = bg.setScale(2, RoundingMode.HALF_UP).floatValue();
        log.info("[SO计算费用] 四舍五入取两位小数后为[{}]元", fee);
        Integer isnotgetsmallchange = feeParamHolder.getParkConfig().getIsnotgetsmallchange();
        if (isnotgetsmallchange != null && isnotgetsmallchange == 1){
            float ys = fee % 1;
            if (ys >= 0.5){
                fee = ((int) fee) + 0.5F;
            }else{
                fee = (int) fee;
            }
        }
        return fee;
    }

    /**
     * 判断费用状态
     * @param paidAmount 已支付金额
     * @param unPayPrice 本次待支付金额
     * @param discountPrice 本次优惠金额
     * @param discountAmount 已优惠金额
     * @param totalAmount 总应收金额
     * @return 费用状态
     */
    protected Integer getFeeStatus(Float paidAmount, Float unPayPrice, Float discountPrice,
                                   Float discountAmount, Float totalAmount) {
        log.info("[判断费用状态] paidAmount[{}], unPayPrice[{}], discountPrice[{}], discountAmount[{}], totalAmount[{}]",
                paidAmount, unPayPrice, discountPrice, discountAmount, totalAmount);
        if (totalAmount == 0) {
            // 免费时长内无需缴费
            return 1;
        }
        if (discountAmount.equals(totalAmount)){
            // 已缴费未超时
            return  3;
        }
        if ((unPayPrice > 0 || discountPrice >= totalAmount) && paidAmount == 0 && discountAmount == 0) {
            // 初次缴费
            return  2;
        }
        if (unPayPrice == 0 && paidAmount > 0) {
            // 已缴费未超时
            return  3;
        }
        if (unPayPrice > 0 && (paidAmount > 0 || discountAmount > 0)) {
            // 已缴费已超时
            return  4;
        }
        if (unPayPrice > 0){
            // 初次缴费
            return  2;
        }else{
            // 免费时长内无需缴费
            return 1;
        }
    }

    /**
     * 月卡车返回
     * @param orderInfo orderInfo
     * @param queryTime queryTime
     * @param parkTime parkTime
     * @param freeTimeAfterPay freeTimeAfterPay
     * @param parkName parkName
     * @return 费用
     */
    protected QueryOrderFeeResponse getFreeRet(OrderInfo orderInfo, Long queryTime, Long parkTime,
                                               int freeTimeAfterPay, String parkName){
        QueryOrderFeeResponse queryOrderFeeResponse = new QueryOrderFeeResponse();
        setBasePara(orderInfo, queryOrderFeeResponse, parkTime, freeTimeAfterPay, parkName, queryTime);

        queryOrderFeeResponse.setStatus(1);
        queryOrderFeeResponse.setPaidAmount(FORMAT.format(0));
        queryOrderFeeResponse.setDiscountAmount(FORMAT.format(0));
        queryOrderFeeResponse.setUnpayPrice(FORMAT.format(0));
        queryOrderFeeResponse.setDiscountPrice(FORMAT.format(0));
        queryOrderFeeResponse.setTotalAmount(FORMAT.format(0));
        return queryOrderFeeResponse;
    }
    /**
     * 本次免费返回，附带已支付金额
     * @param orderInfo orderInfo
     * @param queryTime queryTime
     * @param parkTime parkTime
     * @param freeTimeAfterPay freeTimeAfterPay
     * @param parkName parkName
     * @return 费用
     */
    protected QueryOrderFeeResponse getFreeRetWithPaidPrice(OrderInfo orderInfo, Long queryTime, Long parkTime,
                                               int freeTimeAfterPay, String parkName){
        QueryOrderFeeResponse queryOrderFeeResponse = new QueryOrderFeeResponse();
        setBasePara(orderInfo, queryOrderFeeResponse, parkTime, freeTimeAfterPay, parkName, queryTime);

        OrderSumFeeDto feeDto = getPaidFee(orderInfo.getOrderNum(), orderInfo.getParkId());
        if (feeDto == null) {
            queryOrderFeeResponse.setStatus(1);
            queryOrderFeeResponse.setPaidAmount(FORMAT.format(0));
            queryOrderFeeResponse.setDiscountAmount(FORMAT.format(0));
            queryOrderFeeResponse.setUnpayPrice(FORMAT.format(0));
            queryOrderFeeResponse.setDiscountPrice(FORMAT.format(0));
            queryOrderFeeResponse.setTotalAmount(FORMAT.format(0));
        } else {
            queryOrderFeeResponse.setStatus(3);
            queryOrderFeeResponse.setPaidAmount(FORMAT.format(feeDto.getPaidPrice()));
            queryOrderFeeResponse.setDiscountAmount(FORMAT.format(feeDto.getDiscountPrice()));
            queryOrderFeeResponse.setUnpayPrice(FORMAT.format(0));
            queryOrderFeeResponse.setDiscountPrice(FORMAT.format(0));
            queryOrderFeeResponse.setTotalAmount(FORMAT.format(feeDto.getTotalPrice()));
            queryOrderFeeResponse.setPayTime(feeDto.getLastPayTime());
        }
        return queryOrderFeeResponse;
    }

    /**
     * 设置基础参数
     * @param orderInfo 订单信息
     * @param queryOrderFeeResponse 响应实体
     * @param freeTimeAfterPay 缴费后预留免费时长
     * @param parkName 车场名称
     */
    protected void setBasePara(OrderInfo orderInfo, QueryOrderFeeResponse queryOrderFeeResponse,
                               Long parkTime, int freeTimeAfterPay, String parkName, Long queryTime) {
        queryOrderFeeResponse.setOrderNum(orderInfo.getOrderNum());
        queryOrderFeeResponse.setPlateNum(orderInfo.getPlateNum());
        queryOrderFeeResponse.setEnterTime(orderInfo.getEnterTime());
        queryOrderFeeResponse.setQueryTime(queryTime == null ? DateTools.unixTimestamp() : queryTime);
        queryOrderFeeResponse.setParkTime(parkTime);
        queryOrderFeeResponse.setParkName(parkName);
        queryOrderFeeResponse.setFreeTime((long)(freeTimeAfterPay));
        queryOrderFeeResponse.setCarType(orderInfo.getCarType());
    }

    /**
     * 汇总已支付费用
     *
     * @param orderNum 订单号
     * @param parkId 车场ID
     * @return 订单支付汇总
     */
    protected OrderSumFeeDto getPaidFee(String orderNum, Long parkId) {
        OrderPay orderPay = new OrderPay();
        orderPay.setParkId(parkId);
        orderPay.setOrderNum(orderNum);
        orderPay.setPayStatus(PayStatusConstants.PAID);
        return orderPayDao.sumFee(orderPay);
    }

    /**
     * 新版本按入场往后推24小时最大收费处理
     * @return 最大收费金额
     */
    protected BigDecimal getMaxFeePara(BaseFeeParamHolder feeParamHolder, OrderInfo orderInfo, Integer billId,
                                     Date endDate, Date beginTime, Float dayNightMaxfee) {
        List<SumPayByOrderNumDto> sumPayByOrderNumDtos = orderPayDao.sumPaidPriceByPlateNum(
                orderInfo.getParkId(), orderInfo.getPlateNum(), beginTime.getTime() / 1000, endDate.getTime() / 1000);
        if (CollectionUtils.isEmpty(sumPayByOrderNumDtos)) {
            return null;
        }
        //入场时间晚于开始时间的订单
        List<SumPayByOrderNumDto> geBeginTimeDtos = sumPayByOrderNumDtos.stream()
                .filter(sumPayByOrderNumDto -> sumPayByOrderNumDto.getEnterTime() > beginTime.getTime() / 1000).collect(Collectors.toList());
        //入场时间早于开始时间
        if (CollectionUtils.isEmpty(geBeginTimeDtos) && sumPayByOrderNumDtos.size() > 0) {
            SumPayByOrderNumDto lqBeginTimeDto = sumPayByOrderNumDtos.get(0);
            log.info("[新版本24小时按入场最大收费] 上个订单入场在当前订单入场前24小时之前[{}]", lqBeginTimeDto);
            Long firstEnterTime = lqBeginTimeDto.getEnterTime();
            Long firstExitTime = lqBeginTimeDto.getExitTime();
            //首个订单的停车时长
            long firstOrderParkTime = 0;
            if (firstExitTime != null && firstEnterTime != 0) {
                firstOrderParkTime = firstExitTime - firstEnterTime;
            }
            //首个订单的停车天数，不足1天按1天
            long firstDiffDays = firstOrderParkTime <= 0 ? 1 : firstOrderParkTime / (24 * 3600) + 1;
            //首个订单入场时间往后推整数天的时间在当前订单的入场时间之前，则无需计算最大收费
            if (firstEnterTime + firstDiffDays * 24 * 3600 <= orderInfo.getEnterTime()) {
                return null;
            }
            float paidPriceSum = lqBeginTimeDto.getTotalPrice().floatValue() - dayNightMaxfee * (firstDiffDays - 1);
            if (paidPriceSum == 0) {
                return null;
            }
            //实收和应付比较
            if (lqBeginTimeDto.getPaidPrice().floatValue() < paidPriceSum) {
                log.info("[新版本24小时按入场最大收费] 实收小于应收, 实收[{}]", paidPriceSum);
                paidPriceSum = lqBeginTimeDto.getPaidPrice().floatValue();
            }

            //入场时间往后推24小时
            long next24HoursOfFirstEnterTime = firstEnterTime + 3600 * 24 * firstDiffDays;
            if (next24HoursOfFirstEnterTime >= endDate.getTime()/1000) {
                return new BigDecimal(dayNightMaxfee).subtract(new BigDecimal(paidPriceSum));
            } else {
                //计算下一个24小时开始到出场时间的费用
                float suffixFee = chargeFee(feeParamHolder, orderInfo.getParkId(), billId, (int) (next24HoursOfFirstEnterTime),
                        (int) (endDate.getTime() / 1000),
                        0, false, orderInfo.getCarType(), false, 0,
                        null, null, null, false);
                //24小时最大收费-已支付金额，等于当次停车入场到下一个24小时起点最多收费
                return new BigDecimal(dayNightMaxfee).subtract(new BigDecimal(paidPriceSum))
                        .add(new BigDecimal(suffixFee));
            }
        }
        if (CollectionUtils.isNotEmpty(geBeginTimeDtos)) {
            //最早入场记录的支付记录
            SumPayByOrderNumDto sumPayByOrderNumDto = geBeginTimeDtos.get(0);
            log.info("[新版本24小时按入场最大收费] 上个订单入场在当前订单入场前24小时之后[{}]", sumPayByOrderNumDto);
            Long firstEnterTime = sumPayByOrderNumDto.getEnterTime();
            double paidPriceSum = geBeginTimeDtos.stream().collect(
                    Collectors.summarizingDouble(orderNumPay -> orderNumPay.getPaidPrice().doubleValue())).getSum();
            if (firstEnterTime + 3600 * 24 >= endDate.getTime()/1000) {
                return new BigDecimal(dayNightMaxfee).subtract(new BigDecimal(paidPriceSum));
            } else {
                //计算下一个24小时开始到出场时间的费用
                float suffixFee = chargeFee(feeParamHolder, orderInfo.getParkId(), billId, (int) (firstEnterTime + 3600 * 24),
                        (int) (endDate.getTime() / 1000),
                        0, false, orderInfo.getCarType(), false, 0,
                        null, null, null, false);
                //24小时最大收费-已支付金额，等于当次停车入场到下一个24小时起点最多收费
                return new BigDecimal(dayNightMaxfee).subtract(new BigDecimal(paidPriceSum))
                        .add(new BigDecimal(suffixFee));
            }
        }
        return null;
    }

    protected List<BaseFeeParamHolder.ExtraComputeFeePara> resolveHolsFeeParams(OrderInfo orderInfo, long beginTimeSec, long endTimeSec) {
        log.debug("节假日规则判定|{}|{}|{}|{}|{}", orderInfo.getParkId(), orderInfo.getOrderNum(), orderInfo.getPlateNum(), beginTimeSec, endTimeSec);
        List<BaseFeeParamHolder.ExtraComputeFeePara> partFeeParams = new ArrayList<>();
        ParkRegion parkRegion = NumberUtils.toPrimitive(orderInfo.getRegionId()) == 0 ?
                parkRegionService.lambdaQuery().eq(ParkRegion::getParkId, orderInfo.getParkId()).eq(ParkRegion::getFatherRelationId, 0).last(SqlConstant.LIMIT_ONE).one():
                parkRegionService.getById(orderInfo.getRegionId());
        List<RegionChargeconfig> chargeConfigs = regionChargeconfigService.lambdaQuery().eq(RegionChargeconfig::getParkId, orderInfo.getParkId())
                .eq(RegionChargeconfig::getRegionId, parkRegion.getId())
                .in(RegionChargeconfig::getLicensePlateType, RegionChargePlateTypeConsts.HOLIDAY, RegionChargePlateTypeConsts.HOLS_GREEN_PLATE)
                .list();
        if (CollectionUtils.isEmpty(chargeConfigs)) return partFeeParams; // 未配置节假日
        RegionChargeconfig holsChargeConfig = null, evHolsChargeConfig = null;
        for (RegionChargeconfig chargeconfig : chargeConfigs) {
            if (RegionChargePlateTypeConsts.HOLIDAY.equals(chargeconfig.getLicensePlateType())) holsChargeConfig = chargeconfig;
            else if (RegionChargePlateTypeConsts.HOLS_GREEN_PLATE.equals(chargeconfig.getLicensePlateType())) evHolsChargeConfig = chargeconfig;
        }
        assert holsChargeConfig != null;

        OrderCarInfo carInfo = orderCarInfoService.getByOrderNum(orderInfo.getOrderNum());
        boolean useEv = PlateColorEnum.GREEN.getDesc().equals(carInfo.getPlateColor()) && evHolsChargeConfig != null;
        String holsBillCode = useEv ? evHolsChargeConfig.getBilltypecode() : holsChargeConfig.getBilltypecode();
        if (StringUtils.isBlank(holsBillCode)) return partFeeParams;
        ParkChargeconfig chargeConfig = parkChargeConfigService.lambdaQuery().eq(ParkChargeconfig::getParkId, orderInfo.getParkId())
                .eq(ParkChargeconfig::getBilltypecode, holsBillCode).one();
        if (chargeConfig == null) return partFeeParams;

        // 整理时间段
        Object[] dateTypes = (useEv ? evHolsChargeConfig.getIsNewenergyIncludeWeekends() == 1 : holsChargeConfig.getIsIncludeWeekends() == 1) ?
                new Integer[]{NotWorkDay.DATE_TYPE_HOLIDAY, NotWorkDay.DATE_TYPE_WEEKEND} : new Integer[]{NotWorkDay.DATE_TYPE_HOLIDAY};
        LocalDateTime beginTime = DateUtils.toLocalDateTimeOfSecond(beginTimeSec), endTime = DateUtils.toLocalDateTimeOfSecond(endTimeSec);
        LocalDate beginDate = beginTime.toLocalDate(), endDate = endTime.toLocalDate();
        Set<LocalDate> notWorkDaySet = notWorkDayService.lambdaQuery().between(NotWorkDay::getYmd, beginDate, endDate)
                .in(NotWorkDay::getDateType, dateTypes)
                .orderByAsc(NotWorkDay::getYmd).list().stream()
                .map(NotWorkDay::getYmd).collect(Collectors.toSet());
        long beginEpochDay = beginDate.toEpochDay(), endEpochDay = endDate.toEpochDay();
        LocalDateTime lastBegin = beginTime, lastEnd = beginEpochDay == endEpochDay ? endTime : beginDate.plusDays(1).atStartOfDay();
        boolean lastFlag = notWorkDaySet.contains(beginDate);
        for (long i = beginEpochDay; i <= endEpochDay; i++) {
            LocalDate date = LocalDate.ofEpochDay(i);
            boolean flag = notWorkDaySet.contains(date);
            if (flag != lastFlag) {
                partFeeParams.add(new BaseFeeParamHolder.ExtraComputeFeePara()
                        .setStartTime(DateUtils.toEpochSecond(lastBegin))
                        .setEndTime(DateUtils.toEpochSecond(lastEnd))
                        .setBillId(lastFlag ? chargeConfig.getId() : null));
                lastBegin = date.atStartOfDay();
            }
            if (i != endEpochDay) {
                lastEnd = date.plusDays(1).atStartOfDay();
                lastFlag = flag;
            } else if (!lastBegin.equals(endTime)) {
                partFeeParams.add(new BaseFeeParamHolder.ExtraComputeFeePara()
                        .setStartTime(DateUtils.toEpochSecond(lastBegin))
                        .setEndTime(endTimeSec)
                        .setBillId(flag ? chargeConfig.getId() : null));
            }
        }
        return partFeeParams;
    }
}
