package com.icetech.park.service.park.impl;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.cloudcenter.domain.park.query.ParkTrusteeshipParam;
import com.icetech.common.domain.Page;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.park.dao.ParkTrusteeshipMapper;
import com.icetech.park.domain.entity.ParkTrusteeship;
import com.icetech.park.service.park.ParkTrusteeshipService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;

/**
 * 车场托管 服务实现类
 *
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @date 2023-06-28
 */
@Service
public class ParkTrusteeshipServiceImpl extends BaseServiceImpl<ParkTrusteeshipMapper, ParkTrusteeship> implements ParkTrusteeshipService {

    /**
     * Description: 根据ID查询 车场托管对象信息 <br>
     * Version1.0 2023-06-28 by wgy创建
     *
     * @param id 对象id
     * @return ParkTrusteeship
     */
    @Override
    public ParkTrusteeship getParkTrusteeshipById (Long id) {
        return getById(id);
    }

    /**
     * Description: 新增  车场托管对象信息 <br>
     * Version1.0 2023-06-28 by wgy创建
     *
     * @param entity 对象信息
     * @return Boolean 返回新增后主键
     */
    @Override
    public Boolean addParkTrusteeship (ParkTrusteeship entity) {
        return save(entity);
    }

    /**
     * Description: 修改 车场托管对象信息 <br>
     * Version1.0 2023-06-28 by wgy创建
     *
     * @param entity 对象信息
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean modifyParkTrusteeship (ParkTrusteeship entity) {
        return updateById(entity);
    }

    /**
     * Description: 删除  车场托管对象信息 <br>
     * Version1.0 2023-06-28 by wgy创建
     *
     * @param id 对象id
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean removeParkTrusteeshipById (Long id) {
        return removeById(id);
    }

    @Override
    public Boolean saveParkTrusteeship(ParkTrusteeship parkTrusteeship) {
        return saveOrUpdate(parkTrusteeship);
    }

    @Override
    public ParkTrusteeship getParkTrusteeshipByParkId(Long parkId) {
        return getOne(Wrappers.lambdaQuery(ParkTrusteeship.class).eq(ParkTrusteeship::getParkId, parkId));
    }

    @Override
    public Page<ParkTrusteeship> searchParkTrusteeship(ParkTrusteeshipParam parkTrusteeshipParam, Date currentDate) {
        LambdaQueryWrapper<ParkTrusteeship> wrapper = Wrappers.lambdaQuery(ParkTrusteeship.class);
        wrapper.eq(ParkTrusteeship::getStatus, 1);
        wrapper.in(CollectionUtils.isNotEmpty(parkTrusteeshipParam.getParkIds()), ParkTrusteeship::getParkId, parkTrusteeshipParam.getParkIds());
        wrapper.in(CollectionUtils.isNotEmpty(parkTrusteeshipParam.getInstitutionIds()), ParkTrusteeship::getInstitutionId, parkTrusteeshipParam.getInstitutionIds());
        wrapper.like(StringUtils.isNotEmpty(parkTrusteeshipParam.getSipNum()), ParkTrusteeship::getSipNum, parkTrusteeshipParam.getSipNum());
        wrapper.in(CollectionUtils.isNotEmpty(parkTrusteeshipParam.getTrusteeshipModels()), ParkTrusteeship::getTrusteeshipModel,
                parkTrusteeshipParam.getTrusteeshipModels());
        if (StringUtils.isNotEmpty(parkTrusteeshipParam.getStartTime()) && StringUtils.isNotEmpty(parkTrusteeshipParam.getEndTime())) {
            wrapper.le(ParkTrusteeship::getStartTime, parkTrusteeshipParam.getEndTime());
            wrapper.le(ParkTrusteeship::getEndTime, parkTrusteeshipParam.getStartTime());
        }
        if (Objects.nonNull(parkTrusteeshipParam.getDueFewDays())) {
            wrapper.le(ParkTrusteeship::getEndTime, DateUtil.offsetDay(currentDate, parkTrusteeshipParam.getDueFewDays()));
        }
        String currentDateStr = DateUtil.formatDate(currentDate);
        if (CollectionUtils.isNotEmpty(parkTrusteeshipParam.getTrusteeshipStatus())) {
            StringJoiner trusteeshipStatus = new StringJoiner(",");
            parkTrusteeshipParam.getTrusteeshipStatus().stream().map(Object::toString).forEach(trusteeshipStatus::add);
            wrapper.last(String.format(" and case when start_time > '%s' then 1 when start_time <= '%s' and end_time >= '%s' then" +
                            " 2 when  end_time < '%s' then 3 end in (%s) order by id desc",
                    currentDateStr, currentDateStr, currentDateStr, currentDateStr, trusteeshipStatus));
        } else {
            wrapper.orderByDesc(ParkTrusteeship::getId);
        }
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<ParkTrusteeship> page = page(wrapper, parkTrusteeshipParam.getPageNo(),
                parkTrusteeshipParam.getPageSize());
        return Page.instance((int) page.getPages(), page.getTotal(), page.getRecords());
    }

    @Override
    public List<ParkTrusteeship> getNearExpireWarnPark() {
        LambdaQueryWrapper<ParkTrusteeship> wrapper = Wrappers.lambdaQuery(ParkTrusteeship.class);
        wrapper.eq(ParkTrusteeship::getOpenWarn, 1);
        wrapper.last("  and end_time > DATE_FORMAT(NOW(), '%Y-%m-%d') and end_time  <= DATE_FORMAT(NOW() + INTERVAL near_expire_day DAY,'%Y-%m-%d')");
        return list(wrapper);
    }

    @Override
    public List<ParkTrusteeship> getExpireWarnPark() {
        LambdaQueryWrapper<ParkTrusteeship> wrapper = Wrappers.lambdaQuery(ParkTrusteeship.class);
        wrapper.eq(ParkTrusteeship::getOpenWarn, 1);
        wrapper.last(" and end_time < DATE_FORMAT(NOW(), '%Y-%m-%d') and  end_time >= DATE_FORMAT(NOW() - INTERVAL expire_day DAY,'%Y-%m-%d')");
        return list(wrapper);
    }
}
