package com.icetech.park.service.park.impl;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.Page;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.StringUtils;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.park.dao.park.ParkRecoveryApplyMapper;
import com.icetech.park.domain.entity.park.ParkRecovery;
import com.icetech.park.domain.entity.park.ParkRecoveryApply;
import com.icetech.park.domain.request.ParkRecoveryApplyQuery;
import com.icetech.park.service.park.ParkRecoveryApplyService;
import com.icetech.park.service.park.ParkRecoveryService;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.StringJoiner;

import static cn.hutool.core.date.DatePattern.NORM_DATE_FORMAT;

/**
 * 车场欠费追缴申请记录 服务实现类
 *
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @date 2023-07-20
 */
@Service
@RefreshScope
@RequiredArgsConstructor
public class ParkRecoveryApplyServiceImpl extends BaseServiceImpl<ParkRecoveryApplyMapper, ParkRecoveryApply> implements ParkRecoveryApplyService {
    
    private final ParkRecoveryService parkRecoveryService;

    @Value(value = "${park.recovery.trialDays}")
    private Integer trialDays;

    /**
     * Description: 根据ID查询 车场欠费追缴申请记录对象信息 <br>
     * Version1.0 2023-07-20 by wgy创建
     *
     * @param id 对象id
     * @return ParkRecoveryApply
     */
    @Override
    public ParkRecoveryApply getParkRecoveryApplyById (Long id) {
        return getById(id);
    }

    /**
     * Description: 新增  车场欠费追缴申请记录对象信息 <br>
     * Version1.0 2023-07-20 by wgy创建
     *
     * @param entity 对象信息
     * @return Boolean 返回新增后主键
     */
    @Override
    public Boolean addParkRecoveryApply (ParkRecoveryApply entity) {
        return save(entity);
    }

    /**
     * Description: 修改 车场欠费追缴申请记录对象信息 <br>
     * Version1.0 2023-07-20 by wgy创建
     *
     * @param entity 对象信息
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean modifyParkRecoveryApply (ParkRecoveryApply entity) {
        return updateById(entity);
    }

    /**
     * Description: 删除  车场欠费追缴申请记录对象信息 <br>
     * Version1.0 2023-07-20 by wgy创建
     *
     * @param id 对象id
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean removeParkRecoveryApplyById (Long id) {
        return removeById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean parkRecoveryApply(ParkRecoveryApply parkRecoveryApply) {
        if (parkRecoveryApply.getApplyType() == 1) {
            parkRecoveryApply.setOpenStatus(1);
            parkRecoveryApply.setOpenBy(parkRecoveryApply.getOperator());
            parkRecoveryApply.setOpenTime(new Date());
            parkRecoveryApply.setStartTime(DateUtil.format(new Date(), NORM_DATE_FORMAT));
            parkRecoveryApply.setEndTime(DateUtil.format(DateUtil.offsetDay(new Date(), trialDays - 1), NORM_DATE_FORMAT));
            // 判断之前是否开通过
            ParkRecovery parkRecoveryByParkId = parkRecoveryService.getParkRecoveryByParkId(parkRecoveryApply.getParkId());
            if (Objects.nonNull(parkRecoveryByParkId)) {
                throw new ResponseBodyException(CodeConstantsEnum.ERROR_402.getCode(), "该项目已经试用过。");
            }
        }
        parkRecoveryApply.setOrderNum(CodeTools.GenerateOrderNum());
        parkRecoveryApply.setApplyTime(new Date());
        // 试用开通增加 数据
        if (addParkRecoveryApply(parkRecoveryApply) && parkRecoveryApply.getApplyType() == 1) {
            ParkRecovery parkRecovery = new ParkRecovery();
            BeanUtils.copyProperties(parkRecoveryApply, parkRecovery);
            parkRecovery.setStartTime(DateUtil.format(new Date(), NORM_DATE_FORMAT));
            parkRecovery.setEndTime(DateUtil.format(DateUtil.offsetDay(new Date(), trialDays - 1), NORM_DATE_FORMAT));
            parkRecoveryService.save(parkRecovery);
        }
        return true;
    }

    @Override
    public Page<ParkRecoveryApply> getParkRecoveryApplyPage(ParkRecoveryApplyQuery parkRecoveryApplyQuery, List<Long> parkIds) {
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<ParkRecoveryApply> page = page(getWrapper(parkRecoveryApplyQuery, parkIds),
                parkRecoveryApplyQuery.getPageNo(), parkRecoveryApplyQuery.getPageSize());
        return Page.instance((int) page.getPages(), page.getTotal(), page.getRecords());
    }

    @Override
    public BigDecimal totalServiceCharge(ParkRecoveryApplyQuery parkRecoveryApplyQuery, List<Long> parkIds) {
        LambdaQueryWrapper<ParkRecoveryApply> wrapper = getWrapper(parkRecoveryApplyQuery, parkIds);
        wrapper.select(ParkRecoveryApply::getServiceCharge);
        List<ParkRecoveryApply> recordList = list(wrapper);
        if (CollectionUtils.isNotEmpty(recordList)) {
            return recordList.stream().map(ParkRecoveryApply::getServiceCharge)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return BigDecimal.ZERO;
    }

    @Override
    public ParkRecoveryApply getParkRecoveryApplyByOrder(String orderNum) {
        return getOne(Wrappers.lambdaQuery(ParkRecoveryApply.class).eq(ParkRecoveryApply::getOrderNum, orderNum));
    }

    @Override
    public ParkRecoveryApply getParkRecoveryApplyByOrder(Long parkId) {
        List<ParkRecoveryApply> applyList = list(Wrappers.lambdaQuery(ParkRecoveryApply.class)
                .eq(ParkRecoveryApply::getParkId, parkId)
                .eq(ParkRecoveryApply::getPayModel, 3));
        if (CollectionUtils.isEmpty(applyList)) {
            return null;
        }
        Date currentDate = new Date();
        Optional<ParkRecoveryApply> applyOptional = applyList.stream()
                .filter(a -> DateUtil.compare(currentDate, DateUtil.parse(a.getStartTime())) >= 0 && DateUtil.compare(currentDate, DateUtil.offsetDay(DateUtil.parse(a.getEndTime()), 1)) < 0)
                .findFirst();
        return applyOptional.orElse(null);
    }

    @Override
    public Integer getApplyCount(List<Long> parkIds) {
        return count(Wrappers.lambdaQuery(ParkRecoveryApply.class)
                .eq(ParkRecoveryApply::getOpenStatus, 0)
                .in(CollectionUtils.isNotEmpty(parkIds) && !parkIds.contains(-1L), ParkRecoveryApply::getParkId,
                parkIds));
    }

    @Override
    public ParkRecoveryApply getParkRecoveryApplyByParkId(Long parkId) {
        return selectLimitOne(Wrappers.lambdaQuery(ParkRecoveryApply.class)
                .eq(ParkRecoveryApply::getParkId, parkId).orderByDesc(ParkRecoveryApply::getParkId));
    }

    @Override
    public ParkRecoveryApply getValidateRecoveryApplyByParkId(Long parkId) {
        String todayDate = DateTools.getFormat(DateTools.DF_, new Date());
        return selectLimitOne(Wrappers.lambdaQuery(ParkRecoveryApply.class)
                .eq(ParkRecoveryApply::getParkId, parkId)
                .le(ParkRecoveryApply::getStartTime, todayDate)
                .ge(ParkRecoveryApply::getEndTime, todayDate)
                .orderByDesc(ParkRecoveryApply::getParkId));
    }

    private LambdaQueryWrapper<ParkRecoveryApply> getWrapper(ParkRecoveryApplyQuery parkRecoveryApplyQuery, List<Long> parkIds) {
        LambdaQueryWrapper<ParkRecoveryApply> wrapper = Wrappers.lambdaQuery(ParkRecoveryApply.class);
        wrapper.eq(ParkRecoveryApply::getStatus, 1);
        wrapper.in(CollectionUtils.isNotEmpty(parkRecoveryApplyQuery.getParkId()), ParkRecoveryApply::getParkId, parkRecoveryApplyQuery.getParkId());
        wrapper.in(CollectionUtils.isNotEmpty(parkRecoveryApplyQuery.getInstitutionId()), ParkRecoveryApply::getInstitutionId, parkRecoveryApplyQuery.getInstitutionId());
        wrapper.in(CollectionUtils.isNotEmpty(parkRecoveryApplyQuery.getApplyType()), ParkRecoveryApply::getApplyType, parkRecoveryApplyQuery.getApplyType());
        wrapper.between(StringUtils.isNotEmpty(parkRecoveryApplyQuery.getOpenStartTime()) && StringUtils.isNotEmpty(parkRecoveryApplyQuery.getOpenEndTime()),
                ParkRecoveryApply::getOpenTime, parkRecoveryApplyQuery.getOpenStartTime(), parkRecoveryApplyQuery.getOpenEndTime());
        wrapper.in(!parkIds.contains(-1L), ParkRecoveryApply::getParkId, parkIds);
        String currentDate = DateUtil.formatDate(new Date());
        if (CollectionUtils.isNotEmpty(parkRecoveryApplyQuery.getRecordStatus())) {
            StringJoiner recordStatus = new StringJoiner(",");
            parkRecoveryApplyQuery.getRecordStatus().stream().map(Object::toString).forEach(recordStatus::add);
            wrapper.last(String.format(" and case when open_status = 0 then 1 when start_time > '%s' then 1 when start_time <= '%s' and end_time >= '%s' then" +
                            " 2 when  end_time < '%s' then 3 end in (%s) order by id desc ",
                    currentDate, currentDate, currentDate, currentDate, recordStatus));
        } else {
            wrapper.orderByDesc(ParkRecoveryApply::getId);
        }
        return wrapper;
    }
}
