package com.icetech.park.service.park.impl;

import com.icetech.cloudcenter.api.park.ParkLicenseService;
import com.icetech.cloudcenter.domain.response.AccessInfoDto;
import com.icetech.cloudcenter.domain.response.ParkLicenseDto;
import com.icetech.park.dao.access.AccessInfoDao;
import com.icetech.park.dao.park.ParkLicenseDao;
import com.icetech.park.domain.entity.AccessInfo;
import com.icetech.park.domain.entity.park.ParkLicense;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

@Slf4j
@Service("parkLicenseService")
@Transactional
public class ParkLicenseServiceImpl implements ParkLicenseService {
    private int LICENSE_STATUS_OK = 1;
    private int LICENSE_STATUS_NO = 2;
    private int LICENSE_STATUS_USE = 3;
    private int LICENSE_STATUS_INVALID= 4;
    private int LICENSE_STATUS_OVER= 5;
    @Autowired
    private ParkLicenseDao parkLicenseDao;
    @Autowired
    private AccessInfoDao accessInfoDao;

    @Override
    public ObjectResponse<ParkLicenseDto> selectParkLicense(String licenseNumber) {
        ParkLicense parkLicense = parkLicenseDao.selectByLicense(licenseNumber);
        if (parkLicense != null) {
            ParkLicenseDto parkLicenseDto = new ParkLicenseDto();
            parkLicenseDto.setId(parkLicense.getId());
            parkLicenseDto.setLicenseNumber(parkLicense.getLicenseNumber());
            parkLicenseDto.setGrantStatus(parkLicense.getGrantStatus());
            parkLicenseDto.setEnableStatus(parkLicense.getEnableStatus());
            parkLicenseDto.setType(parkLicense.getType());
            return ObjectResponse.success(parkLicenseDto);
        }
        return ObjectResponse.failed(CodeConstants.ERROR_404);
    }

    @Override
    public ObjectResponse<ParkLicenseDto> registerParkLicense(Long parkId, String licenseNumber) {
        ParkLicenseDto parkLicenseDto = new ParkLicenseDto();
        ParkLicense parkLicense = parkLicenseDao.selectByLicense(licenseNumber);
        if (parkLicense == null) {
            parkLicenseDto.setLicenceStatus(2);
            return ObjectResponse.instance(CodeConstants.ERROR, "无效", parkLicenseDto);
        }
        if (parkLicense.getGrantStatus() == 1) {//未发放
            parkLicenseDto.setLicenceStatus(LICENSE_STATUS_NO);
            return ObjectResponse.instance(CodeConstants.ERROR, "未发放", parkLicenseDto);
        }else if (parkLicense.getEnableStatus() == 3) {//作废
            parkLicenseDto.setLicenceStatus(LICENSE_STATUS_INVALID);
            return ObjectResponse.instance(CodeConstants.ERROR, "作废", parkLicenseDto);
        } else if (parkLicense.getEnableStatus() == 4) {//过期
            parkLicenseDto.setLicenceStatus(LICENSE_STATUS_OVER);
            return ObjectResponse.instance(CodeConstants.ERROR, "过期", parkLicenseDto);
        }else  if (parkLicense.getEnableStatus() == 2) {//使用中
            if (parkLicense.getParkId().intValue() != parkId.intValue()) {
                parkLicenseDto.setLicenceStatus(LICENSE_STATUS_USE);
                return ObjectResponse.instance(CodeConstants.ERROR, "已被使用", parkLicenseDto);
            } else {
                parkLicenseDto = JsonUtils.convert2bean(parkLicense, ParkLicenseDto.class);
                parkLicenseDto.setLicenceStatus(LICENSE_STATUS_OK);
                if(parkLicense.getType() == 2){
                    int day = daysBetween(new Date(), parkLicense.getInvalidTime());
                    parkLicenseDto.setUseDays(day);
                    if(day <= 0){
                        parkLicenseDto.setLicenceStatus(LICENSE_STATUS_OVER);
                        parkLicenseDto.setUseDays(0);
                        parkLicense.setEnableStatus(4);
                        parkLicenseDao.updateLicense(parkLicense);
                    }
                }
                return ObjectResponse.success(parkLicenseDto);
            }
        } else {
            //作废以前
            if(parkLicense.getParkId() != null){
                ParkLicense parkLicense1 = parkLicenseDao.selectByParkId(String.valueOf(parkId));
                parkLicense1.setEnableStatus(3);
                parkLicenseDao.updateLicense(parkLicense1);
            }
            //首次注册
            parkLicense.setParkId(parkId);
            parkLicense.setEnableTime(new Date());
            parkLicense.setEnableStatus(2);
            parkLicense.setUpdateTime(new Date());
            if (parkLicense.getType() == 2 && parkLicense.getValidDays() > 0) {
                Date d = new Date();
                Calendar ca = Calendar.getInstance();
                ca.add(Calendar.DATE, parkLicense.getValidDays());
                d = ca.getTime();
                parkLicense.setInvalidTime(d);
            }

            int i = parkLicenseDao.updateLicense(parkLicense);
            if (i > 0) {
                parkLicenseDto = JsonUtils.convert2bean(parkLicense, ParkLicenseDto.class);
                int day = daysBetween(parkLicense.getEnableTime(), parkLicense.getInvalidTime());
                parkLicenseDto.setUseDays(day);
                parkLicenseDto.setLicenceStatus(LICENSE_STATUS_OK);
                return ObjectResponse.success(parkLicenseDto);
            }



        }
        return ObjectResponse.failed(CodeConstants.ERROR);
    }

    @Override
    public ObjectResponse<AccessInfoDto> getAccessInfoByType(int type) {
        AccessInfo accessInfo = accessInfoDao.selectByType(type);
        if (accessInfo != null) {
            AccessInfoDto accessInfoDto = new AccessInfoDto();
            accessInfoDto.setId(accessInfo.getId());
            accessInfoDto.setAccessKeyId(accessInfo.getAccessKeyId());
            accessInfoDto.setAccessKeySecret(accessInfo.getAccessKeySecret());
            accessInfoDto.setName(accessInfo.getAccessName());
            accessInfoDto.setType(accessInfo.getAccessType());
            return ObjectResponse.success(accessInfoDto);
        }
        return ObjectResponse.failed(CodeConstants.ERROR_404);
    }

    /**
     * 计算两个日期之间相差的天数
     *
     * @param smdate
     * @param bdate
     * @return 相差天数
     * @throws Exception
     */

    private int daysBetween(Date smdate, Date bdate){
        int result = 0;
        if (smdate == null || bdate == null) {
            return result;
        }
        try {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            smdate = sdf.parse(sdf.format(smdate));
            bdate = sdf.parse(sdf.format(bdate));
            Calendar cal = Calendar.getInstance();
            cal.setTime(smdate);
            long time1 = cal.getTimeInMillis();
            cal.setTime(bdate);
            long time2 = cal.getTimeInMillis();
            long between_days = (time2 - time1) / (1000 * 3600 * 24);
            result =  Integer.parseInt(String.valueOf(between_days));
        }catch (Exception e){
            log.error("处理失败: {}. smdate[{}], bdate[{}]", e.getMessage(), smdate, bdate, e);
        }
        return result;

    }
}
