package com.icetech.park.service.park.impl;

import java.util.*;
import java.util.stream.Collectors;

import com.icetech.basics.domain.entity.park.BasePark;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.dao.area.AreaCityDao;
import com.icetech.basics.dao.area.AreaDistrictDao;
import com.icetech.basics.dao.area.AreaProvinceDao;
import com.icetech.basics.domain.entity.AreaCity;
import com.icetech.basics.domain.entity.AreaDistrict;
import com.icetech.basics.domain.entity.AreaProvince;
import com.icetech.third.domain.entity.third.SendInfo;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.park.ParkConfigInfo;
import com.icetech.cloudcenter.domain.park.ParkConfigInfoFind;
import com.icetech.third.service.third.SendInfoServiceImpl;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import com.icetech.cloudcenter.api.park.ParkConfigService;
import com.icetech.basics.dao.park.ParkConfigDao;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.cloudcenter.domain.response.ParkConfigDto;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import org.springframework.transaction.annotation.Transactional;

@Service("parkConfigService")
@Slf4j
public class ParkConfigServiceImpl extends BaseServiceImpl<ParkConfigDao, ParkConfig> implements ParkConfigService {
    @Autowired
    private ParkService parkService;
    @Autowired
    private ParkConfigDao parkConfigDao;
    @Autowired
    private AreaCityDao areaCityDao;
    @Autowired
    private AreaDistrictDao areaDistrictDao;
    @Autowired
    private AreaProvinceDao areaProvinceDao;
    @Autowired
    private SendInfoServiceImpl sendInfoService;
    @Autowired
    private StringRedisTemplate redisTemplate;

    @Override
    public ObjectResponse<ParkConfigDto> selectByParkCode(String parkCode) {
        ParkConfig parkConfig = parkService.getParkConfig(parkCode).getData();
        if (Objects.isNull(parkConfig)){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        ParkConfigDto parkConfigDto = new ParkConfigDto();
        BeanUtils.copyProperties(parkConfig, parkConfigDto);
        return ObjectResponse.success(parkConfigDto);
    }

    @Override
    public ObjectResponse<ParkConfigDto> selectByParkId(Long parkId) {
        ParkConfig parkConfig = parkService.getParkConfig(parkId).getData();
        if (Objects.isNull(parkConfig)){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        ParkConfigDto parkConfigDto = new ParkConfigDto();
        BeanUtils.copyProperties(parkConfig, parkConfigDto);
        return ObjectResponse.success(parkConfigDto);
    }

    /**
     * 新建车场配置
     *
     * @param parkConfigInfo
     * @return
     */
    @Override
    public ObjectResponse saveParkConfigInfo(ParkConfigInfo parkConfigInfo) {
        Park park = parkService.findByParkCode(parkConfigInfo.getParkCode()).getData();
        if (Objects.nonNull(park)) {
            parkConfigInfo.setParkId(park.getId().intValue());
            Integer parkConfigSize = parkConfigDao.selectCountByParkId(park.getId());
            if (parkConfigSize <= 0){
                parkConfigDao.saveParkConfigInfo(parkConfigInfo);
                removeParkConfigFromRedis(park.getId(), park.getParkCode());
                return ObjectResponse.success();
            }
        }
        return ObjectResponse.failed(CodeConstants.ERROR_406);
    }

    /**
     * 移除redis数据
     * @param parkId
     */
    private void removeParkConfigFromRedis(Long parkId, String parkCode) {
        redisTemplate.delete(Arrays.asList(RedisKeyConstants.KEY_PREFIX_PARK_CONFIG_PARK + parkId,
                RedisKeyConstants.KEY_PREFIX_PARK_CONFIG_PARK_CODE, parkCode));
    }

    /**
     * 更新车场配置
     *
     * @param parkConfigInfo
     * @return
     */
    @Override
    public ObjectResponse<Void> updateParkConfigInfo(ParkConfigInfo parkConfigInfo) {
        ObjectResponse<Park> parkResp = parkService.findByParkCode(parkConfigInfo.getParkCode());
        if (ObjectResponse.isSuccess(parkResp)) {
            Park park = parkResp.getData();
            ObjectResponse<ParkConfig> parkConfigResp = parkService.getParkConfig(park.getId());
            ObjectResponse.notError(parkConfigResp);
            parkConfigInfo.setParkId(park.getId().intValue());
            parkConfigDao.updateParkConfigInfo(parkConfigInfo);
            sendInfoService.save(new SendInfo(park.getId(), parkConfigResp.getData().getId(), DownServiceEnum.车场配置.getServiceType()));
            removeParkConfigFromRedis(park.getId(), park.getParkCode());
            return ObjectResponse.success();
        }
        return ObjectResponse.failed(CodeConstants.ERROR_402);
    }

    @Override
    public ObjectResponse<ParkConfigInfoFind> selectParkConfigInfoByParkCode(String parkCode) {
        ParkConfigInfoFind parkConfigInfo = parkConfigDao.selectParkConfigInfoByParkCode(parkCode);
        log.info("selectParkConfigInfoByParkCode parkConfigInfo [{}] ",parkConfigInfo);
        return ObjectResponse.success(parkConfigInfo);
    }

    @Override
    public ObjectResponse<List<AreaCity>> selectAllAreaCityData(Integer id) {
        if (Objects.isNull(id)){
            return ObjectResponse.success(areaCityDao.selectAllData());
        }
        return ObjectResponse.success(areaCityDao.selectByUpperId(id));
    }

    @Override
    public ObjectResponse<List<AreaDistrict>> selectAllAreaDistrictData(Integer id) {
        if (Objects.isNull(id)){
            return ObjectResponse.success(areaDistrictDao.selectAllData());
        }
        return ObjectResponse.success(areaDistrictDao.selectByUpperId(id));
    }

    @Override
    public ObjectResponse<List<AreaProvince>> selectAllAreaProvinceData(Integer id) {
        if (Objects.isNull(id)){
            return ObjectResponse.success(areaProvinceDao.selectAllData());
        }
        return ObjectResponse.success(Collections.singletonList(areaProvinceDao.selectById(id)));
    }

    @Override
    @Transactional
    public ObjectResponse<Boolean> batchUpdateParkConfigPayWayByParkId(Map<Long, String> payWayMap) {
        if (payWayMap == null || payWayMap.isEmpty()) return ObjectResponse.success();
        List<BasePark> parks = parkService.getBaseParkList(payWayMap.keySet()).getData();
        if (CollectionUtils.isEmpty(parks)) return ObjectResponse.success();
        List<ParkConfig> updateParks = payWayMap.entrySet().stream()
                .map(entry -> new ParkConfig().setParkId(entry.getKey()).setIsEpayment(1).setEPayment(entry.getValue()))
                .collect(Collectors.toList());
        boolean result = updateBatchByColumn(updateParks, ParkConfig::getParkId);
        parks.forEach(park -> removeParkConfigFromRedis(park.getId(), park.getParkCode()));
        return ObjectResponse.success(result);
    }
}
