package com.icetech.park.service.order.impl.exit;

import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.order.OrderSimpleExitService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.request.SimpleExitRequest;
import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.order.dao.OrderDiscountDao;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.dao.OrderPayDao;
import com.icetech.park.dao.other.ChannelAlarmDao;
import com.icetech.cloudcenter.domain.charge.dto.OrderSumFeeDto;
import com.icetech.park.domain.entity.ChannelAlarm;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.websocket.WebsocketPushData;
import com.icetech.park.service.order.impl.OrderPayServiceImpl;
import com.icetech.park.service.other.impl.AutopayServiceImpl;
import com.icetech.basics.service.redis.RedisMsgListener;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderDiscount;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;

@Slf4j
@Service("orderSimpleExitService")
public class OrderSimpleExitServiceImpl implements OrderSimpleExitService {
    @Autowired
    private OrderInfoDao orderInfoDao;
    @Resource
    private OrderService orderService;
    @Resource
    private OrderCarInfoDao orderCarInfoDao;
    @Autowired
    private ParkService parkService;
    @Autowired
    private AutopayServiceImpl autopayService;
    @Autowired
    private OrderPayDao orderPayDao;
    @Autowired
    private OrderDiscountDao orderDiscountDao;
    @Autowired
    private OrderPayServiceImpl orderPayService;
    @Autowired
    private ChannelAlarmDao channelAlarmDao;
    @Autowired
    private StringRedisTemplate redisTemplate;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;

    @Transactional
    @Override
    public ObjectResponse exit(SimpleExitRequest exitRequest, String parkCode) {
        if (exitRequest.getIsExceptionExit() != null && exitRequest.getIsExceptionExit() == 1){
            return exit(exitRequest, parkCode, OrderStatusConstants.EXCEPTION);
        }else{
            return exit(exitRequest, parkCode, OrderStatusConstants.LEAVED_PARK);
        }
    }

    public ObjectResponse exit(SimpleExitRequest exitRequest, String parkCode, Integer status) {
        Long parkId = exitRequest.getParkId();
        String orderId = exitRequest.getOrderId();
        String orderNum = exitRequest.getOrderNum();
        if(StringUtils.isBlank(orderId) && StringUtils.isBlank(orderNum)){
            return ObjectResponse.failed(CodeConstants.ERROR_402, "orderId不存在");
        }

        String channelCode = exitRequest.getChannelId();
        ParkInoutdevice parkChannel = parkService.getInoutDeviceByCode(channelCode).getData();
        if (parkChannel == null) {
            log.info("[车辆离场服务-简单版] 通道编号无效：{}", channelCode);
            return ObjectResponse.failed(CodeConstants.ERROR_402, "channelId未被注册");
        }

        String imgFileName = exitRequest.getExitImage();
        if (imgFileName != null && !imgFileName.contains("/image/")){
            String date = DateTools.getFormat(DateTools.DF_, new Date());
            String[] ymd = date.split("-");
            imgFileName = parkCode + "/image/" + ymd[0] + ymd[1] + "/" + ymd[2] + "/"
                    + exitRequest.getExitImage();
        }
        OrderInfo orderInfo = new OrderInfo();
        orderInfo.setLocalOrderNum(orderId);
        orderInfo.setParkId(parkId);
        OrderInfo orderInfoResult = orderInfoDao.selectLimitOneOrderByEnterDesc(orderInfo);
        if (orderInfoResult == null) {
            OrderInfo orderInfoQuery = new OrderInfo();
            orderInfoQuery.setOrderNum(orderNum);
            orderInfoQuery.setParkId(parkId);
            orderInfoResult = orderInfoDao.selectLimitOneOrderByEnterDesc(orderInfo);
        }
        if (orderInfoResult == null){
            log.info("[离场服务-简单版] 未查询到在场订单，参数：[{},{}]",parkId, orderId);
            return ObjectResponse.success();
        }

        int serviceStatus = orderInfoResult.getServiceStatus();
        if (serviceStatus == OrderStatusConstants.LEAVED_PARK){
            log.info("[离场服务-简单版] 重复离场，重复请求参数：{}",exitRequest);
            return new ObjectResponse(CodeConstants.ERROR_405,
                    CodeConstants.getName(CodeConstants.ERROR_405));
        }
        //修改订单状态为离场
        orderInfoResult.setServiceStatus(status);
        orderInfoResult.setExitTime(exitRequest.getExitTime());
        orderInfoResult.setTotalPrice(exitRequest.getTotalAmount());
        orderInfoResult.setPaidPrice(exitRequest.getPaidAmount());
        orderInfoResult.setDiscountPrice(exitRequest.getDiscountAmount());
        orderInfoResult.setOddStatus(exitRequest.getExceptionReason());
        if (exitRequest.getType() != null){
            orderInfoResult.setType(exitRequest.getType());
        }
        if (exitRequest.getCarType() != null){
            orderInfoResult.setCarType(exitRequest.getCarType());
        }
        orderInfoResult.setExitChannelId(exitRequest.getChannelId());
        orderService.updateOrderWithPush(orderInfoResult);
        log.info("[离场服务-简单版] 修改订单主信息完成，orderId：{}", orderId);

        OrderPay orderPay = new OrderPay();
        orderPay.setParkId(parkId);
        orderPay.setOrderNum(orderInfoResult.getOrderNum());
        OrderSumFeeDto orderSumFeeDto = orderPayDao.sumFee(orderPay);
        float totalPrice = orderSumFeeDto == null ? 0 : orderSumFeeDto.getTotalPrice();
        float discountPrice = orderSumFeeDto == null ? 0 : orderSumFeeDto.getDiscountPrice();
        float totalAmount = Float.parseFloat(exitRequest.getTotalAmount());
        float paidAmount = Float.parseFloat(exitRequest.getPaidAmount());
        float discountAmount = Float.parseFloat(exitRequest.getDiscountAmount());
        // 上报金额大于平台已记录金额 && 实收+优惠 = 总应收，平台补全金额
        if (totalPrice < totalAmount && paidAmount + discountAmount == totalAmount){
            log.info("车牌号：{} 在平台记录的总费用为：{}，本地上报的总费用为：{}", exitRequest.getPlateNum(), totalPrice, totalAmount);
            // 补现金缴费记录
            float totalPriceNew = totalAmount - totalPrice;
            float discountPriceNew = discountAmount - discountPrice;
            Long lastPayTime = orderSumFeeDto == null ? null : orderSumFeeDto.getLastPayTime();
            saveOrderPay(exitRequest, parkChannel, orderPay, totalPriceNew, discountPriceNew, lastPayTime);
        }

        OrderCarInfo carInfo = orderCarInfoDao.selectByOrderNum(orderInfoResult.getOrderNum());
        AssertTools.notNull(carInfo, CodeConstants.ERROR, "平台订单数据不完整");
        carInfo.setExitImage(imgFileName);
        carInfo.setExitChannelId(exitRequest.getChannelId());
        carInfo.setExitNo(parkChannel.getInandoutName());
        carInfo.setExitReliability(exitRequest.getReliability());
        orderCarInfoDao.updateById(carInfo);
        log.info("[离场服务-简单版] 更新订单出入表完成，orderId：{}", orderId);

        OrderInfo finalOrderInfoResult = orderInfoResult;
        asyncExecutor.execute(ThreadUtils.wrapTrace(() -> {
            //调用支付中心离场通知(只有临时车和开启无感支付才需要调用接口)
            autopayService.exitNotify(finalOrderInfoResult, parkCode);

            sendWebsocketMessage(exitRequest, parkCode, carInfo, finalOrderInfoResult);
            // 更新通道的报警记录为已处理
            if (exitRequest.getChannelId() != null){
                ChannelAlarm channelAlarm = new ChannelAlarm();
                channelAlarm.setParkId(parkId);
                channelAlarm.setChannelCode(exitRequest.getChannelId());
                channelAlarm.setStatus(ChannelAlarm.Status.已处理.getStatus());
                channelAlarmDao.update(channelAlarm);
            }
        }));
        return ObjectResponse.success();
    }

    private void saveOrderPay(SimpleExitRequest exitRequest, ParkInoutdevice parkInoutdevice, OrderPay orderPay, float totalPriceNew, float discountPriceNew, Long lastPayTime) {
        String tradeNo = CodeTools.GenerateTradeNo();
        orderPay.setTradeNo(tradeNo);
        orderPay.setPayStatus(PayStatusConstants.PAID);
        orderPay.setIsSync(0);
        orderPay.setTotalPrice(String.valueOf(totalPriceNew));
        orderPay.setPaidPrice(String.valueOf(totalPriceNew - discountPriceNew));
        orderPay.setDiscountPrice(String.valueOf(discountPriceNew));
        orderPay.setPayTime(exitRequest.getExitTime());
        orderPay.setOrderTime(exitRequest.getExitTime());
        orderPay.setLastPayTime(lastPayTime);
        orderPay.setPayDate(new Date(exitRequest.getExitTime() * 1000));
        //默认现金
        orderPay.setPayWay(exitRequest.getPayWay() == null ? 1 : exitRequest.getPayWay());
        orderPay.setPayChannel(1);//出口缴费
        orderPay.setPayTerminal(parkInoutdevice.getInandoutName());
        orderPayService.addOrderPay(orderPay);
       if (discountPriceNew > 0){
            List<OrderDiscount> orderDiscounts = orderDiscountDao.selectByOrderNum(orderPay.getOrderNum(), 0);
            if (orderDiscounts != null && orderDiscounts.size() > 0){
                OrderDiscount discount_result = orderDiscounts.get(0);
                orderPayService.useDiscount((double) discountPriceNew, exitRequest.getParkId(), tradeNo, exitRequest.getExitTime(), orderPay.getOrderNum(), discount_result);
            }
        }
    }
    /**
     * 出入场通知 redis 订阅发布
     * @param exitRequest
     * @param parkCode
     * @param orderInfo
     */
    private void sendWebsocketMessage(SimpleExitRequest exitRequest, String parkCode, OrderCarInfo carInfo, OrderInfo orderInfo){
        WebsocketPushData websocketPushData = new WebsocketPushData();
        websocketPushData.setRecordType(2);
        websocketPushData.setEnterTime(orderInfo.getEnterTime().intValue());
        websocketPushData.setExitTime(exitRequest.getExitTime().intValue());
        websocketPushData.setType(exitRequest.getType());
        websocketPushData.setCarType(exitRequest.getCarType());
        websocketPushData.setChannelId(exitRequest.getChannelId());
        websocketPushData.setEnterNo(carInfo.getEnterNo());
        websocketPushData.setExitNo(carInfo.getExitNo());
        websocketPushData.setParkCode(parkCode);
//        websocketPushData.setInoutEvent(orderEnexRecord.getInoutEvent());
//        websocketPushData.setInoutOpening(orderEnexRecord.getInoutOpening());
        websocketPushData.setPlateNum(exitRequest.getPlateNum());
        websocketPushData.setPlateColor(carInfo.getPlateColor());
        websocketPushData.setTotalPrice(exitRequest.getTotalAmount());
        websocketPushData.setPaidPrice(exitRequest.getPaidAmount());
        websocketPushData.setDiscountPrice(exitRequest.getDiscountAmount());
        websocketPushData.setPayStatus(String.valueOf(orderInfo.getServiceStatus()));
        websocketPushData.setOrderNum(orderInfo.getOrderNum());
        log.info("sendWebsocketMessage  >> {}",parkCode);
        redisTemplate.convertAndSend(RedisMsgListener.TOPIC, JsonUtils.toString(websocketPushData));
    }
}
