package com.icetech.park.service.order.impl.exit;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.cloudcenter.api.order.OrderCarInfoService;
import com.icetech.cloudcenter.api.order.OrderExitService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.dao.OrderSonCarInfoDao;
import com.icetech.order.dao.OrderSonInfoDao;
import com.icetech.park.dao.other.ChannelAlarmDao;
import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.park.dao.park.ParkVisitDao;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.domain.entity.ChannelAlarm;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderSonCarInfo;
import com.icetech.order.domain.entity.OrderSonInfo;
import com.icetech.order.domain.entity.OrderTrack;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.park.domain.entity.park.ParkVisit;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.cloudcenter.domain.request.ExitCommonRequest;
import com.icetech.cloudcenter.domain.request.ExitRequest;
import com.icetech.cloudcenter.domain.websocket.WebsocketPushData;
import com.icetech.order.service.OrderNotpayService;
import com.icetech.park.service.order.impl.ExitPayDealServiceImpl;
import com.icetech.order.service.impl.OrderSonInfoServiceImpl;
import com.icetech.order.service.impl.OrderTrackServiceImpl;
import com.icetech.park.service.other.impl.AutopayServiceImpl;
import com.icetech.basics.service.redis.RedisMsgListener;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.AssertTools;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service("orderExitService")
public class OrderExitServiceImpl implements OrderExitService {
    @Resource
    private OrderInfoDao orderInfoDao;
    @Resource
    private OrderService orderService;
    @Resource
    private OrderCarInfoDao orderCarInfoDao;
    @Autowired
    private ParkService parkService;
    @Autowired
    private ExitPayDealServiceImpl exitPayDealService;
    @Autowired
    private AutopayServiceImpl autopayService;
    @Autowired
    private ParkVisitDao parkVisitDao;
    @Autowired
    private ChannelAlarmDao channelAlarmDao;
    @Autowired
    private ParkRegionDao parkRegionDao;
    @Resource
    private OrderSonInfoDao orderSonInfoDao;
    @Autowired
    private OrderSonInfoServiceImpl orderSonInfoService;
    @Resource
    private OrderSonCarInfoDao orderSonCarInfoDao;
    @Autowired
    private StringRedisTemplate redisTemplate;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;
    @Autowired
    private OrderTrackServiceImpl orderTrackService;
    @Autowired
    private RedisUtils redisUtils;

    @Autowired
    protected OrderNotpayService orderNotpayService;
    @Autowired
    protected OrderCarInfoService orderCarInfoService;

    @Transactional
    @Override
    public ObjectResponse exit(ExitRequest exitRequest, String parkCode) {
        if (exitRequest.getIsExceptionExit() != null && exitRequest.getIsExceptionExit() == 1){
            return exit(exitRequest, parkCode, OrderStatusConstants.EXCEPTION);
        }else{
            return exit(exitRequest, parkCode, OrderStatusConstants.LEAVED_PARK);
        }
    }

    public ObjectResponse exit(ExitRequest exitRequest, String parkCode, Integer status) {
        String channelCode = exitRequest.getChannelId();
        if (channelCode == null) return ObjectResponse.failed(CodeConstants.ERROR_400, "通道不存在");
        ParkInoutdevice parkChannel = parkService.getInoutDeviceByCode(channelCode).getData();
        if (parkChannel == null) return ObjectResponse.failed(CodeConstants.ERROR_402, "channelId未被注册");
        ParkRegion parkRegion = StringUtils.isBlank(exitRequest.getRegionCode()) ?
                (parkChannel.getRegionId() == null ? null :  parkRegionDao.selectById(parkChannel.getRegionId()))
                : parkRegionDao.getRegionByCode(exitRequest.getParkId(), exitRequest.getRegionCode());
        long fatherRelationId = 0;

        OrderInfo existsOrder = orderService.getByLocalOrderNumWithHistory(exitRequest.getParkId(), exitRequest.getOrderId());

        // 按主订单处理
        if (parkRegion == null
                || (fatherRelationId = NumberUtils.toPrimitive(parkRegion.getFatherRelationId())) == 0  // 主区域
                || NumberUtils.toPrimitive(parkChannel.getIsMaster()) == 1  // 子区域主通道
        ) {
            saveMasterOrder(parkChannel, exitRequest, parkCode, parkRegion, existsOrder, status);
        } else {
            // 场中场
            // 子区域且子通道，确定通道方向及连接区域
            ParkRegion parentRegion = parkRegionDao.selectById(fatherRelationId);
            if (parentRegion == null) return ObjectResponse.failed(CodeConstants.ERROR_402, "区域所属主区域不存在");
            ParkRegion originRegion = null, destRegion = null;
            if (parkChannel.getInandoutType() == 1) {    // 入口，代表从主区域到子区域
                originRegion = parentRegion;
                destRegion = parkRegion;
            } else { // 出口，代表从子区域到主区域
                originRegion = parkRegion;
                destRegion = parentRegion;
            }
            existsOrder = saveSonOrder(exitRequest, parkCode, originRegion, destRegion, parkChannel, existsOrder, status);
        }
        if (existsOrder == null) {
            return ObjectResponse.success();
        }
        Map<String, Object> map = new HashMap<>();
        map.put("orderId", existsOrder.getLocalOrderNum());
        map.put("orderNum", existsOrder.getOrderNum());
        return ObjectResponse.success(map);
    }

    /**
     * 保存主订单
     * @param parkChannel
     * @param exitRequest
     * @param parkCode
     * @param parkRegion
     * @param status
     * @return
     */
    private void saveMasterOrder(ParkInoutdevice parkChannel, ExitRequest exitRequest, String parkCode, ParkRegion parkRegion, OrderInfo existsOrder, Integer status) {
        AssertTools.notNull(existsOrder, CodeConstants.SUCCESS, "未找到入场记录");

        int serviceStatus = existsOrder.getServiceStatus();
        if (serviceStatus == OrderStatusConstants.LEAVED_PARK
                && exitRequest.getExitTime().equals(existsOrder.getExitTime())) {
            log.info("[离场服务] 重复离场，重复请求参数：{}", exitRequest);
            return;
        }

        existsOrder.setServiceStatus(status);
        existsOrder.setExitTime(exitRequest.getExitTime());
        existsOrder.setOperAccount(StringUtils.isNotBlank(exitRequest.getOperAccount())
                ? exitRequest.getOperAccount() : exitRequest.getUserAccount());
        existsOrder.setOddStatus(exitRequest.getExceptionReason());
        existsOrder.setType(exitRequest.getType());
        existsOrder.setCarType(exitRequest.getCarType());
        fillAmount(exitRequest, existsOrder);

        OrderCarInfo carInfo = orderCarInfoService.getByOrderNumWithHistory(existsOrder.getOrderNum());
        AssertTools.notNull(carInfo, CodeConstants.ERROR, "平台订单数据不完整");

        OrderCarInfo updateCarInfo = new OrderCarInfo();
        updateCarInfo.setId(carInfo.getId());
        String imgFileName = fillImage(exitRequest.getExitImage(), parkCode);
        updateCarInfo.setExitImage(imgFileName);
        updateCarInfo.setSmallExitImage(exitRequest.getSmallImage());
        updateCarInfo.setExitNo(parkChannel.getInandoutName());
        updateCarInfo.setExitChannelId(exitRequest.getChannelId());
        updateCarInfo.setExitWay(exitRequest.getInoutEvent());
        updateCarInfo.setExitTerminal(exitRequest.getExTerminal());
        updateCarInfo.setExitOperAccount(exitRequest.getOperAccount());
        updateCarInfo.setExitRemark(exitRequest.getRemark());
        if (StringUtils.isNotEmpty(exitRequest.getPlateColor())) {
            updateCarInfo.setPlateColor(exitRequest.getPlateColor());
        }
        List<OrderSonInfo> orderSonInfos = null;
        if (NumberUtils.toPrimitive(existsOrder.getHasSon()) == 1) {
            orderSonInfos = orderSonInfoDao.selectList(Wrappers.lambdaQuery(OrderSonInfo.class)
                    .eq(OrderSonInfo::getParkId, exitRequest.getParkId())
                    .eq(OrderSonInfo::getOrderNum, existsOrder.getOrderNum())
                    .in(OrderSonInfo::getServiceStatus, OrderStatusConstants.IN_PARK, OrderStatusConstants.LEAVED_PARK, OrderStatusConstants.EXCEPTION));
            BigDecimal totalAmount = BigDecimal.ZERO, paidAmount = BigDecimal.ZERO, discountAmount = BigDecimal.ZERO, balancePrice = BigDecimal.ZERO;
            for (OrderSonInfo orderSonInfo : orderSonInfos) {
                if (orderSonInfo.getServiceStatus() == OrderStatusConstants.IN_PARK // 在场子订单更新
                        || orderSonInfo.getRegionId().equals(parkRegion.getId())) {  // 上报区域的区域订单更新
                    handleOrderSon(parkCode, exitRequest, parkRegion, parkChannel, existsOrder,
                            new OrderCarInfoWrapper(updateCarInfo), orderSonInfo, status);
                }
                log.debug("端网云子订单离场处理|{}|{}", existsOrder.getOrderNum(), orderSonInfo);
                totalAmount = NumberUtils.decimalAdd(totalAmount, orderSonInfo.getTotalPrice());
                paidAmount = NumberUtils.decimalAdd(paidAmount, orderSonInfo.getPaidPrice());
                discountAmount = NumberUtils.decimalAdd(discountAmount, orderSonInfo.getDiscountPrice());
                balancePrice = NumberUtils.decimalAdd(balancePrice, orderSonInfo.getBalancePrice());
            }
            existsOrder.setTotalPrice(totalAmount.toString()).setPaidPrice(paidAmount.toString()).setDiscountPrice(discountAmount.toString()).setBalancePrice(balancePrice);
        }

        List<ExitCommonRequest.PaidInfo> paidInfo = exitRequest.getPaidInfo();
        if (paidInfo != null) {
            exitPayDealService.exitPayDeal(exitRequest, existsOrder, orderSonInfos);
        }
        existsOrder.setExitChannelId(exitRequest.getChannelId());
        orderService.updateOrderWithPush(existsOrder);
        log.info("[离场服务] 修改订单主信息完成，orderId：{}", exitRequest.getOrderId());
        orderCarInfoService.updateByIdWithHistory(updateCarInfo);
        if (CollectionUtils.isNotEmpty(orderSonInfos)) {
            orderSonInfoService.updateBatchById(orderSonInfos);
        }
        // 欠费订单处理
        exitPayDealService.handleNotPayOrder(exitRequest, existsOrder, parkChannel, parkCode);

        // 补充出场轨迹
        if (orderSonInfos == null) {
            ObjectResponse<Park> parkResp = parkService.findByParkId(existsOrder.getParkId());
            if (ObjectResponse.isSuccess(parkResp) && NumberUtils.toPrimitive(parkResp.getData().getIsInterior()) == 1
                    && NumberUtils.toPrimitive(parkResp.getData().getNestType()) != 0) {
                String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                        String.valueOf(existsOrder.getParkId()), String.class);
                if (version != null && PncVersionEnum.getIndex(version) < PncVersionEnum.版本7.getIndex()) {
                    OrderTrack orderTrack = new OrderTrack()
                            .setRecordType(2)
                            .setParkId(existsOrder.getParkId())
                            .setRegionId(existsOrder.getRegionId())
                            .setChannelName(updateCarInfo.getExitNo())
                            .setOrderNum(existsOrder.getOrderNum())
                            .setEnexTime(exitRequest.getExitTime() == null ? 0 : exitRequest.getExitTime())
                            .setPlateNum(exitRequest.getPlateNum())
                            .setType(existsOrder.getType())
                            .setImage(updateCarInfo.getExitImage())
                            .setInoutEvent(0);
                    orderTrackService.save(orderTrack);
                }
            }
        }

        asyncHandler(exitRequest, parkCode, exitRequest.getParkId(), exitRequest.getOrderId(), existsOrder, carInfo);
    }

    /**
     * 子订单操作
     * @return
     */
    private OrderInfo saveSonOrder(ExitRequest exitRequest, String parkCode, ParkRegion originRegion, ParkRegion destRegion, ParkInoutdevice parkChannel, OrderInfo mainOrder, int status) {
        List<OrderSonInfo> orderSonInfos = orderSonInfoDao.selectList(Wrappers.lambdaQuery(OrderSonInfo.class)
                .eq(OrderSonInfo::getParkId, exitRequest.getParkId())
                .eq(OrderSonInfo::getLocalOrderNum, exitRequest.getOrderId()));
        Map<Long, OrderSonInfo> orderSonInfoMap = orderSonInfos.stream().collect(Collectors.toMap(OrderSonInfo::getRegionId, Function.identity(), (older, newer) -> newer, HashMap::new));
        OrderSonInfo originOrderSon = orderSonInfoMap.get(originRegion.getId());
        OrderSonInfo destOrderSon = orderSonInfoMap.get(destRegion.getId());

        OrderCarInfo mainCarInfo = null;
        if (mainOrder == null) { // 缺失主订单
            mainOrder = new OrderInfo();
            BeanUtils.copyProperties(exitRequest, mainOrder);
            mainOrder.setOrderNum(CodeTools.GenerateOrderNum())
                    .setLocalOrderNum(exitRequest.getOrderId())
                    .setEnterTime(exitRequest.getExitTime())
                    .setExitTime(null)
                    .setServiceStatus(1)
                    .setHasSon(1)
                    .setNoneEnterFlag(1);
            String regionName = null;
            if (NumberUtils.toPrimitive(originRegion.getFatherRelationId()) == 0) {  // 源区域为主区域
                mainOrder.setRegionId(originRegion.getId());
                regionName = originRegion.getRegionName();
            } else if (NumberUtils.toPrimitive(destRegion.getFatherRelationId()) == 0) { // 目的区域为主区域
                mainOrder.setRegionId(destRegion.getId());
                regionName = destRegion.getRegionName();
            } else {
                throw new ResponseBodyException(CodeConstants.ERROR_402, "两个子区域无法切换");
            }
            orderInfoDao.insert(mainOrder);
            log.info("[车辆入场服务] 补充主订单信息完成，orderNum：{}，区域名称：{}", mainOrder.getOrderNum(), regionName);
            mainCarInfo = buildCarInfo(parkCode, exitRequest, mainOrder, parkChannel);
            orderCarInfoDao.insert(mainCarInfo);
            log.info("[车辆入场服务] 补充主订单车辆信息完成，orderNum：{}，区域名称：{}", mainOrder.getOrderNum(), regionName);
        } else if (NumberUtils.toPrimitive(mainOrder.getHasSon()) == 0) {
            mainOrder.setHasSon(1);
            OrderInfo updateMain = new OrderInfo().setId(mainOrder.getId()).setHasSon(1);
            orderInfoDao.updateById(updateMain);
        }
        OrderCarInfoWrapper mainCarInfoWrapper = new OrderCarInfoWrapper(mainCarInfo);
        // 处理源区域订单, 置为离场
        originOrderSon = handleOrderSon(parkCode, exitRequest, originRegion, parkChannel,
                mainOrder, mainCarInfoWrapper, originOrderSon, status);
        // 处理目的区域订单, 置为入场
        destOrderSon = handleOrderSon(parkCode, exitRequest, destRegion, parkChannel,
                mainOrder, mainCarInfoWrapper, destOrderSon, OrderStatusConstants.IN_PARK);
        List<ExitCommonRequest.PaidInfo> paidInfo = exitRequest.getPaidInfo();
        if (paidInfo != null) {
            exitPayDealService.exitPayDeal(exitRequest, originOrderSon, null);
        }
        // 欠费订单处理
        exitPayDealService.handleNotPayOrder(exitRequest, originOrderSon, parkChannel, parkCode);
        //更新访客预约订单状态
        ParkVisit parkVisit = parkVisitDao.selectVisitByParkidPlate(exitRequest.getParkId(), mainOrder.getPlateNum(), new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
        if (parkVisit != null) {
            parkVisit.setVisitStatus(3);
            parkVisit.setOrderNum(mainOrder.getOrderNum());
            parkVisit.setExitTime(Math.toIntExact(exitRequest.getExitTime()));
            parkVisitDao.updateVisit(parkVisit);
            log.info("[离场服务-子区域] 更新访客预约表完成，orderId：{}", exitRequest.getOrderId());
        }
        //调用支付中心离场通知(只有临时车和开启无感支付才需要调用接口)
        OrderSonInfo notifyOrder = originOrderSon;
        asyncExecutor.execute(ThreadUtils.wrapTrace(() -> autopayService.exitNotify(notifyOrder, parkCode)));
        //下发消息
//            sendMQMessage(exitRequest,parkCode,orderEnexRecord_result);
        return originOrderSon;
    }

    private OrderSonInfo handleOrderSon(String parkCode, ExitRequest request, ParkRegion parkRegion, ParkInoutdevice parkChannel,
                                        OrderInfo mainOrder, OrderCarInfoWrapper mainCarInfoWrapper, OrderSonInfo orderSonInfo, int targetStatus) {
        Long exitTime = targetStatus != OrderStatusConstants.IN_PARK
                && (mainOrder.getServiceStatus() != OrderStatusConstants.IN_PARK
                    || !mainOrder.getRegionId().equals(parkRegion.getId())) ? request.getExitTime() : null;
        if (orderSonInfo == null) {    // 补充区域订单
            orderSonInfo = new OrderSonInfo();
            OrderSonCarInfo orderSonCarInfo = null;
            if (mainOrder.getRegionId().equals(parkChannel.getRegionId())) { // 如果区域与主订单区域一致，则拷贝主订单到子订单
                BeanUtils.copyProperties(mainOrder, orderSonInfo);
                orderSonCarInfo = new OrderSonCarInfo();
                if (mainCarInfoWrapper.orderCarInfo == null) {
                    mainCarInfoWrapper.orderCarInfo = orderCarInfoDao.selectByOrderNum(mainOrder.getOrderNum());
                }
                BeanUtils.copyProperties(mainCarInfoWrapper.orderCarInfo, orderSonCarInfo);
            } else {
                BeanUtils.copyProperties(request, orderSonInfo);
                orderSonInfo.setRegionId(parkChannel.getRegionId())
                        .setOrderNum(mainOrder.getOrderNum())
                        .setLocalOrderNum(request.getOrderId())
                        .setEnterTime(request.getExitTime())
                        .setExitTime(exitTime)
                        .setNoneEnterFlag(targetStatus == OrderStatusConstants.IN_PARK ? 0 : 1);
            }
            orderSonInfo.setServiceStatus(targetStatus);
            if (orderSonInfo.getRegionId().equals(parkRegion.getId())) {
                fillAmount(request, orderSonInfo);
            }
            orderSonInfoDao.insertWithPlateNum2(orderSonInfo);
            if (orderSonCarInfo == null) {
                orderSonCarInfo = (OrderSonCarInfo) buildCarInfo(parkCode, request, orderSonInfo, parkChannel);
            }
            orderSonCarInfo.setOrderSonId(orderSonInfo.getId());
            if (StringUtils.isNotEmpty(request.getPlateColor())) {
                orderSonCarInfo.setPlateColor(request.getPlateColor());
            }
            orderSonCarInfoDao.insert(orderSonCarInfo);
        } else {    // 区域订单已存在则更改订单状态
            if (targetStatus == OrderStatusConstants.IN_PARK) {
                orderSonInfoDao.setInPark(orderSonInfo.getId());
            } else {
                OrderSonInfo updateOrderSon = new OrderSonInfo();
                //补离场时间
                if (!orderSonInfo.getRegionId().equals(mainOrder.getRegionId())
                        && !orderSonInfo.getRegionId().equals(parkChannel.getRegionId())
                        && orderSonInfo.getExitTime() == null) {
                    if (PlateTypeEnum.月卡车.getType().equals(orderSonInfo.getType())
                            || PlateTypeEnum.VIP车辆.getType().equals(orderSonInfo.getType())) {
                        exitTime = request.getExitTime();
                    } else {
                        exitTime = orderSonInfo.getEnterTime();
                    }
                    log.debug("====> {}, {}, {}", orderSonInfo, mainOrder, parkChannel);
                    updateOrderSon.setNoneExitFlag(1);
                    orderSonInfo.setNoneExitFlag(1);
                }
                orderSonInfo.setServiceStatus(targetStatus)
                        .setExitTime(exitTime);
                updateOrderSon.setId(orderSonInfo.getId())
                        .setServiceStatus(targetStatus)
                        .setExitTime(exitTime);
                if (orderSonInfo.getRegionId().equals(parkRegion.getId())) {
                    fillAmount(request, orderSonInfo);
                    fillAmount(request, updateOrderSon);
                }
                orderSonInfoDao.updateById(updateOrderSon);
                if (parkChannel.getRegionId().equals(parkRegion.getId())) {
                    OrderSonCarInfo updateOrderSonCarInfo = new OrderSonCarInfo();
                    updateOrderSonCarInfo.setOrderSonId(orderSonInfo.getId())
                            .setExitNo(parkChannel.getInandoutName())
                            .setExitChannelId(parkChannel.getInandoutCode())
                            .setExitImage(fillImage(request.getExitImage(), parkCode))
                            .setSmallExitImage(request.getSmallImage())
                            .setExitWay(request.getInoutEvent())
                            .setExitTerminal(request.getExTerminal())
                            .setExitRemark(request.getRemark())
                            .setExitOperAccount(request.getOperAccount());
//                        .setExitReliability(request.getReliability());
                    if (StringUtils.isNotEmpty(request.getPlateColor())) {
                        updateOrderSonCarInfo.setPlateColor(request.getPlateColor());
                    }
                    orderSonCarInfoDao.updateByOrderSonId(updateOrderSonCarInfo);
                }
            }
        }

        if (targetStatus != OrderStatusConstants.IN_PARK && orderSonInfo.getRegionId().equals(parkChannel.getRegionId())) {
            ObjectResponse<Park> parkResp = parkService.findByParkId(request.getParkId());
            if(ObjectResponse.isSuccess(parkResp) && NumberUtils.toPrimitive(parkResp.getData().getIsInterior()) == 1
                    && NumberUtils.toPrimitive(parkResp.getData().getNestType()) != 0) {
                // 补充离场记录
                String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                        String.valueOf(parkChannel.getParkId()), String.class);
                if (version != null && PncVersionEnum.getIndex(version) < PncVersionEnum.版本7.getIndex()) {
                    OrderTrack orderTrack = new OrderTrack()
                            .setRecordType(2)
                            .setParkId(parkChannel.getParkId())
                            .setOrderNum(orderSonInfo.getOrderNum())
                            .setRegionId(parkChannel.getRegionId())
                            .setChannelName(parkChannel.getInandoutName())
                            .setEnexTime(request.getExitTime() == null ? 0 : request.getExitTime())
                            .setPlateNum(request.getPlateNum())
                            .setImage(fillImage(request.getExitImage(), parkCode))
                            .setType(orderSonInfo.getType())
                            .setInoutEvent(0);
//                .setOperAccount("");
                    orderTrackService.save(orderTrack);
                }
            }
        }
        return orderSonInfo;
    }

    private OrderCarInfo buildCarInfo(String parkCode, ExitRequest request, OrderInfo orderInfo, ParkInoutdevice parkChannel) {
        OrderCarInfo orderCarInfo = null;
        if (orderInfo instanceof OrderSonInfo) {
            orderCarInfo = new OrderSonCarInfo();
        } else {
            orderCarInfo = new OrderCarInfo();
        }
        BeanUtils.copyProperties(request, orderCarInfo);
        orderCarInfo.setOrderNum(orderInfo.getOrderNum());

        String imgFileName = fillImage(request.getExitImage(), parkCode);
//        orderCarInfo.setEnterReliability(request.getReliability());
        if (orderInfo.getServiceStatus() != OrderStatusConstants.IN_PARK) {
            orderCarInfo.setExitNo(parkChannel.getInandoutName());
            orderCarInfo.setExitChannelId(parkChannel.getInandoutCode());
            orderCarInfo.setExitImage(imgFileName);
            orderCarInfo.setSmallExitImage(request.getSmallImage());
//            orderCarInfo.setExitReliability(request.getReliability());
        }
        orderCarInfo.setExitWay(request.getInoutEvent());
        orderCarInfo.setExitOperAccount(request.getOperAccount());
        orderCarInfo.setExitTerminal(request.getExTerminal());
        orderCarInfo.setExitRemark(request.getRemark());
        if (NumberUtils.toPrimitive(orderInfo.getNoneEnterFlag()) == 0) {
            orderCarInfo.setEnterNo(parkChannel.getInandoutName());
            orderCarInfo.setEnterChannelId(parkChannel.getInandoutCode());
            orderCarInfo.setEnterImage(imgFileName);
            orderCarInfo.setSmallEnterImage(request.getSmallImage());
        }
        return orderCarInfo;
    }

    private String fillImage(String image, String parkCode) {
        if (StringUtils.isNotBlank(image) && !image.contains("/image/")) {
            String date = DateTools.getFormat(DateTools.DF_, new Date());
            String[] ymd = date.split("-");
            image = parkCode + "/image/" + ymd[0] + ymd[1] + "/" + ymd[2] + "/" + image;
        }
        return image;
    }

    private void fillAmount(ExitRequest request, OrderInfo orderInfo) {
        orderInfo.setTotalPrice(NumberUtils.parseDecimal(request.getTotalAmount()).toString());
        orderInfo.setPaidPrice(NumberUtils.parseDecimal(request.getPaidAmount()).toString());
        orderInfo.setDiscountPrice(NumberUtils.parseDecimal(request.getDiscountAmount()).toString());
        orderInfo.setBalancePrice(NumberUtils.parseDecimal(request.getBalancePrice()));
    }

    @AllArgsConstructor
    private static class OrderCarInfoWrapper {
        OrderCarInfo orderCarInfo;
    }

    private static final ExecutorService FIXED_THREAD_POOL = Executors.newFixedThreadPool(1);
    private void asyncHandler(ExitRequest exitRequest, String parkCode, Long parkId, String orderId, OrderInfo orderInfo, OrderCarInfo carInfo) {
        FIXED_THREAD_POOL.execute(() -> {
            //更新访客预约订单状态
            ParkVisit parkVisit = parkVisitDao.selectVisitByParkidPlate(parkId, orderInfo.getPlateNum(), new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
            if (parkVisit != null) {
                parkVisit.setVisitStatus(3);
                parkVisit.setOrderNum(orderInfo.getOrderNum());
                parkVisit.setExitTime(Math.toIntExact(exitRequest.getExitTime()));
                parkVisitDao.updateVisit(parkVisit);
                log.info("[离场服务] 更新访客预约表完成，orderId：{}", orderId);
            }
            //调用支付中心离场通知(只有临时车和开启无感支付才需要调用接口)
            orderInfo.setExitTime(exitRequest.getExitTime());
            autopayService.exitNotify(orderInfo, parkCode);
            //下发消息
//            sendMQMessage(exitRequest,parkCode,orderEnexRecord_result);
            sendWebsocketMessage(exitRequest,parkCode,carInfo,orderInfo);

            // 更新通道的报警记录为已处理
            if (exitRequest.getChannelId() != null){
                ChannelAlarm channelAlarm = new ChannelAlarm();
                channelAlarm.setParkId(parkId);
                channelAlarm.setChannelCode(exitRequest.getChannelId());
                channelAlarm.setStatus(ChannelAlarm.Status.已处理.getStatus());
                channelAlarmDao.update(channelAlarm);
            }
        });
    }

    /**
     * 出入场通知 redis 订阅发布
     * @param exitRequest
     * @param parkCode
     * @param carInfo
     * @param orderInfo
     */
    private void sendWebsocketMessage(ExitRequest exitRequest, String parkCode, OrderCarInfo carInfo, OrderInfo orderInfo){
        WebsocketPushData websocketPushData = new WebsocketPushData();
        websocketPushData.setRecordType(2);
        if (orderInfo == null || orderInfo.getEnterTime() == null){
            return;
        }
        websocketPushData.setEnterTime(orderInfo.getEnterTime().intValue());
        websocketPushData.setExitTime(exitRequest.getExitTime().intValue());
        websocketPushData.setType(exitRequest.getType());
        websocketPushData.setCarType(exitRequest.getCarType());
        websocketPushData.setChannelId(exitRequest.getChannelId());
        websocketPushData.setEnterNo(carInfo.getEnterNo());
        websocketPushData.setExitNo(carInfo.getExitNo());
        websocketPushData.setParkCode(parkCode);
//        websocketPushData.setInoutEvent(orderEnexRecord.getInoutEvent());
//        websocketPushData.setInoutOpening(orderEnexRecord.getInoutOpening());
        websocketPushData.setPlateNum(exitRequest.getPlateNum());
        websocketPushData.setPlateColor(carInfo.getPlateColor());
        websocketPushData.setTotalPrice(exitRequest.getTotalAmount());
        websocketPushData.setPaidPrice(exitRequest.getPaidAmount());
        websocketPushData.setDiscountPrice(exitRequest.getDiscountAmount());
        websocketPushData.setPayStatus(String.valueOf(orderInfo.getServiceStatus()));
        websocketPushData.setOrderNum(orderInfo.getOrderNum());
        log.info("sendWebsocketMessage  >> {}",parkCode);
        redisTemplate.convertAndSend(RedisMsgListener.TOPIC, JsonUtils.toString(websocketPushData));
    }
}
