package com.icetech.park.service.order.impl.enter;

import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.dao.OrderPayDao;
import com.icetech.order.dao.OrderSonCarInfoDao;
import com.icetech.order.dao.OrderSonInfoDao;
import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.cloudcenter.domain.charge.dto.OrderSumFeeDto;
import com.icetech.cloudcenter.domain.constants.SpecialConstants;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.order.domain.entity.OrderSonCarInfo;
import com.icetech.order.domain.entity.OrderSonInfo;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.domain.enumeration.OrderOddStatusEnum;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.response.p2c.CarEnterResult;
import com.icetech.park.domain.vo.RegionFreeSpaceUpdateVo;
import com.icetech.park.service.order.impl.OrderServiceImpl;
import com.icetech.order.service.impl.OrderSonInfoServiceImpl;
import com.icetech.park.service.report.ReportParamHolder;
import com.icetech.third.service.third.MqPushService;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 子通道入场
 * @author fangct
 */
@Slf4j
@Service
public class SubChannelCarOrderEnterServiceImpl extends CommonEnterImpl {
    @Autowired
    private OrderInfoDao orderInfoDao;
    @Resource
    private OrderCarInfoDao orderCarInfoDao;
    @Autowired
    private OrderSonInfoDao orderSonInfoDao;
    @Autowired
    private OrderSonInfoServiceImpl orderSonInfoService;
    @Resource
    private OrderSonCarInfoDao orderSonCarInfoDao;
    @Autowired
    private OrderServiceImpl orderService;
    @Autowired
    private ParkRegionDao regionDao;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;
    @Autowired
    private MqPushService pushService;
    @Autowired
    private OrderPayDao orderPayDao;
    @Autowired
    private MqPushService mqPushService;

    @Transactional
    public ObjectResponse<CarEnterResult> enter(CarEnterRequest enterRequest, ReportParamHolder paramHolder) {
        Long parkId = enterRequest.getParkId();

        //验证车辆类型
        validateCarInfo(enterRequest, enterRequest.getPlateNum(), enterRequest.getParkId());

        /*//判断是否重复入场，并将旧的订单记录异常离场
        try {
            influenceLines = checkSamePlate(enterRequest.getPlateNum(), enterRequest.getParkId(),
                    enterRequest.getEnterTime(), enterRequest.getEnterWay());
        } catch (ResponseBodyException e) {
            if (e.getErrCode().equals(CodeConstants.ERROR_405)) {
                log.warn("处理失败: {}:{}. enterRequest[{}]", e.getErrCode(), e.getMessage(),
                        enterRequest, e);
                //返回
                CarEnterResult result = new CarEnterResult();
                result.setOrderNum(enterRequest.getOrderNum());
                return ObjectResponse.success(result);
            }
        }*/
        boolean isUpdateFreeSpace = orderHandle(enterRequest, paramHolder);
        //记录轨迹表
        saveTrack(enterRequest, paramHolder.getParkChannel().getRegionId());

        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
            @Override
            public void afterCommit() {
                asyncExecutor.execute(ThreadUtils.wrapTrace(() -> asyncHandle(enterRequest)));
            }
        });
        //返回
        CarEnterResult result = new CarEnterResult();
        result.setOrderNum(enterRequest.getOrderNum());
        return ObjectResponse.success(result);
    }

    /**
     * 处理正常订单
     *
     * @param enterRequest 入场参数
     */
    private boolean orderHandle(CarEnterRequest enterRequest, ReportParamHolder paramHolder) {
        Long parkId = enterRequest.getParkId();
        ParkInoutdevice parkChannel = paramHolder.getParkChannel();
        Long regionId = parkChannel.getRegionId();
        String channelCode = enterRequest.getInandoutCode();

        List<RegionFreeSpaceUpdateVo> freeSpaceUpdateVos = new ArrayList<>();
        String orderNum;
        //查询主订单
        ObjectResponse<OrderInfo> infoObjectResponse = orderService.findInParkId(enterRequest.getPlateNum(), enterRequest.getParkId());
        OrderInfo orderInfo = infoObjectResponse.getData();
        boolean hasMainOrderEnter = true;
        if (orderInfo != null) {
            orderNum = orderInfo.getOrderNum();
            orderInfo.setHasSon(1);
            enterRequest.setOrderNum(orderInfo.getOrderNum());
            orderInfoDao.updateByOrderNum(orderInfo);
        } else {
            ObjectResponse<OrderInfo> orderInfoObjectResponse =
                    orderService.fuzzyPlate(enterRequest.getParkId(), channelCode, enterRequest.getPlateNum());
            if (ObjectResponse.isSuccess(orderInfoObjectResponse)) {
                orderInfo = orderInfoObjectResponse.getData();
                orderNum = orderInfo.getOrderNum();
                enterRequest.setPlateNum(orderInfo.getPlateNum());
                enterRequest.setOrderNum(orderInfo.getPlateNum());
                orderInfo.setHasSon(1);
                orderInfoDao.updateByOrderNum(orderInfo);
                log.info("[端云-子区域通道车辆入场服务] 模糊匹配到订单：{}", orderInfo);
            } else {
                hasMainOrderEnter = false;
                orderNum = enterRequest.getOrderNum();
                //不存在主订单信息 补录主订单信息
                orderInfo = new OrderInfo();
                BeanUtils.copyProperties(enterRequest, orderInfo);
                orderInfo.setOrderNum(orderNum);
                orderInfo.setHasSon(1);
                //orderInfo.setNoneEnterFlag(1);
                ParkRegion parkRegion = regionDao.selectOutByParkid(enterRequest.getParkId());
                if (parkRegion != null) {
                    orderInfo.setRegionId(parkRegion.getId());
                } else {
                    log.warn("[端云-补主订单信息] 无外区域数据，参数：{}", enterRequest);
                    throw new ResponseBodyException(CodeConstants.ERROR_400, "车场缺少外区域");
                }
                //纠正在大场的车辆类型
                CarEnterRequest bigParkEnterRequest = new CarEnterRequest();
                bigParkEnterRequest.setType(orderInfo.getType());
                validateType(bigParkEnterRequest, orderInfo.getPlateNum(), orderInfo.getParkId(), parkRegion.getId());
                orderInfo.setType(bigParkEnterRequest.getType());
                orderInfo.setCarDesc(bigParkEnterRequest.getCarDesc());
                orderInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
                if (NumberUtils.toPrimitive(orderInfo.getNoneEnterFlag()) == 0) {
                    orderInfo.setEnterChannelId(parkChannel.getInandoutCode());
                }
                try {
                    orderService.addOrderInfo(orderInfo);
                } catch (DuplicateKeyException e) {
                    orderNum = CodeTools.GenerateOrderNum();
                    orderInfo.setOrderNum(orderNum);
                    enterRequest.setOrderNum(orderNum);
                    orderService.addOrderInfo(orderInfo);
                }
                // 插入主订单拓展信息
                OrderCarInfo carInfo = new OrderCarInfo();
                BeanUtils.copyProperties(enterRequest, carInfo);
                carInfo.setEnterOperAccount(enterRequest.getOperaUser());
                carInfo.setOrderNum(orderNum);
                orderCarInfoDao.insert(carInfo);
            }
        }
        if (enterRequest.isReplenishOrder()) {
            paramHolder.setReplenishOrder(orderInfo);
        }
        enterRequest.setOrderNum(orderNum);
        if (hasMainOrderEnter) {
            freeSpaceUpdateVos.add(RegionFreeSpaceUpdateVo.builder()
                    .regionId(orderInfo.getRegionId()).type(orderInfo.getType()).num(1).build());
        } else {
            ParkConfig parkConfig = paramHolder.getParkConfig();
            if (!orderInfo.getType().equals(PlateTypeEnum.月卡车.getType())
                    || NumberUtils.toPrimitive(parkConfig.getIsCardcount()) == 1) {
                // 更新空车位
                if (parkConfig.getCalcSpaceMethod(1) == 1) {
                    updateRegionFreeSpaceService.addFreeSpace(parkId, regionId,-1);
                } else {
                    updateRegionFreeSpaceService.syncFreeSpace(parkId, regionId);
                }
                mqPushService.pushParkFreeSpace(parkId);
            }
        }

        //当前区域的子订单
        OrderSonInfo innerOrderSonInfo = null;
        //外区域对应的子订单
        OrderSonInfo outOrderSonInfo = null;
        OrderSonInfo orderSonInfoParam = new OrderSonInfo();
        orderSonInfoParam.setOrderNum(orderNum);
        orderSonInfoParam.setParkId(parkId);
        List<OrderSonInfo> orderSonInfos = orderSonInfoDao.selectList(orderSonInfoParam);
        if (CollectionUtils.isNotEmpty(orderSonInfos)) {
            Optional<OrderSonInfo> first1 = orderSonInfos.stream()
                    .filter(osi -> !regionId.equals(osi.getRegionId())).findFirst();
            if (first1.isPresent()) {
                outOrderSonInfo = first1.get();
            }
            Optional<OrderSonInfo> first2 = orderSonInfos.stream()
                    .filter(osi -> regionId.equals(osi.getRegionId())).findFirst();
            if (first2.isPresent()) {
                innerOrderSonInfo = first2.get();
            }
        }

        ParkRegion parkRegion = regionDao.selectById(regionId);
        if (innerOrderSonInfo != null) {
            if (innerOrderSonInfo.getServiceStatus() == 1) {
                //小出未识别，重复入场不处理空车位
                freeSpaceUpdateVos.clear();
            }
            innerOrderSonInfo.setExitTime(null);
            innerOrderSonInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
            orderSonInfoDao.updateById(innerOrderSonInfo);
            log.info("[子通道入场] 重复驶入内场[{}]", innerOrderSonInfo);
        } else {
            innerOrderSonInfo = new OrderSonInfo();
            // 插入子订单信息
            if (regionId.equals(orderInfo.getRegionId())) {
                BeanUtils.copyProperties(orderInfo, innerOrderSonInfo);
                setOrderSonPrice(orderInfo.getParkId(), orderNum, innerOrderSonInfo);
                innerOrderSonInfo.setRegionId(regionId);
                innerOrderSonInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
                if (!hasMainOrderEnter) {
                    innerOrderSonInfo.setNoneEnterFlag(1);
                }
                orderSonInfoDao.insertWithPlateNum2(innerOrderSonInfo);
                log.info("[入场服务-子通道] 插入子订单表的内区域记录[{}]", innerOrderSonInfo);

                OrderCarInfo carInfo = orderCarInfoDao.selectByOrderNum(orderNum);

                OrderSonCarInfo sonCarInfoMain = new OrderSonCarInfo();
                BeanUtils.copyProperties(carInfo, sonCarInfoMain);
                sonCarInfoMain.setEnterOperAccount(enterRequest.getOperaUser());
                sonCarInfoMain.setOrderSonId(innerOrderSonInfo.getId());
                if (!hasMainOrderEnter) {
                    sonCarInfoMain.setEnterWay(null);
                    sonCarInfoMain.setEnterChannelId(null);
                    sonCarInfoMain.setEnterNo(null);
                    sonCarInfoMain.setEnterImage(null);
                }
                orderSonCarInfoDao.insert(sonCarInfoMain);
                log.info("[入场服务-子通道] 插入子订单详情表完成，sonCarInfoMain：{}", sonCarInfoMain);
            } else {
                BeanUtils.copyProperties(enterRequest, innerOrderSonInfo);
                innerOrderSonInfo.setOrderNum(orderNum);
                innerOrderSonInfo.setRegionId(regionId);
                if (enterRequest.getNoneEnterFlag()) {
                    innerOrderSonInfo.setNoneEnterFlag(1);
                }
                orderSonInfoDao.insertWithPlateNum2(innerOrderSonInfo);

                //插入子入场信息
                OrderSonCarInfo sonCarInfo = new OrderSonCarInfo();
                BeanUtils.copyProperties(enterRequest, sonCarInfo);
                sonCarInfo.setOrderSonId(innerOrderSonInfo.getId());
                sonCarInfo.setOrderNum(orderNum);

                if (!enterRequest.getNoneEnterFlag()) {
                    sonCarInfo.setEnterChannelId(enterRequest.getInandoutCode());
                    sonCarInfo.setEnterNo(enterRequest.getInandoutName());
                    sonCarInfo.setEnterImage(enterRequest.getMaxImage());
                    sonCarInfo.setSmallEnterImage(enterRequest.getSmallImage());
                    sonCarInfo.setEnterReliability(enterRequest.getReliability());
                    sonCarInfo.setEnterOperAccount(enterRequest.getOperaUser());
                }
                //处理标签
                delTags(enterRequest, orderNum, regionId, sonCarInfo);
                orderSonCarInfoDao.insert(sonCarInfo);

                //发送MQ消息，同步给其他相机
                pushService.pushOrderSonEnter(innerOrderSonInfo);
            }
            freeSpaceUpdateVos.add(RegionFreeSpaceUpdateVo.builder()
                    .regionId(regionId).type(innerOrderSonInfo.getType()).num(-1).build());
        }

        if (outOrderSonInfo != null) {
            //设置金额
            //setOrderSonPrice(parkId, orderNum, mainOrderSonInfo);
            outOrderSonInfo.setServiceStatus(OrderStatusConstants.LEAVED_PARK);
            outOrderSonInfo.setExitTime(enterRequest.getEnterTime());
            outOrderSonInfo.setOperAccount(enterRequest.getOperaUser());
            orderSonInfoDao.updateById(outOrderSonInfo);
            log.info("更新子订单表，子订单信息：{}", outOrderSonInfo);

            dealAbMonth(parkId, outOrderSonInfo, enterRequest.getEnterTime());
        } else {
            outOrderSonInfo = saveOutOrderSonInfo(orderNum, orderInfo, parkRegion.getFatherRelationId(),
                    hasMainOrderEnter, enterRequest.getEnterTime(), enterRequest);
        }

        // 空车位处理
        updateRegionFreeSpace(paramHolder, freeSpaceUpdateVos);

        return true;
    }

    private void dealAbMonth(Long parkId, OrderSonInfo orderSonInfo, Long enterTime) {
        //月卡多位多车处理
        if (orderSonInfo.getType().equals(PlateTypeEnum.月卡车.getType())) {
            asyncExecutor.execute(ThreadUtils.wrapTrace(() ->
                    monthCarService.innerAreaExitAbMonthDeal(
                            parkId, orderSonInfo.getPlateNum(),
                            enterTime, orderSonInfo.getRegionId())
            ));
        }
    }

    private OrderSonInfo saveOutOrderSonInfo(String orderNum, OrderInfo orderInfo, Long regionId,
                                             boolean hasMainOrderEnter, Long enterTime, CarEnterRequest enterRequest) {
        OrderSonInfo orderSonInfoMain = new OrderSonInfo();
        if (orderInfo.getRegionId() != null && orderInfo.getRegionId().equals(regionId)) {
            BeanUtils.copyProperties(orderInfo, orderSonInfoMain);
            setOrderSonPrice(orderInfo.getParkId(), orderNum, orderSonInfoMain);
            orderSonInfoMain.setRegionId(regionId);
            orderSonInfoMain.setServiceStatus(OrderStatusConstants.LEAVED_PARK);
            orderSonInfoMain.setOperAccount(enterRequest.getOperaUser());
            if (!hasMainOrderEnter) {
                orderSonInfoMain.setNoneEnterFlag(1);
            }
            orderSonInfoDao.insertWithPlateNum2(orderSonInfoMain);
            log.info("[入场服务-子通道] 插入子订单表的主订单信息[{}]", orderSonInfoMain);

            //如果车辆进过内区，主离场记录---->子出入场记录表
            OrderCarInfo carInfo = orderCarInfoDao.selectByOrderNum(orderNum);

            OrderSonCarInfo sonCarInfoMain = new OrderSonCarInfo();
            BeanUtils.copyProperties(carInfo, sonCarInfoMain);
            sonCarInfoMain.setEnterOperAccount(enterRequest.getOperaUser());
            sonCarInfoMain.setOrderSonId(orderSonInfoMain.getId());
            if (!hasMainOrderEnter) {
                sonCarInfoMain.setEnterWay(null);
                sonCarInfoMain.setEnterNo(null);
                sonCarInfoMain.setEnterChannelId(null);
                sonCarInfoMain.setEnterImage(null);
            }
            orderSonCarInfoDao.insert(sonCarInfoMain);
            log.info("[入场服务-子通道] 插入子订单信息表外区域信息完成，sonCarInfoMain：{}", sonCarInfoMain);

            dealAbMonth(orderSonInfoMain.getParkId(), orderSonInfoMain, enterTime);
        } else {
            BeanUtils.copyProperties(enterRequest, orderSonInfoMain);
            orderSonInfoMain.setOrderNum(orderNum);
            orderSonInfoMain.setRegionId(regionId);
            orderSonInfoMain.setExitTime(enterRequest.getEnterTime());
            orderSonInfoMain.setServiceStatus(OrderStatusConstants.LEAVED_PARK);
            orderSonInfoMain.setNoneEnterFlag(1);
            //纠正在大场的车辆类型
            CarEnterRequest bigParkEnterRequest = new CarEnterRequest();
            bigParkEnterRequest.setType(orderInfo.getType());
            validateType(bigParkEnterRequest, orderSonInfoMain.getPlateNum(), orderSonInfoMain.getParkId(), regionId);
            orderSonInfoMain.setType(bigParkEnterRequest.getType());
            orderSonInfoMain.setOperAccount(enterRequest.getOperaUser());
            orderSonInfoDao.insertWithPlateNum2(orderSonInfoMain);
            log.info("[入场服务-子通道] 插入子订单表的外区子订单[{}]", orderSonInfoMain);

            //插入子入场信息
            OrderSonCarInfo sonCarInfo = new OrderSonCarInfo();
            BeanUtils.copyProperties(enterRequest, sonCarInfo);
            sonCarInfo.setOrderSonId(orderSonInfoMain.getId());
            sonCarInfo.setOrderNum(orderNum);
            sonCarInfo.setExitChannelId(enterRequest.getInandoutCode());
            sonCarInfo.setExitNo(enterRequest.getInandoutName());
            sonCarInfo.setExitImage(enterRequest.getMaxImage());
            sonCarInfo.setSmallExitImage(enterRequest.getSmallImage());
            sonCarInfo.setExitReliability(enterRequest.getReliability());
            orderSonCarInfoDao.insert(sonCarInfo);
            log.info("[入场服务-子通道] 插入子订单信息表外区域信息完成，sonCarInfo：{}", sonCarInfo);

            dealAbMonth(orderSonInfoMain.getParkId(), orderSonInfoMain, enterTime);
        }
        return orderSonInfoMain;
    }

    /**
     * 判断是否有场内重复车牌，并修改状态为异常离场
     * @param plateNum 车牌号
     * @param parkId 车场ID
     * @param currEnterTime 入场时间
     * @param enterWay 入场方式
     * @param property
     * @param replenishOrder 实录入场标识
     * @param type
     * @return 受影响的行数
     */
    @Override
    protected int checkSamePlate(String plateNum, Long parkId, Long currEnterTime, Integer enterWay, Integer property, boolean replenishOrder, Integer type) {
        //未识别车牌不做重复处理
        if (plateNum.equals(SpecialConstants.NO_PLATE_NUM)) {
            return 0;
        }
        String part = plateNum.substring(1);
        OrderSonInfo orderInfoRes = orderSonInfoDao.fuzzyInParkPlate(parkId, part);
        if (orderInfoRes != null) {
            //上次的入场时间大于当次入场时间（网络等其他原因导致延迟接收），并且上次入场时间小于当前时间+偏移量
            //如果上次入场时间大于等于当前时间+偏移量，则可以认为上次的入场时间有问题，取消上次问题订单，以新订单为准
            if (orderInfoRes.getEnterTime() > currEnterTime
                    && orderInfoRes.getEnterTime() < DateTools.unixTimestamp() + OFFSET) {
                log.warn("[车辆入场服务] 后上报的入场时间比前次小[{}]秒，车牌号：{}", orderInfoRes.getEnterTime() - currEnterTime, plateNum);
                throw new ResponseBodyException(CodeConstants.ERROR_405, "后上报的入场时间比前次小");
            }
            log.info("[车辆入场服务] 重复入场，当前车牌号：{}，在场车牌号{}", plateNum, orderInfoRes.getPlateNum());
            OrderSonInfo existsParam = new OrderSonInfo();
            existsParam.setServiceStatus(OrderStatusConstants.IN_PARK);
            existsParam.setPlateNum(plateNum);
            existsParam.setParkId(parkId);
            List<OrderSonInfo> existsOrders = orderSonInfoService.list(existsParam);
            if (CollectionUtils.isEmpty(existsOrders)) {
                return 0;
            }

            List<OrderSonInfo> updateOrders = new ArrayList<>(existsOrders.size());
            for (OrderSonInfo existsOrder : existsOrders) {
                OrderCarInfo orderCarInfo = orderCarInfoDao.selectByOrderNum(existsOrder.getOrderNum());
                //如果库里在场订单是断电应急并且当前订单不是
                if (orderCarInfo != null && orderCarInfo.getEnterWay() != null
                        && (!orderCarInfo.getEnterWay().equals(enterWay)
                        && Arrays.asList(orderCarInfo.getEnterWay(), enterWay).contains(5))) {
                    continue;
                }
                OrderSonInfo updateOrder = new OrderSonInfo();
                updateOrder.setId(existsOrder.getId());
                // 车辆重复入场,设置订单状态为异常状态
                updateOrder.setServiceStatus(OrderStatusConstants.EXCEPTION);
                updateOrder.setOddStatus(OrderOddStatusEnum.重复入场.getVal());
                updateOrder.setExitTime(orderInfoRes.getEnterTime());
                updateOrders.add(updateOrder);
            }
            orderSonInfoService.updateBatchById(updateOrders);
            return updateOrders.size();
        }
        return 0;
    }

    private void setOrderSonPrice(Long parkId, String orderNum, OrderSonInfo orderSonInfo) {
        // 正常离场时，汇总平台数据
        OrderPay orderPay = new OrderPay();
        orderPay.setParkId(parkId);
        orderPay.setOrderNum(orderNum);
        OrderSumFeeDto orderSumFeeDto = orderPayDao.sumFee(orderPay);
        if (orderSumFeeDto != null) {
            orderSonInfo.setTotalPrice(String.valueOf(orderSumFeeDto.getTotalPrice()));
            orderSonInfo.setPaidPrice(String.valueOf(orderSumFeeDto.getPaidPrice()));
            orderSonInfo.setDiscountPrice(String.valueOf(orderSumFeeDto.getDiscountPrice()));
        } else {
            orderSonInfo.setTotalPrice("0.00");
            orderSonInfo.setPaidPrice("0.00");
            orderSonInfo.setDiscountPrice("0.00");
        }
    }
}
