package com.icetech.park.service.order.impl.enter;

import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.response.p2c.CarEnterResult;
import com.icetech.park.service.freespace.UpdateRegionFreeSpaceServiceImpl;
import com.icetech.park.service.order.impl.OrderServiceImpl;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.park.service.report.ReportParamHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.annotation.Resource;
import java.util.concurrent.ThreadPoolExecutor;

@Slf4j
@Service
public class MasterChannelCarOrderEnterServiceImpl extends CommonEnterImpl {
    @Resource
    private OrderCarInfoDao orderCarInfoDao;
    @Autowired
    private UpdateRegionFreeSpaceServiceImpl updateRegionFreeSpaceService;
    @Autowired
    private OrderServiceImpl orderService;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;
    @Autowired
    private ParkRegionDao parkRegionDao;

    @Transactional
    public ObjectResponse<CarEnterResult> enter(CarEnterRequest enterRequest, ReportParamHolder paramHolder) {
        Long parkId = enterRequest.getParkId();
        //验证车辆类型
        validateCarInfo(enterRequest, enterRequest.getPlateNum(), enterRequest.getParkId());
        int influenceLines = 0;
        //判断是否重复入场，并将旧的订单记录异常离场
        try {
            influenceLines = checkSamePlate(enterRequest.getPlateNum(), enterRequest.getParkId(),
                    enterRequest.getEnterTime(), enterRequest.getEnterWay(), enterRequest.getProperty(), enterRequest.isReplenishOrder(),
                    enterRequest.getType());
        } catch (ResponseBodyException e) {
            if (e.getErrCode().equals(CodeConstants.ERROR_405)) {
                log.warn("处理失败: {}:{}. enterRequest[{}]", e.getErrCode(), e.getMessage(),
                        enterRequest, e);
                //返回
                CarEnterResult result = new CarEnterResult();
                result.setOrderNum(enterRequest.getOrderNum());
                return ObjectResponse.success(result);
            }
        }
        orderHandle(enterRequest, paramHolder);
        //记录轨迹表
        saveTrack(enterRequest, paramHolder.getParkChannel().getRegionId());

        //空车位处理
        ParkConfig parkConfig = paramHolder.getParkConfig();
        if (influenceLines == 0
                && (!isMonthCard(enterRequest, paramHolder)
                || NumberUtils.toPrimitive(parkConfig.getIsCardcount()) == 1)) {
            if (parkConfig.getCalcSpaceMethod(1) == 1) {
                updateRegionFreeSpaceService.addFreeSpace(parkId, paramHolder.getParkChannel().getRegionId(), -1);
            } else {
                updateRegionFreeSpaceService.syncFreeSpace(parkId, paramHolder.getParkChannel().getRegionId());
            }
            mqPushService.pushParkFreeSpace(parkId);
        }
        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
            @Override
            public void afterCommit() {
                asyncExecutor.execute(ThreadUtils.wrapTrace(() -> asyncHandle(enterRequest)));
            }
        });
        //返回
        CarEnterResult result = new CarEnterResult();
        result.setOrderNum(enterRequest.getOrderNum());
        return ObjectResponse.success(result);
    }

    private void orderHandle(CarEnterRequest enterRequest, ReportParamHolder paramHolder) {
        ParkInoutdevice parkChannel = paramHolder.getParkChannel();

        OrderInfo orderInfo = new OrderInfo();
        // 新增订单记录表 和 出入场记录表
        BeanUtils.copyProperties(enterRequest, orderInfo);
        String orderNum = enterRequest.getOrderNum();
        orderInfo.setOrderNum(orderNum);
        orderInfo.setHasSon(0);
        orderInfo.setRegionId(parkChannel.getRegionId());
        if (enterRequest.getNoneEnterFlag()) {
            orderInfo.setNoneEnterFlag(1);
        } else {
            orderInfo.setEnterChannelId(enterRequest.getInandoutCode());
        }
        orderInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
        try {
            orderService.addOrderInfo(orderInfo);
        } catch (DuplicateKeyException e) {
            orderNum = CodeTools.GenerateOrderNum();
            enterRequest.setOrderNum(orderNum);
            orderInfo.setOrderNum(orderNum);
            orderService.addOrderInfo(orderInfo);
        }
        log.info("[端云-主通道车辆入场服务] 插入订单信息表完成，orderNum：{}", orderNum);
        // 先查询订单号是否存在，再插入记录
        OrderCarInfo orderCarInfo = orderCarInfoDao.selectByOrderNum(orderNum);
        if (orderCarInfo == null) {
            // 新增订单进出场记录表
            OrderCarInfo carInfo = new OrderCarInfo();
            BeanUtils.copyProperties(enterRequest, carInfo);
            carInfo.setOrderNum(orderNum);
            if (NumberUtils.toPrimitive(orderInfo.getNoneEnterFlag()) == 0) {
                carInfo.setEnterChannelId(enterRequest.getInandoutCode());
                carInfo.setEnterNo(enterRequest.getInandoutName());
                carInfo.setEnterImage(enterRequest.getMaxImage());
                carInfo.setSmallEnterImage(enterRequest.getSmallImage());
                carInfo.setEnterOperAccount(enterRequest.getOperaUser());
                carInfo.setEnterReliability(enterRequest.getReliability());
            } else {
                carInfo.setEnterWay(null);
            }
            //处理标签
            delTags(enterRequest, orderNum, parkChannel.getRegionId(), carInfo);
            orderCarInfoDao.insert(carInfo);
            log.info("[端云-主通道车辆入场服务] 插入订单进出场表完成，orderNum：{}", orderNum);
        }
        if (enterRequest.isReplenishOrder()) {
            paramHolder.setReplenishOrder(orderInfo);
        }
    }


}
