package com.icetech.park.service.order.impl.enter;

import com.icetech.cloudcenter.api.order.CarOrderEnterService;
import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.response.p2c.CarEnterResult;
import com.icetech.park.service.freespace.UpdateParkFreeSpaceServiceImpl;
import com.icetech.park.service.order.impl.OrderServiceImpl;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderCarInfoConstant;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.service.report.ReportParamHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.annotation.Resource;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 端云入场处理
 * @author fangct
 */
@Slf4j
@Service("carOrderEnterService")
public class CarOrderEnterServiceImpl extends CommonEnterImpl implements CarOrderEnterService {
    @Resource
    private OrderCarInfoDao orderCarInfoDao;
    @Autowired
    private UpdateParkFreeSpaceServiceImpl updateParkFreeSpaceService;
    @Autowired
    private OrderServiceImpl orderService;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;
    @Autowired
    private MasterChannelCarOrderEnterServiceImpl masterChannelCarOrderEnterService;
    @Autowired
    private SubChannelCarOrderEnterServiceImpl subChannelCarOrderEnterService;

    @Override
    @Transactional
    public ObjectResponse<CarEnterResult> enter(CarEnterRequest enterRequest) {
        ReportParamHolder paramHolder = new ReportParamHolder(enterRequest.getParkId(), enterRequest.getParkCode(),
                enterRequest.getInandoutCode(), null, null, enterRequest.getPlateNum(), 1, null);
        return enter(enterRequest, paramHolder);
    }

    @Transactional
    public ObjectResponse<CarEnterResult> enter(CarEnterRequest enterRequest, ReportParamHolder paramHolder) {
        if (paramHolder == null) {
            return enter(enterRequest);
        }
        if (enterRequest.isAddedOrder()) {
            //返回
            CarEnterResult result = new CarEnterResult();
            result.setOrderNum(enterRequest.getOrderNum());
            return ObjectResponse.success(result);
        }
        String orderNum = StringUtils.isBlank(enterRequest.getOrderNum()) ? CodeTools.GenerateOrderNum() : enterRequest.getOrderNum();
        enterRequest.setOrderNum(orderNum);
        Park park = paramHolder.getParkInfo();
        //是否普通车场（非场中场）
        boolean normalPark = park.getIsInterior() == null || park.getIsInterior() == 0 || enterRequest.getInandoutCode() == null;
        //入场流程处理
        if (normalPark) {
            //普通车场，非多区域的车场
            return normalHandle(enterRequest, paramHolder);
        } else {
            //多区域车场
            ParkInoutdevice parkChannel = paramHolder.getParkChannel();
            if (Integer.valueOf(1).equals(parkChannel.getIsMaster())) {
                return masterChannelCarOrderEnterService.enter(enterRequest, paramHolder);
            } else {
                return subChannelCarOrderEnterService.enter(enterRequest, paramHolder);
            }
        }
    }

    private ObjectResponse<CarEnterResult> normalHandle(CarEnterRequest enterRequest, ReportParamHolder paramHolder) {
        String plateNum = enterRequest.getPlateNum();
        Long parkId = enterRequest.getParkId();

        //验证车牌类型
        validateCarInfo(enterRequest, plateNum, parkId);
        //判断是否重复入场，并取消旧的订单记录
        int influenceLines = 0;
        try {
            influenceLines = checkSamePlate(enterRequest.getPlateNum(), enterRequest.getParkId(),
                    enterRequest.getEnterTime(), enterRequest.getEnterWay(), enterRequest.getProperty(),
                    enterRequest.isReplenishOrder(), enterRequest.getType());
        } catch (ResponseBodyException e) {
            if (e.getErrCode().equals(CodeConstants.ERROR_405)) {
                log.warn("处理失败: {}:{}. enterRequest[{}]", e.getErrCode(), e.getMessage(),
                        enterRequest, e);
                //返回
                CarEnterResult result = new CarEnterResult();
                result.setOrderNum(enterRequest.getOrderNum());
                return ObjectResponse.success(result);
            }
        }
        //普通订单处理
        orderHandle(enterRequest, paramHolder);
        //空车位处理
        ParkConfig parkConfig = paramHolder.getParkConfig();
        if (influenceLines == 0
                && (!PlateTypeEnum.月卡车.getType().equals(enterRequest.getType())
                || NumberUtils.toPrimitive(parkConfig.getIsCardcount()) == 1)) {
            if (parkConfig.getCalcSpaceMethod(1) == 1) {
                updateParkFreeSpaceService.addFreeSpace(parkId, null,-1);
            } else {
                updateParkFreeSpaceService.syncFreeSpace(parkId, null);
            }
            mqPushService.pushParkFreeSpace(parkId);
        }
        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
            @Override
            public void afterCommit() {
                asyncExecutor.execute(ThreadUtils.wrapTrace(() -> asyncHandle(enterRequest)));
            }
        });
        //返回
        CarEnterResult result = new CarEnterResult();
        result.setOrderNum(enterRequest.getOrderNum());
        return ObjectResponse.success(result);
    }

    /**
     * 处理正常订单
     *
     * @param enterRequest 入场参数
     * @param paramHolder
     */
    private void orderHandle(CarEnterRequest enterRequest, ReportParamHolder paramHolder) {
        OrderInfo orderInfo = new OrderInfo();
        BeanUtils.copyProperties(enterRequest, orderInfo);
        String orderNum = enterRequest.getOrderNum();
        orderInfo.setOrderNum(orderNum);
        //断电应急入场 写入本地订单号为平台订单号
        if (OrderCarInfoConstant.IN_OUT_WAY_OFF_LINE == NumberUtils.toPrimitive(enterRequest.getEnterWay())){
            orderInfo.setLocalOrderNum(orderNum);
        }
        if(enterRequest.getNoneEnterFlag()) {
            orderInfo.setNoneEnterFlag(1);
        } else {
            orderInfo.setEnterChannelId(enterRequest.getInandoutCode());
        }
        orderInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
        try {
            orderService.addOrderInfo(orderInfo);
        } catch (DuplicateKeyException e) {
            orderNum = CodeTools.GenerateOrderNum();
            enterRequest.setOrderNum(orderNum);
            orderInfo.setOrderNum(orderNum);
            orderService.addOrderInfo(orderInfo);
        }
        log.info("[端云-车辆入场服务] 插入订单信息表完成，orderNum：{}", orderNum);
        // 先查询订单号是否存在，再插入记录
        OrderCarInfo orderCarInfo = orderCarInfoDao.selectByOrderNum(orderNum);
        if (orderCarInfo == null) {
            // 新增订单进出场记录表
            OrderCarInfo carInfo = new OrderCarInfo();
            BeanUtils.copyProperties(enterRequest, carInfo);
            carInfo.setOrderNum(orderNum);
            if (NumberUtils.toPrimitive(orderInfo.getNoneEnterFlag()) == 0) {
                carInfo.setEnterChannelId(enterRequest.getInandoutCode());
                carInfo.setEnterNo(enterRequest.getInandoutName());
                carInfo.setEnterImage(enterRequest.getMaxImage());
                carInfo.setSmallEnterImage(enterRequest.getSmallImage());
                carInfo.setEnterOperAccount(enterRequest.getOperaUser());
                carInfo.setEnterReliability(enterRequest.getReliability());
            } else {
                carInfo.setEnterWay(null);
            }
            //处理标签
            delTags(enterRequest, orderNum, null, carInfo);
            orderCarInfoDao.insert(carInfo);
            log.info("[端云-车辆入场服务] 插入订单进出场表完成，orderNum：{}", orderNum);
        }
        if (enterRequest.isReplenishOrder()) {
            paramHolder.setReplenishOrder(orderInfo);
        }
    }

}
