package com.icetech.park.service.order.impl;

import com.icetech.cloudcenter.api.fee.QueryOrderFeeService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.domain.request.pnc.EtcPayStatusRequest;
import com.icetech.cloudcenter.domain.response.pnc.EtcPayStatusResponse;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.basics.dao.park.ParkConfigDao;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.constants.SpecialConstants;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderNotpay;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.request.QueryOrderFeeRequest;
import com.icetech.cloudcenter.domain.response.NotPayDetail;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.order.service.OrderNotpayService;
import com.icetech.basics.service.park.impl.ParkServiceImpl;
import com.icetech.park.domain.vo.ParkRecoveryVo;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.down.pnc.impl.EtcPayStatusServiceImpl;
import com.icetech.park.service.park.ParkRecoveryService;
import com.icetech.park.service.queryfee.impl.P2cQueryFeeServiceImpl;
import com.icetech.park.service.queryfee.impl.PncQueryFeeServiceImpl;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.DataCollectionEnum;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.AssertTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

@Service("queryOrderFeeService")
@Slf4j
public class QueryOrderFeeServiceImpl implements QueryOrderFeeService {
    @Autowired
    private ParkConfigDao parkConfigDao;
    @Autowired
    private OrderInfoDao orderInfoDao;
    @Autowired
    private P2cQueryFeeServiceImpl p2cQueryFeeService;
    @Autowired
    private PncQueryFeeServiceImpl pncQueryFeeService;
    @Autowired
    protected ParkServiceImpl parkService;
    @Autowired
    protected CacheHandle cacheHandle;
    @Autowired
    protected RedisUtils redisUtils;
    @Autowired
    protected OrderNotpayService orderNotpayService;
    @Autowired
    protected EtcPayStatusServiceImpl etcPayStatusService;
    @Autowired
    protected OrderService orderService;
    @Autowired
    private ParkRecoveryService parkRecoveryService;

    @Override
    public ObjectResponse<QueryOrderFeeResponse> queryOrderFee(QueryOrderFeeRequest queryOrderFeeRequest) {
        String parkCode = queryOrderFeeRequest.getParkCode();
        String channelId = queryOrderFeeRequest.getChannelId();
        String plateNum = queryOrderFeeRequest.getPlateNum();
        String orderNum = queryOrderFeeRequest.getOrderNum();
        if (StringUtils.isNotBlank(parkCode) && (StringUtils.isNotBlank(channelId) || StringUtils.isNotBlank(plateNum))
                || StringUtils.isNotBlank(orderNum)) {
            try {
                return execute(queryOrderFeeRequest);
            } catch (ResponseBodyException e) {
                log.warn("拉取费用错误: {}:{}. queryOrderFeeRequest[{}]", e.getErrCode(), e.getMessage(), queryOrderFeeRequest, e);
                return ObjectResponse.failed(e.getErrCode(), e.getMessage());
            } catch (Exception e) {
                log.error("拉取费用错误: {}. queryOrderFeeRequest[{}]", e.getMessage(), queryOrderFeeRequest, e);
                return ObjectResponse.failed(CodeConstants.ERROR_3001);
            }
        } else {
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
    }

    @Override
    public ObjectResponse<QueryOrderFeeResponse> queryOrderFeeWithNotPay(QueryOrderFeeRequest queryOrderFeeRequest) {
        queryOrderFeeRequest.setWithNotPay(true);
        //查询当前订单费用
        return queryOrderFee(queryOrderFeeRequest);
    }

    @Override
    public ObjectResponse<EtcPayStatusResponse> etcPayStatus(EtcPayStatusRequest queryOrderFeeRequest, Long parkId, String topic) {
        return etcPayStatusService.etcPayStatus(queryOrderFeeRequest,parkId,topic);
    }

    @Override
    public ObjectResponse<QueryOrderFeeResponse> updateAndGetChannelFee(String parkCode, String channelCode, String plateNum,
                                                                        List<String> notPayOrderNums) {
        if (parkCode == null || channelCode == null) {
            throw new ResponseBodyException(CodeConstants.ERROR_400, "车场编号或通道编号未传");
        }
        String key = RedisConstants.CHANNEL_FEE_LOCK_KEY + parkCode + ":" + channelCode;
        boolean lock = redisUtils.tryLock(key, String.valueOf(notPayOrderNums), 1000);
        if (!lock) {
            return ObjectResponse.failed(CodeConstants.ERROR_405, "操作太快，请稍后重试");
        }
        try {
            QueryOrderFeeResponse channelFee = cacheHandle.getChannelFee(parkCode, channelCode);
            if (channelFee == null) {
                return ObjectResponse.failed(CodeConstants.ERROR_404, "车辆已离场");
            }
            if (!channelFee.getPlateNum().equals(plateNum)) {
                throw new ResponseBodyException(CodeConstants.ERROR_400, "当前通道车辆已发生变化，请刷新重试");
            }
            if (CollectionUtils.isEmpty(channelFee.getNotPayDetails()) && CollectionUtils.isEmpty(notPayOrderNums)) {
                return ObjectResponse.success(channelFee);
            }
            if (CollectionUtils.isEmpty(notPayOrderNums)) {
                channelFee.setNotPayDetails(null);
                channelFee.setTotalNotPayPrice(String.valueOf(0.00d));
                channelFee.setSumPrice(channelFee.getUnpayPrice());
                cacheHandle.setChannelFee(parkCode, channelCode, channelFee);
                return ObjectResponse.success(channelFee);
            }
            List<OrderNotpay> orderNotpays = orderNotpayService.getListByOrderNums(notPayOrderNums);
            if (CollectionUtils.isEmpty(orderNotpays)) {
                channelFee.setNotPayDetails(null);
                channelFee.setTotalNotPayPrice(String.valueOf(0.00d));
                channelFee.setSumPrice(channelFee.getUnpayPrice());
                cacheHandle.setChannelFee(parkCode, channelCode, channelFee);
                return ObjectResponse.success(channelFee);
            }
            List<NotPayDetail> notPayDetails = new ArrayList<>();
            BigDecimal totalNotPayPrice = new BigDecimal(0);
            for (OrderNotpay orderNotPay : orderNotpays) {
                if (orderNotPay.getOrderNum().equals(channelFee.getOrderNum())) {
                    log.info("[查询欠费记录] 已作为当次订单计费, 跳过欠费记录[{}]", orderNotPay);
                    continue;
                }
                NotPayDetail notPayDetail = NotPayDetail.builder()
                        .parkId(orderNotPay.getParkId())
                        .orderNum(orderNotPay.getOrderNum())
                        .enterTime(orderNotPay.getEnterTime())
                        .exitTime(orderNotPay.getExitTime())
                        .totalPrice(orderNotPay.getTotalPrice().toString())
                        .discountPrice(orderNotPay.getDiscountPrice().toString())
                        .unPayPrice(orderNotPay.getTotalPrice().toString())
                        .parkTime(orderNotPay.getEnterTime() == null
                                ? null : (orderNotPay.getExitTime() - orderNotPay.getEnterTime())).build();
                notPayDetails.add(notPayDetail);
                totalNotPayPrice = totalNotPayPrice.add(orderNotPay.getTotalPrice());
            }
            channelFee.setNotPayDetails(notPayDetails);
            channelFee.setTotalNotPayPrice(totalNotPayPrice.toString());
            BigDecimal sumPrice = new BigDecimal(channelFee.getUnpayPrice()).add(totalNotPayPrice);
            channelFee.setSumPrice(sumPrice.toString());
            cacheHandle.setChannelFee(parkCode, channelCode, channelFee);
            return ObjectResponse.success(channelFee);
        } finally {
            redisUtils.releaseLock(key);
        }
    }

    private ObjectResponse<QueryOrderFeeResponse> execute(QueryOrderFeeRequest queryOrderFeeRequest) {
        if (Boolean.TRUE.equals(queryOrderFeeRequest.getIsOffline())) {
            QueryOrderFeeResponse channelFee = cacheHandle.getChannelFee(queryOrderFeeRequest.getParkCode(), queryOrderFeeRequest.getChannelId());
            if (channelFee == null) {
                return ObjectResponse.failed(CodeConstants.ERROR_3008);
            }
            return ObjectResponse.success(channelFee);
        }
        String orderNum = queryOrderFeeRequest.getOrderNum();
        String parkCode = queryOrderFeeRequest.getParkCode();
        String plateNum = queryOrderFeeRequest.getPlateNum();
        OrderInfo orderInfo = new OrderInfo();
        Park park;
        //有订单号查询
        if (StringUtils.isNotBlank(orderNum)){
            orderInfo = orderInfoDao.selectByOrderNum(orderNum);
            AssertTools.notNull(orderInfo, CodeConstants.ERROR_402, "订单号不存在");
            park = parkService.findByParkId(orderInfo.getParkId()).getData();
        } else {
            park = parkService.findByParkCode(parkCode).getData();
            AssertTools.notNull(park, CodeConstants.ERROR_402, "车场未注册");
            if (plateNum != null) { // 有无车牌号参数的标识
                orderInfo.setPlateNum(plateNum);
                orderInfo.setParkId(park.getId());
                ObjectResponse<OrderInfo> infoObjectResponse = orderService.findInParkId(plateNum, park.getId());
                orderInfo = infoObjectResponse.getData();
            }
        }
        if (orderInfo != null && orderInfo.getServiceStatus() != null
                && (orderInfo.getServiceStatus().equals(OrderStatusConstants.LEAVED_PARK)
                    || orderInfo.getServiceStatus().equals(OrderStatusConstants.CANCEL))) {
            orderInfo = null;
        }
        if (orderInfo != null && orderInfo.getOrderNum() != null) {
            if (redisUtils.exists(RedisConstants.OFF_LINE_FEE_PROFILE + orderInfo.getOrderNum())) {
                QueryOrderFeeResponse channelFee = cacheHandle.getChannelFee(queryOrderFeeRequest.getParkCode(), queryOrderFeeRequest.getChannelId());
                if (channelFee == null) {
                    return ObjectResponse.failed(CodeConstants.ERROR_3001, "费用过期，请联系管理员重新查费");
                }
                return ObjectResponse.success(channelFee);
            }
        }
        //入口缴费模式下将orderInfo设为空
        ParkConfig parkConfig = parkConfigDao.selectByParkId(park.getId());
        if (parkConfig.getDataCollection().equals(DataCollectionEnum.端网云.getType())) {
            // 入口缴费模式,并是入口通道时将orderInfo设为空
            if (plateNum != null && parkConfig.getEntryPayFlag() != null && Integer.valueOf("1").equals(parkConfig.getEntryPayFlag())) {
                if (queryOrderFeeRequest.getChannelId() != null) {
                    ObjectResponse<ParkInoutdevice> channelInfo = parkService.getInOutDeviceByCode(park.getId(), queryOrderFeeRequest.getChannelId());
                    ParkInoutdevice parkInoutdevice = channelInfo.getData();
                    if (parkInoutdevice.getInandoutType() != null && Integer.valueOf("1").equals(parkInoutdevice.getInandoutType())) {
                        orderInfo = null;
                    }
                }
            }
            /*
             * 端网云架构
             */
            ObjectResponse<Void> objectResponse = pncQueryFeeService.queryFee(queryOrderFeeRequest, orderInfo, park);
            ObjectResponse<QueryOrderFeeResponse> response = new ObjectResponse<>();
            response.setCode(objectResponse.getCode());
            response.setMsg(objectResponse.getMsg());
            response.setTraceId(objectResponse.getTraceId());
            return response;
        } else {
            //入口欠费查询
            if (Integer.valueOf(1).equals(queryOrderFeeRequest.getExType())) {
                QueryOrderFeeResponse channelFee = cacheHandle.getChannelFee(queryOrderFeeRequest.getParkCode(), queryOrderFeeRequest.getChannelId());
                if (channelFee == null) {
                    channelFee = queryNotPay(queryOrderFeeRequest, parkConfig, park);
                }
                if (channelFee == null) {
                    return ObjectResponse.failed(CodeConstants.ERROR_3008);
                }
                return ObjectResponse.success(channelFee);
            }
            if (orderInfo == null) {
                if (!Boolean.TRUE.equals(queryOrderFeeRequest.getWithNotPay())) {
                    throw new ResponseBodyException(CodeConstants.ERROR_3001, "未找到入场信息");
                }
                //查询补缴
                QueryOrderFeeResponse queryOrderFeeResponse = queryNotPay(queryOrderFeeRequest, plateNum, parkConfig, null, park.getParkName());
                if (queryOrderFeeResponse == null) {
                    throw new ResponseBodyException(CodeConstants.ERROR_3001, "未找到入场信息");
                } else {
                    return ObjectResponse.success(queryOrderFeeResponse);
                }
            } else {
                ObjectResponse<QueryOrderFeeResponse> objectResponse = p2cQueryFeeService.queryFee(queryOrderFeeRequest, orderInfo, park, parkConfig);
                if (Boolean.TRUE.equals(queryOrderFeeRequest.getWithNotPay())) {
                    QueryOrderFeeResponse queryOrderFeeResponse = queryNotPay(queryOrderFeeRequest, plateNum, parkConfig, objectResponse.getData(), park.getParkName());
                    if (queryOrderFeeResponse == null) {
                        return objectResponse;
                    }
                    return ObjectResponse.success(queryOrderFeeResponse);
                }
                return objectResponse;
            }
        }
    }

    public QueryOrderFeeResponse queryNotPay(QueryOrderFeeRequest queryOrderFeeRequest, ParkConfig parkConfig, Park park) {
        if (queryOrderFeeRequest.getPlateNum() == null) {
            throw new ResponseBodyException(CodeConstants.ERROR_400, "车牌号不能为空");
        }
        return queryNotPay(queryOrderFeeRequest, queryOrderFeeRequest.getPlateNum(), parkConfig,
                QueryOrderFeeResponse.buildEmptyOrderFee(queryOrderFeeRequest.getPlateNum(), park.getParkName(),
                        parkConfig.getIsfreeAfterpay(15), queryOrderFeeRequest.getCarType(), queryOrderFeeRequest.getOrderNum()),
                park.getParkName());
    }
    public QueryOrderFeeResponse queryNotPay(QueryOrderFeeRequest queryOrderFeeRequest, String plateNum, ParkConfig parkConfig,
                             QueryOrderFeeResponse queryOrderFeeResponse, String parkName) {
        Long parkId = parkConfig.getParkId();
        ParkRecoveryVo recovery = parkRecoveryService.getParkRecoveryByParkId(parkId);
        if (recovery == null || !recovery.isEnableRecovery()) {
            return queryOrderFeeResponse;
        }
        if (plateNum == null) {
            plateNum = queryOrderFeeResponse == null? null : queryOrderFeeResponse.getPlateNum();
        }
        if (StringUtils.isEmpty(plateNum)
                || SpecialConstants.NO_PLATE_NUM.equals(plateNum) || SpecialConstants.NO_PLATE_NUM2.equals(plateNum)){
            return queryOrderFeeResponse;
        }
        if (queryOrderFeeResponse != null && NumberUtils.parseDouble(queryOrderFeeResponse.getTotalNotPayPrice()) > 0) {
            return queryOrderFeeResponse;
        }
        if (queryOrderFeeRequest.getParkCode() != null && queryOrderFeeRequest.getChannelId() != null && plateNum == null) {
            //直接从通道获取的费用
            return queryOrderFeeResponse;
        }
        if (parkConfig.getEnterpayType() == null && parkConfig.getExitpayType() == null) {
            return queryOrderFeeResponse;
        }
        Integer exType = queryOrderFeeRequest.getExType();
        if (queryOrderFeeRequest.getChannelId() != null) {
            ObjectResponse<ParkInoutdevice> parkInoutDeviceObjectResponse =
                    parkService.getInOutDeviceByCode(parkId, queryOrderFeeRequest.getChannelId());
            if (ObjectResponse.isSuccess(parkInoutDeviceObjectResponse)) {
                exType = parkInoutDeviceObjectResponse.getData().getInandoutType();
            }
        }
        List<OrderNotpay> orderNotPays = orderNotpayService.queryNotPayFee(parkId, plateNum, exType, parkConfig);
        if (CollectionUtils.isEmpty(orderNotPays)) {
            return queryOrderFeeResponse;
        }
        if (queryOrderFeeResponse == null) {
            queryOrderFeeResponse = QueryOrderFeeResponse.buildEmptyOrderFee(plateNum, parkName, parkConfig.getIsfreeAfterpay(15),
                    queryOrderFeeRequest.getCarType(), null);
        }
        List<NotPayDetail> notPayDetails = new ArrayList<>();
        BigDecimal totalNotPayPrice = new BigDecimal(0);
        for (OrderNotpay orderNotPay : orderNotPays) {
            if (orderNotPay.getOrderNum().equals(queryOrderFeeResponse.getOrderNum())) {
                log.info("[查询欠费记录] 已作为当次订单计费, 跳过欠费记录[{}]", orderNotPay);
                continue;
            }
            NotPayDetail notPayDetail = NotPayDetail.builder()
                    .parkId(orderNotPay.getParkId())
                    .orderNum(orderNotPay.getOrderNum())
                    .enterTime(orderNotPay.getEnterTime())
                    .exitTime(orderNotPay.getExitTime())
                    .totalPrice(orderNotPay.getTotalPrice().toString())
                    .discountPrice(orderNotPay.getDiscountPrice().toString())
                    .unPayPrice(orderNotPay.getTotalPrice().toString())
                    .parkTime(orderNotPay.getEnterTime() == null
                            ? null : (orderNotPay.getExitTime() - orderNotPay.getEnterTime()))
                    .orderSource(orderNotPay.getOrderSource())
                    .build();
            notPayDetails.add(notPayDetail);
            totalNotPayPrice = totalNotPayPrice.add(orderNotPay.getTotalPrice());
        }
        queryOrderFeeResponse.setNotPayDetails(notPayDetails);
        queryOrderFeeResponse.setTotalNotPayPrice(totalNotPayPrice.toString());
        BigDecimal sumPrice = new BigDecimal(queryOrderFeeResponse.getUnpayPrice()).add(totalNotPayPrice);
        queryOrderFeeResponse.setSumPrice(sumPrice.toString());

        return queryOrderFeeResponse;
    }

}
