package com.icetech.park.service.order.impl;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.icetech.basics.domain.entity.park.BasePark;
import com.icetech.basics.service.park.impl.ParkServiceImpl;
import com.icetech.cloudcenter.api.order.OrderCarInfoService;
import com.icetech.cloudcenter.api.order.OrderPayService;
import com.icetech.cloudcenter.api.order.OrderRefundService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.MoneyTool;
import com.icetech.order.dao.OrderPayDao;
import com.icetech.order.dao.OrderRefundDao;
import com.icetech.order.dao.OrderRefundRecordMapper;
import com.icetech.order.domain.dto.OrderRefundDTO;
import com.icetech.order.domain.dto.OrderRefundRecordDTO;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.order.domain.entity.OrderRefund;
import com.icetech.order.domain.entity.OrderRefundRecord;
import com.icetech.park.domain.request.OrderRefundParam;
import com.icetech.park.domain.request.RefundParam;
import com.icetech.park.domain.request.RefundRecordParam;
import com.icetech.paycenter.api.IPayCenterService;
import com.icetech.paycenter.domain.normalpay.request.RefundRequest;
import com.icetech.paycenter.domain.normalpay.response.RefundResponse;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 订单退款
 */
@Slf4j
@Service
public class OrderRefundServiceImpl implements OrderRefundService {

    @Autowired
    private OrderService orderService;

    @Autowired
    private OrderCarInfoService orderCarInfoService;

    @Autowired
    private OrderPayService orderPayService;

    @Autowired
    private OrderPayDao orderPayDao;

    @Autowired
    private OrderRefundDao orderRefundDao;

    @Autowired
    private ParkServiceImpl parkService;

    @Autowired
    private IPayCenterService payCenterService;

    @Autowired
    private OrderRefundRecordMapper orderRefundRecordMapper;


    /**
     * 订单退款
     *
     * @param param 请求参数
     * @return obj
     */
    @Override
    public ObjectResponse<Object> orderRefund(OrderRefundParam param) {
        log.info("订单：【{}】开始退款", param.getOrderNum());
        try {
            //校验是否有待审核订单
            OrderRefund refund = orderRefundDao.selectOne(Wrappers.lambdaQuery(OrderRefund.class)
                    .eq(OrderRefund::getOrderNum, param.getOrderNum())
                    .in(OrderRefund::getRefundStatus, Arrays.asList(0, 1, 2, 4))
                    .last("LIMIT 1")
            );
            if (refund != null) {
                return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), "当前订单存在未退款审核，请先完成退款审核！");
            }
            // 获取订单
            OrderInfo order = this.getOrderByOrderNum(param.getOrderNum());
            // 获取车辆信息
            OrderCarInfo carInfo = this.getOrderCarInfoByOrderNum(param.getOrderNum());
            // 获取支付明细
            List<OrderPay> orderPays = this.getOrderPayByOrderNum(param.getOrderNum());
            // 校验退款条件
            checkOrderAmount(orderPays, order, param);
            // 执行退款逻辑
            executeRefund(orderPays, order, carInfo, param);
            return ObjectResponse.success();
        } catch (IllegalArgumentException e) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), e.getMessage());
        } catch (Exception e) {
            log.error("退款处理失败 | orderNum:{}", param.getOrderNum(), e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR);
        }
    }

    /**
     * 退款记录
     *
     * @param param 请求参数
     * @return 返回列表
     */
    @Override
    public ObjectResponse<Map<String, Object>> getOrderRefundList(RefundParam param, List<Long> parkIds) {
        Page<OrderRefundDTO> page = new Page<>(param.getPageNo(), param.getPageSize());
        Page<OrderRefundDTO> refundPage = orderRefundDao.getOrderRefundList(page, param);
        Map<String, BigDecimal> sumRefundPrice = orderRefundDao.sumRefundPrice(param);
        Map<String, Object> map = new HashMap<>();
        map.put("total", refundPage.getTotal());
        map.put("orderRefunds", refundPage.getRecords());
        map.put("sumRefundPrice", sumRefundPrice.get("sumRefundPrice"));
        map.put("sumActualRefundPrice", sumRefundPrice.get("sumActualRefundPrice"));
        return ObjectResponse.success(map);
    }

    /**
     * 订单退款审核
     *
     * @param param 请求参数
     * @return obj
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ObjectResponse<Object> orderRefundAudit(OrderRefundParam param) {
        OrderRefund refund = orderRefundDao.selectById(param.getId());
        if (refund == null) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), "未找到退款记录");
        }
        try {
            OrderInfo order = this.getOrderByOrderNum(refund.getOrderNum());
            ObjectResponse<Object> response = this.checkRefund(order, refund, param);
            if (!ObjectResponse.isSuccess(response)) {
                return response;
            }
            OrderPay pay = orderPayDao.selectLimitOne(Wrappers.lambdaQuery(OrderPay.class)
                    .eq(OrderPay::getId, refund.getPayId()));
            if (param.getRefundStatus() != null && param.getRefundStatus() == -1) {
                refund.setRefundStatus(param.getRefundStatus());
                refund.setUpdateTime(new Date());
                refund.setRefundReason(param.getRefundReason());
                orderRefundDao.updateById(refund);
                return ObjectResponse.success();
            }
            return processSingleRefundAsync(order, pay, refund, param);
        } catch (Exception e) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), e.getMessage());
        }
    }

    @Override
    public ObjectResponse<Page<OrderRefundRecordDTO>> orderRefundRecord(RefundRecordParam param) {
        Page<OrderRefundRecordDTO> page = new Page<>(param.getPageNo(), param.getPageSize());
        Page<OrderRefundRecordDTO> refundPage = orderRefundRecordMapper.getOrderRefundRecordList(page, param);
        return ObjectResponse.success(refundPage);
    }

    /**
     * 执行退款逻辑，使用线程池并行处理多笔退款
     *
     * @param orderPays 支付订单列表
     * @param order     主订单
     * @param param     退款参数
     */
    private void executeRefund(List<OrderPay> orderPays, OrderInfo order, OrderCarInfo carInfo, OrderRefundParam param) {
        Map<String, BigDecimal> refundAllocations = allocateRefundAmounts(orderPays, param.getRefundPrice());
        Map<String, OrderPay> payMap = orderPays.stream()
                .collect(Collectors.toMap(OrderPay::getTradeNo, Function.identity()));
        refundAllocations.forEach((k, v) -> {
            OrderPay pay = payMap.get(k);
            saveRefund(pay, order, carInfo, v, param);
        });
    }

    /**
     * 处理单笔退款
     *
     * @param order 订单
     * @param pay   支付
     * @param param 入参
     */
    private ObjectResponse<Object> processSingleRefundAsync(OrderInfo order, OrderPay pay,
                                                            OrderRefund refund, OrderRefundParam param) {
        BasePark park = parkService.getById(pay.getParkId());
        String newTradeNo = "R" + CodeTools.GenerateTradeNo();
        try {
            log.info("开始处理单笔退款 | orderNum:{} | tradeNo:{} | refundPrice:{}",
                    order.getOrderNum(), newTradeNo, param.getRefundPrice());
            // 调用退款接口
            ObjectResponse<RefundResponse> response = callRefundApi(park.getParkCode(),
                    pay, newTradeNo, param.getRefundPrice(), param.getReason());
            log.info("退款返回信息：{}", response);
            refund.setActualRefundPrice(param.getRefundPrice());
            if (!ObjectResponse.isSuccess(response) || response.getData() == null) {
                // 校验退款响应
                String msg = response.getData() == null ? "退款失败" : response.getMsg();
                return handleValidationFailureAndUpdate(order, pay, param, refund, msg, 4, 1);
            }
            String msg = response.getMsg();
            int refundStatus = 0;
            if (response.getData().getRefundStatus() == 4) {
                refundStatus = 4;
            }
            if (response.getData().getRefundStatus() == 3) {
                refund.setTraceNo(park.getParkCode() + newTradeNo);
                refundStatus = 3;
                updateOrderAndPay(order, pay, param); // 更新订单和支付记录
            }
            return handleValidationFailureAndUpdate(order, pay, param, refund, msg, refundStatus, 0);
        } catch (Exception e) {
            return ObjectResponse.failed("退款失败，请联系管理员！");
        }
    }

    /**
     * 调用退款接口
     *
     * @param parkCode    停车场信息
     * @param orderPay    支付订单
     * @param tradeNo     交易号
     * @param refundPrice 退款金额
     * @return 退款接口响应字符串
     */
    @SneakyThrows
    private ObjectResponse<RefundResponse> callRefundApi(String parkCode, OrderPay orderPay, String tradeNo,
                                                         BigDecimal refundPrice, String reason) {
        RefundRequest request = new RefundRequest();
        request.setParkCode(parkCode);
        request.setTradeNo(orderPay.getTradeNo());
        request.setOutTradeNo(orderPay.getThirdTradeNo());
        request.setRefundTradeNo(tradeNo);
        request.setPrice(String.valueOf(MoneyTool.fromYuanToFen(String.valueOf(refundPrice))));
        request.setOrderNote(reason);
        return payCenterService.refund(request);
    }


    /**
     * 分配每笔支付订单的退款金额
     *
     * @param orderPays        支付订单列表
     * @param totalRefundPrice 总退款金额
     * @return 每笔支付订单的退款金额映射，key为tradeNo，value为退款金额
     */
    private Map<String, BigDecimal> allocateRefundAmounts(List<OrderPay> orderPays, BigDecimal totalRefundPrice) {
        Map<String, BigDecimal> allocations = new HashMap<>();
        BigDecimal remaining = totalRefundPrice;
        for (OrderPay orderPay : orderPays) {
            BigDecimal available = strToBig(orderPay.getPaidPrice()).subtract(nullToZero(orderPay.getRefundPrice()));
            if (available.compareTo(BigDecimal.ZERO) > 0 && remaining.compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal refundPrice = remaining.min(available);
                allocations.put(orderPay.getTradeNo(), refundPrice);
                remaining = remaining.subtract(refundPrice);
            }
        }
        return allocations;
    }

    /**
     * 更新订单和支付记录（批量更新）
     *
     * @param order 主订单
     */
    private void updateOrderAndPay(OrderInfo order, OrderPay pay, OrderRefundParam param) {
        pay.setRefundPrice(nullToZero(pay.getRefundPrice()).add(param.getRefundPrice()));
        pay.setPayStatus(2); // 设置为已退款状态
        pay.setUpdateUser(param.getUserName()); // 设置更新用户
        orderPayService.modifyOrderPay(pay);
        order.setRefundPrice(nullToZero(order.getRefundPrice()).add(param.getRefundPrice())); // 更新订单退款金额
        orderService.updateOrderInfo(order); // 更新主订单
        log.info("退款完成 | orderNum:{} | totalRefunded:{}", order.getOrderNum(), param.getRefundPrice());
    }

    /**
     * 计算订单剩余可退款金额
     *
     * @param orderPays 支付订单列表
     * @return 剩余可退款金额
     */
    private BigDecimal calculateRemainAmount(List<OrderPay> orderPays) {
        return orderPays.stream().map(pay -> strToBig(pay.getPaidPrice())
                        .subtract(nullToZero(pay.getRefundPrice())))
                .filter(amount -> amount.compareTo(BigDecimal.ZERO) > 0)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 获取订单信息
     *
     * @param orderNum 订单号
     * @return order
     */
    private OrderInfo getOrderByOrderNum(String orderNum) {
        ObjectResponse<OrderInfo> info = orderService.findByOrderNum(orderNum);
        if (!ObjectResponse.isSuccess(info)) {
            throw new IllegalArgumentException("未查到订单");
        }
        return info.getData();
    }

    /**
     * 获取订单车辆
     *
     * @param orderNum 订单号
     * @return order
     */
    private OrderCarInfo getOrderCarInfoByOrderNum(String orderNum) {
        return orderCarInfoService.getByOrderNum(orderNum);
    }

    /**
     * 支付明细校验
     *
     * @param orderNum 订单号
     * @return order
     */
    private List<OrderPay> getOrderPayByOrderNum(String orderNum) {
        ObjectResponse<List<OrderPay>> pay = orderPayService.selectByOrderNum(orderNum);
        if (!ObjectResponse.isSuccess(pay)) {
            throw new IllegalArgumentException("未找到支付信息！");
        }
        List<OrderPay> pays = pay.getData();
        if (CollectionUtils.isEmpty(pays)) {
            throw new IllegalArgumentException("未找到支付信息！");
        }
        List<OrderPay> filterPays = pays.stream()
                .filter(this::isValidPayCondition)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(filterPays)) {
            throw new IllegalArgumentException("暂不支持退款！");
        }
        return filterPays;
    }

    /**
     * 校验订单金额和退款条件
     *
     * @param orderPays 支付订单列表
     * @param order     主订单
     * @param param     退款参数
     * @throws IllegalArgumentException 如果校验失败，抛出异常
     */
    private void checkOrderAmount(List<OrderPay> orderPays, OrderInfo order, OrderRefundParam param) {
        if (Objects.isNull(order)) {
            throw new IllegalArgumentException("未找到当前订单");
        }
        if (orderPays.isEmpty()) {
            throw new IllegalArgumentException("未查到支付明细");
        }
        if (param.getRefundPrice().compareTo(BigDecimal.ZERO) <= 0) {
            throw new IllegalArgumentException("退款金额必须大于0");
        }
        BigDecimal remainAmount = calculateRemainAmount(orderPays); // 计算剩余可退金额
        if (remainAmount.compareTo(BigDecimal.ZERO) <= 0) {
            throw new IllegalArgumentException("已缴金额已退还");
        }
        if (remainAmount.compareTo(param.getRefundPrice()) < 0) {
            throw new IllegalArgumentException("订单剩余可退款金额不足");
        }
    }


    /**
     * 校验退款信息
     * 检查退款记录是否存在、状态是否可操作，并校验申请退款金额是否合法。
     *
     * @param refund 退款记录实体
     * @param param  退款请求参数，包含申请退款金额等
     * @return 校验结果，成功返回 success，失败返回 failed 并附带错误信息
     */
    private ObjectResponse<Object> checkRefund(OrderInfo order, OrderRefund refund, OrderRefundParam param) {
        if (Objects.isNull(refund)) {
            log.warn("退款校验失败：退款记录不存在 | refundId:{}", param.getId());
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), "退款记录不存在");
        }
        if (refund.getRefundStatus() == -1 || refund.getRefundStatus() == 3) {
            log.warn("退款校验失败：退款记录状态不可操作 | refundId:{} | status:{}", param.getId(), refund.getRefundStatus());
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), "已取消或退款成功不可操作");
        }
        List<OrderPay> pays = orderPayDao.selectList(Wrappers.lambdaQuery(OrderPay.class)
                .eq(OrderPay::getOrderNum, refund.getOrderNum())
                .and(wrapper -> wrapper
                        .in(OrderPay::getPayChannel, Arrays.asList(2, 9))
                        .or()
                        .eq(OrderPay::getPayChannel, 1)
                        .in(OrderPay::getPayWay, Arrays.asList(2, 3))
                )
        );
        Map<Long, OrderPay> payMap = pays.stream().collect(Collectors.toMap(OrderPay::getId, Function.identity()));
        OrderPay pay = payMap.get(refund.getPayId());
        if (Objects.isNull(pay)) {
            log.error("退款校验失败：未找到当前退款记录对应的支付明细 | refundId:{} | payId:{}", param.getId(), refund.getPayId());
            return handleValidationFailureAndUpdate(order, null, param, refund, "支付记录不存在", -1, 1);
        }
        List<OrderRefund> successfulRefunds = orderRefundDao.selectList(refund.getOrderNum());
        BigDecimal totalPaidAmount = pays.stream()
                .map(orderPay -> strToBig(orderPay.getPaidPrice()))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal totalRefundedAmount = successfulRefunds.stream()
                .map(r -> nullToZero(r.getActualRefundPrice())) // 使用 nullToZero 处理可能为 null 的实际退款金额
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal orderRemainAmount = totalPaidAmount.subtract(totalRefundedAmount);
        if (orderRemainAmount.compareTo(BigDecimal.ZERO) <= 0) {
            log.warn("退款校验失败：订单已全额退款 | orderNum:{} | remain:{}", refund.getOrderNum(), orderRemainAmount);
            return handleValidationFailureAndUpdate(order, pay, param, refund, "订单已全额退款", -1, 1);
        }
        if (param.getRefundPrice().compareTo(orderRemainAmount) > 0) {
            log.warn("退款校验失败：申请退款金额大于订单剩余可退金额 | orderNum:{} | apply:{} | remain:{}",
                    refund.getOrderNum(), param.getRefundPrice(), orderRemainAmount);
            return handleValidationFailureAndUpdate(order, pay, param, refund, "申请退款金额大于订单剩余可退金额", -1, 1);
        }
        BigDecimal payRemainAmount = strToBig(pay.getPaidPrice()).subtract(nullToZero(pay.getRefundPrice()));
        if (payRemainAmount.compareTo(BigDecimal.ZERO) <= 0) {
            log.warn("退款校验失败：此笔支付明细已全额退款 | payId:{} | remain:{}", pay.getId(), payRemainAmount);
            return handleValidationFailureAndUpdate(order, pay, param, refund, "此笔支付明细已全额退款", -1, 1);
        }
        if (param.getRefundPrice().compareTo(payRemainAmount) > 0) {
            log.warn("退款校验失败：申请退款金额大于此笔支付明细剩余可退金额 | payId:{} | apply:{} | remain:{}",
                    pay.getId(), param.getRefundPrice(), payRemainAmount);
            return handleValidationFailureAndUpdate(order, pay, param, refund, "申请退款金额大于此笔支付明细剩余可退金额", -1, 1);
        }
        log.info("退款校验成功 | refundId:{} | orderNum:{} | applyPrice:{}", param.getId(), refund.getOrderNum(), param.getRefundPrice());
        return ObjectResponse.success();
    }

    /**
     * 处理校验失败情况：设置退款记录状态为取消，记录失败原因，更新数据库，并返回失败响应。
     * 提取公共代码，避免重复。
     *
     * @param refund  退款记录实体
     * @param message 失败原因消息
     * @return 失败的 ObjectResponse
     */
    private ObjectResponse<Object> handleValidationFailureAndUpdate(OrderInfo order, OrderPay pay, OrderRefundParam param, OrderRefund refund,
                                                                    String message, Integer refundStatus, int type) {
        if (type == 1) {
            refund.setActualRefundPrice(BigDecimal.ZERO);
        }
        refund.setRefundStatus(refundStatus); // 设置状态为已取消
        refund.setRefundFailMsg(message); // 设置失败消息
        orderRefundDao.updateById(refund); // 更新退款记录到数据库
        if (pay != null) {
            saveOrderRefundRecord(order, pay, refund, param);
        }
        if (type == 1) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), message); // 返回失败响应
        }
        return ObjectResponse.success();
    }

    /**
     * 保存操作记录
     *
     * @param order  订单
     * @param refund 退款信息
     * @param param  退款参数
     */
    private void saveOrderRefundRecord(OrderInfo order, OrderPay pay, OrderRefund refund, OrderRefundParam param) {
        OrderRefundRecord refundRecord = new OrderRefundRecord();
        refundRecord.setParkId(order.getParkId().intValue());
        refundRecord.setOrderNum(order.getOrderNum());
        refundRecord.setTradeNo(pay.getTradeNo());
        refundRecord.setPlateNo(order.getPlateNum());
        refundRecord.setAmount(param.getRefundPrice());
        refundRecord.setPayWay(pay.getPayWay());
        refundRecord.setPayTime(DateUtil.date(pay.getPayTime() * 1000));
        refundRecord.setRefundStatus(refund.getRefundStatus());
        refundRecord.setRefundReason(refund.getRefundReason());
        refundRecord.setRefundTime(refund.getCreateTime());
        refundRecord.setOptionName(param.getUserName());
        refundRecord.setOptionTime(new Date());
        refundRecord.setCreateTime(new Date());
        refundRecord.setRefundFailMsg(refund.getRefundFailMsg());
        orderRefundRecordMapper.insert(refundRecord);
    }


    /**
     * 保存退款记录到数据库
     *
     * @param pay         支付明细
     * @param order       订单明细
     * @param carInfo     车辆明细
     * @param refundPrice 退款金额
     * @param param       退款参数
     */
    private void saveRefund(OrderPay pay, OrderInfo order, OrderCarInfo carInfo,
                            BigDecimal refundPrice, OrderRefundParam param) {
        Date date = new Date();
        OrderRefund refund = new OrderRefund();
        refund.setParkId(pay.getParkId());
        refund.setPayId(pay.getId());
        refund.setOrderNum(order.getOrderNum());
        refund.setCarType(order.getType());
        refund.setPlateNum(order.getPlateNum());
        refund.setPlateColor(carInfo.getPlateColor());
        refund.setEnterTime(order.getEnterTime());
        refund.setExitTime(order.getExitTime());
        refund.setRefundPrice(refundPrice);
        refund.setRefundStatus(0);
        refund.setRefundReason(param.getReason());
        refund.setCreateTime(date);
        refund.setUpdateTime(date);
        refund.setDeleted(0);
        refund.setPayChannel(pay.getPayChannel());
        orderRefundDao.insert(refund);
    }

    private boolean isValidPayCondition(OrderPay p) {
        return p.getPayChannel() == 2 || p.getPayChannel() == 9
                || (p.getPayChannel() == 1 && (p.getPayWay() == 2 || p.getPayWay() == 3));
    }

    /**
     * 将null转换为BigDecimal.ZERO
     *
     * @param value 输入值
     * @return 非null的BigDecimal值
     */
    public static BigDecimal nullToZero(BigDecimal value) {
        return value != null ? value : BigDecimal.ZERO;
    }

    public BigDecimal strToBig(String value) {
        return value != null ? new BigDecimal(value) : BigDecimal.ZERO;
    }

}
