package com.icetech.park.service.monthcar.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import com.icetech.basics.dao.charge.ParkChargeconfigDao;
import com.icetech.basics.dao.park.ParkDao;
import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.basics.domain.dto.ParkRegionDto;
import com.icetech.basics.domain.entity.park.ParkChargeconfig;
import com.icetech.cloudcenter.api.month.MonthProductService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.park.query.ParkQuery;
import com.icetech.cloudcenter.domain.park.query.ParkSpaceInfo;
import com.icetech.cloudcenter.domain.park.query.ParkSpaceMsgInfo;
import com.icetech.cloudcenter.domain.pay.MonthOpenCardDto;
import com.icetech.common.domain.Page;
import com.icetech.fee.dao.monthcar.MonthInfoDao;
import com.icetech.fee.dao.monthcar.MonthProductDao;
import com.icetech.fee.domain.entity.monthcar.MonthInfo;
import com.icetech.fee.domain.entity.monthcar.MonthProduct;
import com.icetech.order.dao.OrderNotpayDao;
import com.icetech.order.domain.entity.OrderNotpay;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.cloudcenter.domain.response.MonthProductDto;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.NumberUtils;
import com.icetech.third.anno.DS_SLAVE;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Slf4j
@Service("monthProductService")
public class MonthProductServiceImpl implements MonthProductService {
    @Resource
    private MonthProductDao monthProductDao;
    @Resource
    private MonthInfoDao monthInfoDao;
    @Resource
    private MonthCarServiceImpl monthCarService;
    @Resource
    private ParkService parkService;

    @Resource
    private ParkRegionDao parkRegionDao;

    @Resource
    private OrderNotpayDao orderNotpayDao;

    @Autowired
    private ParkDao parkDao;

    @Autowired
    private ParkChargeconfigDao parkChargeconfigDao;
    //有效期月卡
    public static final Integer VALID_CARD = 1;

    @Override
    public MonthProduct selectMonthProductById(Long monthProductId) {
        return monthProductDao.selectById(monthProductId);
    }

    @Override
    public ObjectResponse<MonthProduct> getValidMonthProduct(Long parkId, String plateNum) {
        //获取有效月卡记录信息
        MonthInfo monthInfo = monthCarService.getMonthInfo(parkId, plateNum);
        if (monthInfo != null) {
            Long productId = monthInfo.getProductId();
            MonthProduct monthProduct = monthProductDao.selectById(productId);
            return ObjectResponse.success(monthProduct);
        }
        return ObjectResponse.failed(CodeConstants.ERROR_404, CodeConstants.getName(CodeConstants.ERROR_404));
    }

    @Override
    public ObjectResponse<MonthProduct> getValidMonthProduct(Long parkId, String plateNum, Long regionId) {
        //获取有效月卡记录信息
        MonthInfo monthInfo = monthCarService.findMonthInfo(parkId, plateNum, regionId, VALID_CARD);
        if (monthInfo != null) {
            Long productId = monthInfo.getProductId();
            MonthProduct monthProduct = monthProductDao.selectById(productId);
            return ObjectResponse.success(monthProduct);
        }
        return ObjectResponse.failed(CodeConstants.ERROR_404, CodeConstants.getName(CodeConstants.ERROR_404));
    }

    @Override
    //@DS_SLAVE
    public ObjectResponse<List<MonthProductDto>> getMonthProductList(String parkCode, Long monthInfoId) {
        ObjectResponse<Park> parkResp = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(parkResp);
        Park park = parkResp.getData();
        List<MonthProduct> monthProducts = monthProductDao.selectListByParkId(park.getId());
        //封装返回参数
        List<MonthProductDto> monthProductDtos = Lists.newArrayList();
        MonthInfo monthInfo = monthInfoDao.load(Math.toIntExact(monthInfoId));
        Long productId = null;
        if (Objects.nonNull(monthInfo)) {
            MonthProduct monthProduct = monthProductDao.selectById(monthInfo.getProductId());
            if (Objects.nonNull(monthProduct)) {
                if (monthProduct.getStatus() == 1 && NumberUtils.toDouble(monthProduct.getPaymoney()) > 0) {
                    MonthProductDto monthProductDto = new MonthProductDto();
                    monthProductDto.setId(monthProduct.getId());
                    monthProductDto.setDuration(monthProduct.getDuration());
                    monthProductDto.setPaymoney(monthProduct.getPaymoney());
                    monthProductDto.setName(monthProduct.getName());
                    productId = monthProduct.getId();
                    monthProductDtos.add(monthProductDto);
                }
            }
        }
        Long finalProductId = productId;
        monthProducts.forEach(monthProduct -> {
            if (finalProductId != null && finalProductId.equals(monthProduct.getId())) {
                log.info("当前月卡产品已经存在");
            } else {
                //0元月卡不允许续费
                if (NumberUtils.toDouble(monthProduct.getPaymoney()) > 0) {
                    MonthProductDto monthProductDto = new MonthProductDto();
                    monthProductDto.setId(monthProduct.getId());
                    monthProductDto.setDuration(monthProduct.getDuration());
                    monthProductDto.setPaymoney(monthProduct.getPaymoney());
                    monthProductDto.setName(monthProduct.getName());
                    monthProductDtos.add(monthProductDto);
                }
            }
        });
        return ObjectResponse.success(monthProductDtos);
    }

    @Override
    //@DS_SLAVE
    public ObjectResponse<List<MonthProductDto>> getMonthProductList(String parkCode) {
        ObjectResponse<Park> parkResp = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(parkResp);
        Park park = parkResp.getData();
        List<MonthProduct> monthProducts = monthProductDao.selectListByParkId(park.getId());
        if (CollectionUtil.isEmpty(monthProducts)) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        }
        //封装返回参数
        List<MonthProductDto> monthProductDtos = monthProducts.stream().map(monthProduct -> {
            MonthProductDto monthProductDto = new MonthProductDto();
            monthProductDto.setId(monthProduct.getId());
            monthProductDto.setDuration(monthProduct.getDuration());
            monthProductDto.setPaymoney(monthProduct.getPaymoney());
            monthProductDto.setName(monthProduct.getName());
            return monthProductDto;
        }).collect(Collectors.toList());
        return ObjectResponse.success(monthProductDtos);
    }

    /**
     * 获取停车场空间信息分页列表
     *
     * @param parkQuery 请求参数
     * @return 停车场空间信息分页列表
     */
    @Override
    @DS_SLAVE
    public ObjectResponse<Page<ParkSpaceMsgInfo>> getParkSpacePage(ParkQuery parkQuery) {
        // 根据查询条件获取停车场列表
        List<ParkSpaceInfo> parkList = parkDao.getParkList(parkQuery);
        // 如果停车场列表为空，直接返回空分页结果
        if (CollectionUtils.isEmpty(parkList)) {
            return ObjectResponse.success(Page.instance(0, 0, Collections.emptyList()));
        }
        // 将停车场列表转换为 Map，方便后续根据 parkId 查找
        Map<Long, ParkSpaceInfo> parkMap = parkList.stream().collect(Collectors.toMap(ParkSpaceInfo::getId, Function.identity()));
        // 提取停车场 ID 列表
        List<Long> parkIds = new ArrayList<>(parkMap.keySet());
        // 根据停车场 ID 列表和卡类型分页查询月卡产品
        com.github.pagehelper.Page<MonthProduct> productsPage = PageHelper.startPage(parkQuery.getPageNo(), parkQuery.getPageSize())
                .doSelectPage(() ->
                        monthProductDao.selectListBy(parkIds, parkQuery.getCardType())
                );
        // 如果月卡产品列表为空，直接返回空分页结果（基于 PageHelper 返回空 Page 的假设）
        List<MonthProduct> monthProducts = productsPage.getResult();
        if (CollectionUtils.isEmpty(monthProducts)) {
            return ObjectResponse.success(Page.instance(productsPage.getPages(), productsPage.getTotal(), Collections.emptyList()));
        }
        List<Long> products = monthProducts.stream().map(MonthProduct::getId).distinct().collect(Collectors.toList());
        //获取月卡剩余数量
        Map<Long, Integer> monthInfoMap = getMonthPlateNumCount(products);
        // 根据停车场 ID 列表查询默认收费配置
        List<ParkChargeconfig> parkChargeconfigs = parkChargeconfigDao.selectList(
                Wrappers.lambdaQuery(ParkChargeconfig.class)
                        .eq(ParkChargeconfig::getDefaultCharge, 1)
                        .eq(ParkChargeconfig::getStatus, 0)
                        .in(ParkChargeconfig::getParkId, parkIds)
        );
        // 将收费配置按 parkId 分组，方便查找
        Map<Long, List<ParkChargeconfig>> chargeConfigMap = parkChargeconfigs.stream()
                .collect(Collectors.groupingBy(ParkChargeconfig::getParkId));
        //获取区域信息
        Map<Long, ParkRegionDto> regionDtoMap = parkRegionDao.getParkRegionListByProductId(products)
                .stream().collect(Collectors.toMap(ParkRegionDto::getProductId, Function.identity()));
        // 组装最终结果列表 ParkSpaceMsgInfo
        List<ParkSpaceMsgInfo> msgInfos = monthProducts.stream()
                .map(monthProduct -> {
                    Long parkId = monthProduct.getParkId();
                    ParkSpaceMsgInfo info = new ParkSpaceMsgInfo();
                    ParkSpaceInfo park = parkMap.get(parkId);
                    if (park != null) {
                        info.setId(park.getId());
                        info.setParkCode(park.getParkCode());
                        info.setParkName(park.getParkName());
                        info.setParkProperty(park.getParkProperty());
                        info.setLng(park.getLng());
                        info.setLat(park.getLat());
                        info.setLocation(park.getLocation());
                    }
                    // 设置月卡产品信息
                    MonthProductDto productDto = new MonthProductDto();
                    BeanUtils.copyProperties(monthProduct, productDto);
                    productDto.setRemainMonthCards(calculateRemainMonthCards
                            (monthProduct.getTotalOnlineQuantity(), monthInfoMap.get(monthProduct.getId())));
                    productDto.setRegionName(getRegionName(regionDtoMap, monthProduct.getId()));
                    info.setMonthProduct(productDto);
                    // 设置默认收费类型
                    List<ParkChargeconfig> parkChargeconfigList = chargeConfigMap.get(parkId);
                    if (CollectionUtils.isNotEmpty(parkChargeconfigList)) {
                        info.setSBilltype(parkChargeconfigList.get(0).getBilltype());
                    }
                    return info;
                })
                .collect(Collectors.toList());
        return ObjectResponse.success(Page.instance(productsPage.getPages(), productsPage.getTotal(), msgInfos));
    }

    /**
     * 找到月卡下所有生效中跟待生效的月卡信息
     *
     * @param productIds 月卡ID
     * @return 套餐使用数量
     */
    public Map<Long, Integer> getMonthPlateNumCount(List<Long> productIds) {
        //找到月卡下所有生效中跟待生效的月卡信息
        List<MonthInfo> monthInfos = monthInfoDao.getMonthInfosByProductIds(productIds);
        Map<Long, List<MonthInfo>> listMap = monthInfos.stream().collect(Collectors.groupingBy(MonthInfo::getProductId));
        Map<Long, Integer> map = new HashMap<>();
        listMap.forEach((productId, monthInfoList) -> {
            map.put(productId, monthInfoList.size());
        });
        return map;
    }

    /**
     * 查询当前生效中&待生效的
     *
     * @param productId 月卡ID
     * @return 条数
     */
    public Integer getMonthPlateNumCount(Long productId) {
        //找到月卡下所有生效中跟待生效的月卡信息
        List<MonthInfo> monthInfos = monthInfoDao.getMonthInfosByProductIds(Collections.singletonList(productId));
        return monthInfos.size();
    }

    /**
     * 计算剩余月份卡数量，处理空值和负数
     *
     * @param totalOnlineQuantity 总在线数量
     * @param presaleQuantity     预售数量
     * @return 处理后的剩余月份卡数量
     */
    public Integer calculateRemainMonthCards(Integer totalOnlineQuantity, Integer presaleQuantity) {
        // 处理空值
        if (totalOnlineQuantity == null) {
            totalOnlineQuantity = 0;
        }
        if (presaleQuantity == null) {
            presaleQuantity = 0;
        }
        // 计算剩余月份卡数量
        int remainMonthCards = totalOnlineQuantity - presaleQuantity;
        // 处理负数
        if (remainMonthCards < 0) {
            remainMonthCards = 0;
        }
        return remainMonthCards;
    }

    /**
     * 获取月卡对应的区域
     *
     * @param regionDtoMap 区域集合
     * @param productId    月卡ID
     * @return 区域名称
     */
    private String getRegionName(Map<Long, ParkRegionDto> regionDtoMap, Long productId) {
        ParkRegionDto regionDto = regionDtoMap.get(productId);
        if (Objects.isNull(regionDto)) {
            return "全车场";
        }
        if (regionDto.getRegionId() == 0) {
            return "全车场";
        }
        return regionDto.getRegionNames();
    }

    /**
     * 查询当前策车牌购买月卡的状态
     *
     * @param req 购买信息
     */
    @Override
    public void getMonthStatus(MonthOpenCardDto req) {
        if (req.getPlateNumData().isEmpty()) {
            throw new RuntimeException("未获取到车牌!");
        }
        // 查询当前车牌购买当前月卡记录
        List<MonthInfo> monthInfos = monthInfoDao.getMonthInfosByProductIdAndPlateNum(req.getPlateNumData().get(0).getPlateNum(), req.getProductId());
        if (monthInfos.isEmpty()) {
            return;
        }
        //找到同一个月卡ID的开卡状态
        List<Integer> statusList = monthInfos.stream().map(MonthInfo::getCardStatus).collect(Collectors.toList());
        if (statusList.contains(2)) {
            throw new RuntimeException("当前有一个待生效的套餐不允许续费");
        }
        if (statusList.contains(1)) {
            if (req.getCardOpertype() == 1) {
                throw new RuntimeException("当前有一个生效中的套餐不允许购买");
            }
            return;
        }
        //查找购买月卡数量
        Integer count = this.getMonthPlateNumCount(req.getProductId().longValue());
        MonthProduct monthProduct = monthProductDao.selectById(req.getProductId());
        if (monthProduct.getTotalOnlineQuantity() - count <= 0) {
            throw new RuntimeException("月卡已售罄");
        }
        //检查是否允许欠费开卡
        if (monthProduct.getIsAllowedCardArrears() == 0){
            List<OrderNotpay> list = orderNotpayDao.selectList(Wrappers.lambdaQuery(OrderNotpay.class)
                    .eq(OrderNotpay::getPlateNum,req.getPlateNumData().get(0).getPlateNum())
                    .in(OrderNotpay::getStatus, Arrays.asList(1,4)));
            if (!list.isEmpty()){
                throw new RuntimeException("车辆有欠费，请联系客服人员");
            }
        }
    }

    /**
     * 获取月卡信息
     *
     * @param productId 月卡 ID
     * @return page
     */
    @Override
    public ObjectResponse<Page<ParkSpaceMsgInfo>> getMonthById(Long productId) {
        MonthProduct monthProduct = monthProductDao.selectById(productId);
        if (Objects.isNull(monthProduct)) {
            return ObjectResponse.success(Page.instance(0, 0, Collections.emptyList()));
        }
        Park park = parkDao.selectParkById(monthProduct.getParkId());
        //获取月卡剩余数量
        Map<Long, Integer> monthInfoMap = getMonthPlateNumCount(Collections.singletonList(monthProduct.getId()));
        Map<Long, ParkRegionDto> regionDtoMap = parkRegionDao.getParkRegionListByProductId(
                Collections.singletonList(monthProduct.getId()))
                .stream().collect(Collectors.toMap(ParkRegionDto::getProductId, Function.identity()));
        List<ParkSpaceMsgInfo> msgInfos = new ArrayList<>();
        ParkSpaceMsgInfo info = new ParkSpaceMsgInfo();
        if (park != null) {
            info.setId(park.getId());
            info.setParkCode(park.getParkCode());
            info.setParkName(park.getParkName());
            info.setParkProperty(park.getParkProperty());
            info.setLng(Double.valueOf(park.getLng()));
            info.setLat(Double.valueOf(park.getLat()));
        }
        // 设置月卡产品信息
        MonthProductDto productDto = new MonthProductDto();
        BeanUtils.copyProperties(monthProduct, productDto);
        productDto.setRemainMonthCards(calculateRemainMonthCards
                (monthProduct.getTotalOnlineQuantity(), monthInfoMap.get(monthProduct.getId())));
        productDto.setRegionName(getRegionName(regionDtoMap, monthProduct.getId()));
        info.setMonthProduct(productDto);
        msgInfos.add(info);
        return ObjectResponse.success(Page.instance(1, 1, msgInfos));
    }
}
