package com.icetech.park.service.monthcar;

import cn.hutool.core.collection.CollectionUtil;
import com.google.common.collect.Lists;
import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.cloudcenter.api.lcd.LcdService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.fee.dao.monthcar.MonthInfoDao;
import com.icetech.fee.dao.monthcar.MonthPlateDao;
import com.icetech.fee.dao.monthcar.MonthProductDao;
import com.icetech.fee.dao.monthcar.MonthRegionDao;
import com.icetech.fee.domain.entity.monthcar.MonthPlate;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.dao.OrderSonInfoDao;
import com.icetech.basics.dao.park.ParkConfigDao;
import com.icetech.basics.dao.park.ParkDao;
import com.icetech.fee.domain.entity.monthcar.MonthInfo;
import com.icetech.fee.domain.entity.monthcar.MonthRegion;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderSonInfo;
import com.icetech.park.domain.entity.more.MoreMonthPark;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.domain.response.MonthAbDto;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.park.service.more.MoreMonthParkService;
import com.icetech.park.service.more.MoreMonthPlateService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 月卡服务类
 * @author wangzw
 */
@Slf4j
public class MonthCarServiceBase {
    @Autowired
    protected MonthInfoDao monthInfoDao;
    @Autowired
    protected ParkService parkService;
    @Autowired
    protected ParkDao parkDao;
    @Autowired
    protected MonthPlateDao monthPlateDao;
    @Autowired
    protected MonthProductDao monthProductDao;
    @Autowired
    protected OrderInfoDao orderInfoDao;
    @Autowired
    protected ParkConfigDao parkConfigDao;
    @Autowired
    protected LcdService lcdService;
    @Autowired
    protected OrderSonInfoDao orderSonInfoDao;
    @Autowired
    protected ParkRegionDao regionDao;
    @Autowired
    protected MonthRegionDao monthRegionDao;
    @Autowired
    protected MoreMonthPlateService moreMonthPlateService;
    @Autowired
    private MoreMonthParkService moreMonthParkService;

    //有效期月卡
    public static final Integer VALID_CARD = 1;
    public static final Integer TIMEOUT_CARD = 4;

    public Boolean byExpireMonth(Long parkId, MonthInfo monthInfo){

        //过期多少天按过期卡
        Integer expireMcDays = 0;
        Integer expireDaysMonth = 0;
        ParkConfig parkConfig = parkConfigDao.selectByParkId(parkId);
        if (parkConfig != null){
            expireMcDays = parkConfig.getExpireMcDays();
            expireDaysMonth = parkConfig.getIsExpireMc();
        }
        if (expireMcDays != null && expireMcDays > 0) {
            //过期天数
            int differentDays = DateTools.differentDays(monthInfo.getEndTime(), new Date());
            if (differentDays <= expireMcDays) {
                return expireDaysMonth != null && expireDaysMonth == 1;
            }
        }

        return false;
    }
    /**
     * AB车是否可作为月卡车
     * @param parkId 车场ID
     * @param plateNum 车牌号
     * @param monthId 月卡ID
     * @param plotCount 车位数
     * @param regionId 区域ID
     * @return 是否为月卡车
     */
    protected Boolean isAbCarMonth(Long parkId, String plateNum, long monthId, int plotCount, Long regionId) {
        MonthAbDto monthAbDto = abCar(parkId, plateNum, monthId, plotCount, regionId);
        return monthAbDto.getMonthCar();
    }

    protected MonthAbDto groupAbCar(String plateNum, long monthId, int plotCount){
        List<MonthPlate> monthPlates = monthPlateDao.selectAllByMonthId(monthId);
        MonthInfo monthInfo = monthInfoDao.load(monthId);
        List<String> plateNumList = monthPlates.stream().map(MonthPlate::getPlateNum).collect(Collectors.toList());
        List<MoreMonthPark> moreMonthPlateByPlateNum = moreMonthParkService.getMonthParkByMoreMonthId(monthInfo.getMoreMonthId());
        List<Long> parkIds = moreMonthPlateByPlateNum.stream().map(MoreMonthPark::getParkId).collect(Collectors.toList());
        MonthAbDto monthAbDto = MonthAbDto.builder()
                .plotCount(plotCount)
                .plateNumCount(plateNumList.size()).build();
        if (isAbCarMonth(plotCount, plateNumList)){
            monthAbDto.setAbCar(true);
            log.info("[是否月卡车判断] 车牌号[{}]属于集团多位多车", plateNum);
            List<String> collect = plateNumList.stream()
                    .filter(pn -> !plateNum.equals(pn)).collect(Collectors.toList());
            if (CollectionUtil.isEmpty(collect)){
                monthAbDto.setMonthCar(true);
                monthAbDto.setIndex(1);
                return monthAbDto;
            }
            int inParkMonthCardCount = orderInfoDao.countInMoreParkMonthCardByPlateNums(parkIds, collect);
            monthAbDto.setIndex(inParkMonthCardCount + 1);
            //如果场内月卡车辆数大于等于月卡的车位个数，则按临时车入场
            if (inParkMonthCardCount >= plotCount){
                monthAbDto.setMonthCar(false);
                log.info("[是否月卡车判断] monthId[{}]有[{}]个月卡车在场,车牌号[{}]不能按月卡车处理", monthId, inParkMonthCardCount, plateNum);
                return monthAbDto;
            }else{
                monthAbDto.setMonthCar(true);
                return monthAbDto;
            }
        }else{
            monthAbDto.setAbCar(false);
            monthAbDto.setMonthCar(true);
            monthAbDto.setIndex(1);
            return monthAbDto;
        }

    }
    protected MonthAbDto abCar(Long parkId, String plateNum, long monthId, int plotCount, Long regionId) {
        //为了防止漏判 重新判断一下月卡车类型
        MonthInfo monthInfo = monthInfoDao.load(monthId);
        if (monthInfo.getMoreMonthType() != null && monthInfo.getMoreMonthType() == 1){
            log.info("集团月卡多位多车的判断 {} {}",monthInfo,plateNum);
            return groupAbCar(plateNum, monthId, plotCount);
        }
        List<String> plateNumList = monthPlateDao.selectByMonthId(monthId);
        MonthAbDto monthAbDto = MonthAbDto.builder()
                .plotCount(plotCount)
                .plateNumCount(plateNumList.size()).build();
        //如果车牌个数和车位个数不相等，并且车牌个数小于车位个数时，定义为多位多车
        if (isAbCarMonth(plotCount, plateNumList)){
            monthAbDto.setAbCar(true);
            log.info("[是否月卡车判断] 车牌号[{}]属于多位多车", plateNum);
            List<String> collect = plateNumList.stream()
                    .filter(pn -> !plateNum.equals(pn)).collect(Collectors.toList());
            if (CollectionUtil.isEmpty(collect)){
                monthAbDto.setMonthCar(true);
                monthAbDto.setIndex(1);
                return monthAbDto;
            }
            int inParkMonthCardCount;
            if (regionId != null) {
                ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(parkId);
                ObjectResponse.notError(parkObjectResponse);
                Park park = parkObjectResponse.getData();
                if (park != null && Integer.valueOf(1).equals(park.getIsInterior())) {
                    List<MonthRegion> monthRegions = monthRegionDao.selectByMonthId(monthId);
                    if (CollectionUtils.isNotEmpty(monthRegions)) {
                        MonthRegion parkRegion = monthRegions.stream()
                                .filter(mr -> NumberUtils.toPrimitive(mr.getRegionId()) == 0)
                                .findFirst().orElse(null);
                        if (parkRegion == null) {
                            List<String> inParkPlates = orderSonInfoDao.selectInParkMonthCardByPlateNums(parkId, collect, regionId);
                            log.info("占用车牌号[{}]", inParkPlates);
                            inParkMonthCardCount = inParkPlates.size();
                        } else {
                            inParkMonthCardCount = orderInfoDao.countInParkMonthCardByPlateNums(parkId, collect);
                        }
                    } else {
                        inParkMonthCardCount = orderInfoDao.countInParkMonthCardByPlateNums(parkId, collect);
                    }
                } else {
                    inParkMonthCardCount = orderInfoDao.countInParkMonthCardByPlateNums(parkId, collect);
                }
            } else {
                inParkMonthCardCount = orderInfoDao.countInParkMonthCardByPlateNums(parkId, collect);
            }
            monthAbDto.setIndex(inParkMonthCardCount + 1);
            //如果场内月卡车辆数大于等于月卡的车位个数，则按临时车入场
            if (inParkMonthCardCount >= plotCount){
                monthAbDto.setMonthCar(false);
                log.info("[是否月卡车判断] monthId[{}]有[{}]个月卡车在场,车牌号[{}]不能按月卡车处理", monthId, inParkMonthCardCount, plateNum);
                return monthAbDto;
            }else{
                monthAbDto.setMonthCar(true);
                return monthAbDto;
            }
        }else{
            monthAbDto.setAbCar(false);
            monthAbDto.setMonthCar(true);
            monthAbDto.setIndex(1);
            return monthAbDto;
        }
    }

    public MonthInfo getMonthInfo(Long parkId, String plateNum) {
        //获取有效月卡记录信息
        return monthInfoDao.selectByPlateNum(parkId, plateNum, VALID_CARD);
    }

    /**
     * 多位多车离场处理
     * @param parkId 车场ID
     * @param plateNum 车牌号
     * @param exitTime 离场时间
     */
    public void exitAbMonthDeal(Long parkId, String plateNum, Long exitTime){
        MonthInfo monthInfoDto = getMonthInfo(parkId, plateNum);
        if (monthInfoDto == null) {
            return;
        }
        long monthId = monthInfoDto.getId();
        int plotCount = monthInfoDto.getPlotCount();
        List<String> plateNumList = monthPlateDao.selectByMonthId(monthId);
        if (!isAbCarMonth(plotCount, plateNumList)) {
            return;
        }
        log.info("[多位多车离场处理] plateNum[{}]是AB车", plateNum);
        plateNumList.remove(plateNum);
        OrderInfo recentOne;

        //集团月卡
        if (monthInfoDto.getMoreMonthType()!= null && monthInfoDto.getMoreMonthType()== 1){
            List<MonthPlate> monthPlates = monthPlateDao.selectAllByMonthId(monthId);
            List<MoreMonthPark> moreMonthPlateByPlateNum = moreMonthParkService.getMonthParkByMoreMonthId(monthInfoDto.getMoreMonthId());
            List<Long> parkIds = moreMonthPlateByPlateNum.stream().map(MoreMonthPark::getParkId).collect(Collectors.toList());
            List<String> groupPlateNums = monthPlates.stream().map(MonthPlate::getPlateNum).collect(Collectors.toList());
            //获取集团月卡在场车辆
            List<OrderInfo> orderInfoList = getGroupInParkOrderInfos(parkIds, groupPlateNums, null);
            recentOne = CollectionUtils.isEmpty(orderInfoList) ? null : orderInfoList.get(0);
            if (recentOne == null) {
                log.info("[多位多车离场处理] 在场车辆中不存在与[{}]同一用户的临时车", plateNum);
                return;
            }
        }else {
            List<OrderInfo> orderInfoList = getInParkOrderInfos(parkId, plateNumList, null);
            recentOne = CollectionUtils.isEmpty(orderInfoList) ? null : orderInfoList.get(0);
            if (recentOne == null) {
                log.info("[多位多车离场处理] 在场车辆中不存在与[{}]同一用户的临时车", plateNum);
                return;
            }
        }

        log.info("[多位多车离场处理] 最近入场的场内车辆：{}", recentOne);
        ObjectResponse<ParkConfig> configObjectResponse = parkService.getParkConfig(parkId);
        ObjectResponse.notError(configObjectResponse);
        ParkConfig parkConfig = configObjectResponse.getData();

        /*
         * 判断场内切换时间是否大于0
         */
        long intervalTime = exitTime - recentOne.getEnterTime();
        log.info("[多位多车离场处理] 当前车牌离场时间到上个订单的入场时间间隔是：{}，plateNum：{}", intervalTime, plateNum);
        if (intervalTime <= parkConfig.getSwitchTm() * 60) {
            OrderInfo orderInfoUpdate = new OrderInfo();
            orderInfoUpdate.setOrderNum(recentOne.getOrderNum());
            orderInfoUpdate.setType(PlateTypeEnum.月卡车.getType());
            orderInfoDao.updateByOrderNum(orderInfoUpdate);
            if (Integer.valueOf(1).equals(recentOne.getHasSon())) {
                OrderSonInfo orderSonInfoUpdate = new OrderSonInfo();
                orderSonInfoUpdate.setOrderNum(recentOne.getOrderNum());
                orderSonInfoUpdate.setType(PlateTypeEnum.月卡车.getType());
                Long regionId = recentOne.getRegionId();
                ParkRegion parkRegion = regionDao.selectById(regionId);
                if (parkRegion != null && !Long.valueOf(0).equals(parkRegion.getFatherRelationId())) {
                    regionId = parkRegion.getFatherRelationId();
                }
                orderSonInfoUpdate.setRegionId(regionId);
                orderSonInfoDao.updateByOrderNumAndRegionId(orderSonInfoUpdate);
            }
            log.info("[多位多车离场处理] 从在场记录中查找到[{}]最近入场并且是临时车，未超过切换时间，车辆类型更新为月卡车", plateNum);
        } else {
            if (parkConfig.getSwitchType() == 1) {
                abCarSwitchTimeout(exitTime, recentOne);
            }
        }
    }

    /**
     * AB车切换超时执行
     * @param exitTime 离场时间
     * @param recentOne 最近一次订单
     */
    protected void abCarSwitchTimeout(Long exitTime, OrderInfo recentOne) {
        //switchType解释：支持多车位管理超时处理规则1:B车计费结束时间按A车出场时间收费 2:B车计费结束时间按B车出场时间收费
        OrderInfo orderInfoUpdate = new OrderInfo();
        orderInfoUpdate.setOrderNum(recentOne.getOrderNum());
        orderInfoUpdate.setType(PlateTypeEnum.月卡车.getType());
        orderInfoUpdate.setSwitchTime(exitTime);
        orderInfoDao.updateByOrderNum(orderInfoUpdate);
        if (Integer.valueOf(1).equals(recentOne.getHasSon())) {
            OrderSonInfo orderSonInfoUpdate = new OrderSonInfo();
            orderSonInfoUpdate.setOrderNum(recentOne.getOrderNum());
            orderSonInfoUpdate.setType(PlateTypeEnum.月卡车.getType());
            orderSonInfoUpdate.setSwitchTime(exitTime);
            Long regionId = recentOne.getRegionId();
            ParkRegion parkRegion = regionDao.selectById(regionId);
            if (parkRegion != null && !Long.valueOf(0).equals(parkRegion.getFatherRelationId())) {
                regionId = parkRegion.getFatherRelationId();
            }
            orderSonInfoUpdate.setRegionId(regionId);
            orderSonInfoDao.updateByOrderNumAndRegionId(orderSonInfoUpdate);
        }
        log.info("[多位多车离场处理] 从在场记录中查找到[{}]最近入场并且是临时车，已超过切换时间，车辆类型更新为月卡车，车辆切换时间更新为当前车辆的出场时间[{}]", recentOne.getPlateNum(), exitTime);
    }

    protected List<OrderInfo> getInParkOrderInfos(Long parkId, List<String> plates, Long regionId) {
        OrderInfo orderInfo = new OrderInfo();
        orderInfo.setParkId(parkId);
        orderInfo.setRegionId(regionId);
        orderInfo.setType(PlateTypeEnum.临时车.getType());
        orderInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
        return orderInfoDao.selectListByPlates(orderInfo, plates);
    }

    protected List<OrderInfo> getGroupInParkOrderInfos(List<Long> parkIds, List<String> plates, Long regionId) {
        OrderInfo orderInfo = new OrderInfo();

        orderInfo.setRegionId(regionId);
        orderInfo.setType(PlateTypeEnum.临时车.getType());
        orderInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
        return orderInfoDao.selectListByPlates(orderInfo, parkIds,plates);
    }

    protected List<OrderSonInfo> getInParkOrderSonInfos(Long parkId, List<String> plates, Long regionId) {
        OrderSonInfo orderInfo = new OrderSonInfo();
        orderInfo.setParkId(parkId);
        orderInfo.setRegionId(regionId);
        orderInfo.setType(PlateTypeEnum.临时车.getType());
        orderInfo.setServiceStatus(OrderStatusConstants.IN_PARK);
        return orderSonInfoDao.selectListByPlates(orderInfo, plates);
    }

    /**
     * 判断是否AB车
     * @param plotCount 车位数
     * @param plateNumList 车牌数
     * @return 是否AB车
     */
    protected boolean isAbCarMonth(int plotCount, List<String> plateNumList) {
        if (plateNumList == null) {
            return false;
        }
        int plateCount = plateNumList.size();
        //如果车牌个数和车位个数不相等，并且车牌个数大于车位个数时，定义为多位多车
        return plateCount > plotCount;
    }

}
