package com.icetech.park.service.month;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.icetech.basics.domain.entity.park.BasePark;
import com.icetech.basics.service.park.impl.ParkServiceImpl;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.MoneyTool;
import com.icetech.fee.dao.monthcar.MonthInfoDao;
import com.icetech.fee.dao.monthcar.MonthOrderDao;
import com.icetech.fee.domain.entity.monthcar.MonthInfo;
import com.icetech.fee.domain.entity.monthcar.MonthOrder;
import com.icetech.park.dao.month.MonthDepositOperateRecordDao;
import com.icetech.park.dao.month.MonthDepositRecordDao;
import com.icetech.park.domain.dto.month.MonthDepositOperateRecordDTO;
import com.icetech.park.domain.dto.month.MonthDepositRecordDTO;
import com.icetech.park.domain.entity.month.MonthDepositOperateRecord;
import com.icetech.park.domain.entity.month.MonthDepositRecord;
import com.icetech.park.domain.request.month.DepositOperateParam;
import com.icetech.park.domain.request.month.DepositParam;
import com.icetech.paycenter.api.IPayCenterService;
import com.icetech.paycenter.domain.normalpay.request.RefundRequest;
import com.icetech.paycenter.domain.normalpay.response.RefundResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;

@Slf4j
@Service
public class MonthService {

    @Resource
    private MonthDepositRecordDao monthDepositRecordDao;

    @Resource
    private MonthDepositOperateRecordDao monthDepositOperateRecordDao;

    @Resource
    private ParkServiceImpl parkService;

    @Resource
    private IPayCenterService payCenterService;

    @Resource
    private MonthOrderDao monthOrderDao;

    @Resource
    private MonthInfoDao monthInfoDao;

    /**
     * 月卡押金记录
     *
     * @param param 请求参数
     * @return list
     */
    public ObjectResponse<?> monthDepositRecordList(DepositParam param) {
        Page<MonthDepositRecordDTO> page = new Page<>(param.getPageNo(), param.getPageSize());
        Page<MonthDepositRecordDTO> monthDepositPage = monthDepositRecordDao.getPage(page, param);
        return ObjectResponse.success(monthDepositPage);
    }

    /**
     * 月卡押金记录
     *
     * @param param 请求参数
     * @return list
     */
    public ObjectResponse<?> monthDepositOperateRecordList(DepositParam param) {
        Page<MonthDepositOperateRecordDTO> page = new Page<>(param.getPageNo(), param.getPageSize());
        Page<MonthDepositOperateRecordDTO> monthDepositPage = monthDepositOperateRecordDao.getPage(page, param);
        return ObjectResponse.success(monthDepositPage);
    }

    /**
     * 生成押金退返记录
     *
     * @param id 记录id
     * @return obj
     */
    @Transactional(rollbackFor = {Exception.class, RuntimeException.class})
    public ObjectResponse<?> monthDepositAddRecord(Integer id) {
        try {
            MonthDepositRecord record = monthDepositRecordDao.selectById(id);
            if (record == null) {
                throw new RuntimeException("未获取到月卡记录");
            }
            //判断逻辑
            checkMonthRecord(record);
            //保存逻辑
            saveDepositOperateRecord(record);
            return ObjectResponse.success();
        } catch (RuntimeException e) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getDesc(), e.getMessage());
        }
    }

    /**
     * 生成押金退返记录
     *
     * @param param 审核参数
     * @return obj
     */
    @Transactional(rollbackFor = {Exception.class, RuntimeException.class})
    public ObjectResponse<?> monthDepositAuditOperate(DepositOperateParam param) {
        try {
            MonthDepositOperateRecord operateRecord = monthDepositOperateRecordDao.selectById(param.getId());
            if (operateRecord == null) {
                throw new RuntimeException("未获取到月卡退卡记录");
            }
            MonthDepositRecord record = monthDepositRecordDao.selectById(operateRecord.getDepositId());
            if (record == null) {
                throw new RuntimeException("未获取到月卡记录");
            }
            //审核不通过
            if (param.getAuditResult() == 0) {
                updateDepositOperateRecord(operateRecord, 0, param.getAuditReason(), 0, null);
                return ObjectResponse.success();
            }
            //现金支付直接退款
            if (record.getPayMethod() == 0) {
                updateDepositRecord(record, 1);
                updateDepositOperateRecord(operateRecord, 1, null, param.getAuditResult(), null);
                return ObjectResponse.success();
            }
            //处理平台退款
            return monthRefund(record, operateRecord);
        } catch (RuntimeException e) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getDesc(), e.getMessage());
        }
    }

    /**
     * 校验月卡记录
     *
     * @param record 月卡信息
     */
    private void checkMonthRecord(MonthDepositRecord record) {
        if (record.getDepositStatus() == 1) {
            throw new RuntimeException("押金已退还");
        }
        MonthInfo monthInfo = monthInfoDao.load(record.getMonthId());
        log.info("月卡信息：{}", monthInfo);
        if (monthInfo == null) {
            throw new RuntimeException("未获取到月卡信息");
        }
        if (monthInfo.getCardStatus() == 1) {
            throw new RuntimeException("月卡生效中，暂不支持退费！");
        }
        if (monthInfo.getCardStatus() == 2) {
            throw new RuntimeException("月卡待生效，暂不支持退费！");
        }
        if (monthInfo.getCardStatus() == 5) {
            throw new RuntimeException("月卡冻结，暂不支持退费！");
        }
        MonthDepositOperateRecord operateRecord = monthDepositOperateRecordDao.selectOne(
                Wrappers.lambdaQuery(MonthDepositOperateRecord.class)
                        .eq(MonthDepositOperateRecord::getDepositId, record.getId())
                        .eq(MonthDepositOperateRecord::getAuditResult, 2)
                        .last("LIMIT 1")
        );
        if (operateRecord != null) {
            throw new RuntimeException("已存在待审核押金退返记录");
        }
    }

    /**
     * 保存审核记录
     *
     * @param record 押金记录
     */
    private void saveDepositOperateRecord(MonthDepositRecord record) {
        MonthDepositOperateRecord operateRecord = new MonthDepositOperateRecord();
        operateRecord.setDepositId(record.getId());
        operateRecord.setCardOpertype(record.getCardOpertype());
        operateRecord.setDepositStatus(2);
        operateRecord.setAuditResult(2);
        operateRecord.setCreateTime(new Date());
        operateRecord.setPayChannel(record.getPayChannel());
        monthDepositOperateRecordDao.insert(operateRecord);
    }


    /**
     * 月卡押金记录
     *
     * @param record        记录
     * @param depositStatus 押金状态
     */
    private void updateDepositRecord(MonthDepositRecord record, Integer depositStatus) {
        record.setDepositStatus(depositStatus);
        record.setDepositTime(new Date());
        monthDepositRecordDao.updateById(record);
    }

    /**
     * 更新押金退返记录
     *
     * @param operateRecord 押金退返记录
     * @param depositStatus 押金退返状态(0-退返失败，1-退返成功，2-退款中）
     * @param auditReason   不通过原因
     * @param auditResult   审核状态
     */
    private void updateDepositOperateRecord(MonthDepositOperateRecord operateRecord, Integer depositStatus,
                                            String auditReason, Integer auditResult, String remark) {
        operateRecord.setDepositStatus(depositStatus);
        operateRecord.setAuditReason(auditReason);
        operateRecord.setAuditResult(auditResult);
        operateRecord.setUpdateTime(new Date());
        operateRecord.setRemark(remark);
        monthDepositOperateRecordDao.updateById(operateRecord);
    }

    /**
     * 退还押金
     *
     * @param record        退还记录
     * @param operateRecord 押金记录
     * @return obj
     */
    private ObjectResponse<?> monthRefund(MonthDepositRecord record, MonthDepositOperateRecord operateRecord) {
        BasePark park = parkService.getById(record.getParkId());
        String newTradeNo = "R" + CodeTools.GenerateTradeNo();
        MonthOrder order = monthOrderDao.load(record.getOrderId());
        log.info("开始处理单笔退款 | tradeNo:{} | newTradeNo:{} | refundPrice:{}",
                order.getTradeNo(), newTradeNo, record.getDeposit());
        ObjectResponse<RefundResponse> response = callMonthRefundApi(park.getParkCode(),
                order, newTradeNo, record.getDeposit());
        log.info("退款返回信息：{}", response);
        if (!ObjectResponse.isSuccess(response) || response.getData() == null) {
            // 校验退款响应
            String msg = response.getData() == null ? "退款失败" : response.getMsg();
            updateDepositRecord(record, 2);
            updateDepositOperateRecord(operateRecord, 0, null, 1, msg);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getDesc(), msg);
        }
        if (response.getData().getRefundStatus() == 4) {
            updateDepositRecord(record, 2);
            updateDepositOperateRecord(operateRecord, 0, null, 1, response.getMsg());
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getDesc(), response.getMsg());
        }
        if (response.getData().getRefundStatus() == 3) {
            updateDepositRecord(record, 1);
            updateDepositOperateRecord(operateRecord, 1, null, 1, response.getMsg());
            return ObjectResponse.success();
        }
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getDesc(), "退款失败请联系管理员");
    }

    /**
     * 押金退款
     *
     * @param parkCode    车场
     * @param order       订单
     * @param tradeNo     退款流水号(系统生成)
     * @param refundPrice 退款金额
     * @return obj
     */
    private ObjectResponse<RefundResponse> callMonthRefundApi(String parkCode, MonthOrder order, String tradeNo,
                                                              BigDecimal refundPrice) {
        RefundRequest request = new RefundRequest();
        request.setParkCode(parkCode);
        request.setOutTradeNo(order.getOutTradeNo());
        request.setTradeNo(order.getTradeNo());
        request.setRefundTradeNo(tradeNo);
        request.setPrice(String.valueOf(MoneyTool.fromYuanToFen(String.valueOf(refundPrice))));
        request.setOrderNote("月卡押金退还");
        return payCenterService.refund(request);
    }

}
