package com.icetech.park.service.ledsound;

import com.icetech.basics.config.GrayProperties;
import com.icetech.cloudcenter.api.lcd.LedService;
import com.icetech.cloudcenter.domain.response.LedShowDto;
import com.icetech.cloudcenter.domain.response.LedSoundDto;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.dao.ledsound.LedConfigDao;
import com.icetech.park.dao.ledsound.LedShowDao;
import com.icetech.park.dao.ledsound.LedSoundDao;
import com.icetech.park.dao.ledsound.LedTipsDao;
import com.icetech.park.domain.entity.led.LedConfig;
import com.icetech.park.domain.entity.led.LedShow;
import com.icetech.park.domain.entity.led.LedSound;
import com.icetech.park.domain.entity.led.LedTips;
import com.icetech.redis.handle.RedisHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Nullable;
import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
@Slf4j
public class LedServiceImpl implements LedService {
    @Autowired
    private LedShowDao ledShowDao;
    @Autowired
    private LedSoundDao ledSoundDao;
    @Autowired
    private LedConfigDao ledConfigDao;
    @Autowired
    private LedTipsDao ledTipsDao;
    @Resource
    private RedisHandle redisHandle;
    @Autowired
    private GrayProperties grayProperties;

    @Override
    public ObjectResponse<List<LedShowDto>> getLedShowDtoByChannel(Long channelId) {
        Map<String, List<LedShow>> configMap = getLedShowMapByChannel(channelId);
        Map<String, List<LedShow>> defaultMap = getLedShowMapByChannel(null);
        List<LedShowDto> ledShowDtoList = new ArrayList<>(LedShow.DisplayTypeEnum.values().length);

        for (LedShow.DisplayTypeEnum typeEnum : LedShow.DisplayTypeEnum.values()) {
            List<LedShow> ledShows = configMap.get(String.valueOf(typeEnum.type));
            if (CollectionUtils.isEmpty(ledShows)) {
                ledShows = defaultMap.get(String.valueOf(typeEnum.type));
            }
            if (CollectionUtils.isNotEmpty(ledShows)) {
                LedShowDto ledShowDto = mergeLedContent(ledShows, typeEnum.type);
                ledShowDtoList.add(ledShowDto);
            }
        }
        return ObjectResponse.success(ledShowDtoList);
    }

    @Override
    public ObjectResponse<LedShowDto> getLedShowByType(Long channelId, int type) {
        List<LedShow> ledShows = selectLedConfigByType(channelId, type);
        if (ledShows == null || ledShows.size() == 0){
            ledShows = selectLedConfigByType(null, type);
        }
        LedShowDto ledShowDto = mergeLedContent(ledShows, type);
        return ObjectResponse.success(ledShowDto);
    }

    @Override
    public Map<Integer, LedShowDto> getLedShowDtoMapByChannel(Long channelId) {
        Map<String, List<LedShow>> typeMap = getLedShowMapByChannel(channelId);
        if (typeMap == Collections.EMPTY_MAP) typeMap = new LinkedHashMap<>();
        Map<String, List<LedShow>> defaultTypeMap = getLedShowMapByChannel(null);
        Map<Integer, LedShowDto> dtoMap = new HashMap<>(defaultTypeMap.size());
        for (Map.Entry<String, List<LedShow>> defaultTypeEntry : defaultTypeMap.entrySet()) {
            List<LedShow> ledShows = typeMap.get(defaultTypeEntry.getKey());
            if (CollectionUtils.isEmpty(ledShows)) {
                ledShows = defaultTypeEntry.getValue();
            }
            if (CollectionUtils.isNotEmpty(ledShows)) {
                int type = Integer.parseInt(defaultTypeEntry.getKey());
                dtoMap.put(type, mergeLedContent(ledShows, type));
            }
        }
        return dtoMap;
    }

    @Override
    public ObjectResponse<List<LedSoundDto>> getLedSoundDtoByChannel(Long channelId) {
        Map<Integer, LedSoundDto> soundDtoMap = getLedSoundDtoMapByChannel(channelId);
        List<LedSoundDto> ledSoundDtoList = Stream.of(LedSound.SoundTypeEnum.values()).map(typeEnum -> soundDtoMap.get(typeEnum.type)).collect(Collectors.toList());
        return ObjectResponse.success(ledSoundDtoList);
    }

    @Override
    public ObjectResponse<LedSoundDto> getLedSoundDtoByType(Long channelId, int type) {
        LedSound ledSound = selectLedSoundByType(channelId, type);
        if (ledSound == null){
            ledSound = selectLedSoundByType(null, type);
        }
        LedSoundDto ledSoundDto = mergeSoundContent(ledSound, type);
        return ObjectResponse.success(ledSoundDto);
    }

    @Override
    public Map<Integer, LedSoundDto> getLedSoundDtoMapByChannel(Long channelId) {
        Map<String, LedSound> typeMap = getLedSoundMapByChannel(channelId);
        Map<String, LedSound> defaultTypeMap = getLedSoundMapByChannel(null);
        Map<Integer, LedSoundDto> dtoMap = new HashMap<>(defaultTypeMap.size());
        for (Map.Entry<String, LedSound> defaultSoundEntry : defaultTypeMap.entrySet()) {
            LedSound sound = typeMap.get(defaultSoundEntry.getKey());
            if (sound == null) sound = defaultSoundEntry.getValue();
            int type = Integer.parseInt(defaultSoundEntry.getKey());
            LedSoundDto dto = mergeSoundContent(sound, type);
            dtoMap.put(type, dto);
        }

        return dtoMap;
    }

    @Override
    public ObjectResponse<LedConfig> getLedConfigByChannel(Long channelId) {
        LedConfig ledConfig = selectLedConfig(channelId);
        if (ledConfig == null){
            ledConfig = selectLedConfig(null);
        }
        if (ledConfig != null){
            return ObjectResponse.success(ledConfig);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<LedTips> getLedTipsByChannel(Long channelId) {
        LedTips ledTips = cacheLedTipsByChannel(channelId);
        if (ledTips == null){
            ledTips = cacheLedTipsByChannel(null);
        }
        if (ledTips != null){
            return ObjectResponse.success(ledTips);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    private LedTips cacheLedTipsByChannel(Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LED_TIPS_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LED_TIPS_CHANNEL + channelId;
        if (grayProperties.isCacheEnable()) cacheKey = RedisKeyConstants.KEY_PREFIX_GRAY + cacheKey;
        return redisHandle.cacheObject(cacheKey, LedTips.class, () -> {
            if(channelId == null) {
                return ledTipsDao.selectDefault();
            } else {
                return ledTipsDao.selectByChannelId(channelId);
            }
        }, RedisKeyConstants.EXPIRE_LED_TIPS);
    }

    /**
     * 合并屏显内容
     * @param ledShows
     * @param type
     * @return
     */
    public LedShowDto mergeLedContent(List<LedShow> ledShows, int type){
        LedShowDto ledShowDto = new LedShowDto();
        ledShowDto.setDisplayType(type);

        int lines = ledShows.size();
        String[] lineContentArr = new String[lines];
        if (!ledShows.isEmpty()){
            //当前显示类型的每行内容
            String content = "";
            //当前显示类型的每行屏显颜色
            String ledColor = "";
            for (int l = 0;l < lines;l++){
                LedShow ledShow = ledShows.get(l);

                ledColor += "/" + ledShow.getLedColor();
                String lineContent = transferFormat(ledShow.getCustomContent(), ledShow.getDynamicContent());
                lineContentArr[l] = lineContent;
            }
            for (String lc : lineContentArr){
                content += "/" + (StringUtils.isNotBlank(lc) ? lc : "");
            }
            if (content.startsWith("/")){
                ledShowDto.setContent(content.substring(1));
            }else{
                ledShowDto.setContent(content);
            }
            ledShowDto.setLedColor(ledColor.substring(1));
        }
        return ledShowDto;
    }

    /**
     * 合并播报内容
     * @param ledSound
     * @param type
     * @return
     */
    public LedSoundDto mergeSoundContent(LedSound ledSound, int type){
        LedSoundDto ledSoundDto = new LedSoundDto();
        ledSoundDto.setSoundType(type);

        if (ledSound == null) {
            ledSoundDto.setContent("");
            return ledSoundDto;
        }

        String lineContent = transferFormat(ledSound.getCustomContent(), ledSound.getDynamicContent());
        ledSoundDto.setContent(lineContent.trim());
        return ledSoundDto;
    }

    public List<LedShow> selectLedConfigByType(@Nullable Long channelId, int type) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LED_SHOW_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LED_SHOW_CHANNEL + channelId;
        if (grayProperties.isCacheEnable()) cacheKey = RedisKeyConstants.KEY_PREFIX_GRAY + cacheKey;
        return redisHandle.getListFromMap(cacheKey, String.valueOf(type), LedShow.class, cacheLedShowMap(channelId), RedisKeyConstants.EXPIRE_LED_SHOW);
    }

    @Override
    public Map<String, List<LedShow>> getLedShowMapByChannel(@Nullable Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LED_SHOW_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LED_SHOW_CHANNEL + channelId;
        if (grayProperties.isCacheEnable()) cacheKey = RedisKeyConstants.KEY_PREFIX_GRAY + cacheKey;
        return redisHandle.cacheListMap(cacheKey, LedShow.class, cacheLedShowMap(channelId), RedisKeyConstants.EXPIRE_LED_SHOW);
    }

    private Supplier<Map<String, List<LedShow>>> cacheLedShowMap(@Nullable Long channelId) {
        return () -> {
            List<LedShow> ledShows = null;
            if (channelId != null)
                ledShows = ledShowDao.selectByChannelId(channelId);
            if (CollectionUtils.isEmpty(ledShows))
                ledShows = ledShowDao.selectParkDefault();
            Map<String, List<LedShow>> typeMap = ledShows.stream().collect(Collectors.groupingBy(led -> led.getDisplayType().toString()));
            for (LedShow.DisplayTypeEnum type : LedShow.DisplayTypeEnum.values()) {
                typeMap.putIfAbsent(String.valueOf(type.type), null);
            }
            return typeMap;
        };
    }

    public LedConfig selectLedConfig(@Nullable Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LED_CONFIG_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LED_CONFIG_CHANNEL + channelId;
        if (grayProperties.isCacheEnable()) cacheKey = RedisKeyConstants.KEY_PREFIX_GRAY + cacheKey;
        return redisHandle.cacheObject(cacheKey, LedConfig.class, () -> {
            if (channelId == null) {
                return ledConfigDao.selectDefault();
            } else {
                return ledConfigDao.selectByChannelId(channelId);
            }
        }, RedisKeyConstants.EXPIRE_LED_CONFIG);
    }

    public LedSound selectLedSoundByType(@Nullable Long channelId, int type) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LED_SOUND_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LED_SOUND_CHANNEL + channelId;
        if (grayProperties.isCacheEnable()) cacheKey = RedisKeyConstants.KEY_PREFIX_GRAY + cacheKey;
        return redisHandle.getObjectFromMap(cacheKey, String.valueOf(type), LedSound.class, cacheChannelSoundMap(channelId), RedisKeyConstants.EXPIRE_LED_SOUND);
    }

    @Override
    public Map<String, LedSound> getLedSoundMapByChannel(@Nullable Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LED_SOUND_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LED_SOUND_CHANNEL + channelId;
        if (grayProperties.isCacheEnable()) cacheKey = RedisKeyConstants.KEY_PREFIX_GRAY + cacheKey;
        return redisHandle.cacheObjectMap(cacheKey, LedSound.class, cacheChannelSoundMap(channelId), RedisKeyConstants.EXPIRE_LED_SOUND);
    }

    private Supplier<Map<String, LedSound>> cacheChannelSoundMap(@Nullable Long channelId) {
        return () -> {
            List<LedSound> ledSounds = null;
            if (channelId != null)
                ledSounds = ledSoundDao.selectByChannelId(channelId);
            if (CollectionUtils.isEmpty(ledSounds))
                ledSounds = ledSoundDao.selectParkDefault();
            Map<String, LedSound> typeMap = ledSounds.stream().collect(Collectors.toMap(sound -> sound.getSoundType().toString(), Function.identity(), (old, newer) -> newer));
            for (LedSound.SoundTypeEnum type : LedSound.SoundTypeEnum.values()) {
                typeMap.putIfAbsent(String.valueOf(type.type), null);
            }
            return typeMap;
        };
    }

    /**
     * 转换格式为：{变量类型} 一路顺风
     * @param customContent
     * @param dynamicContent
     * @return
     */
    private String transferFormat(String customContent, String dynamicContent) {
        customContent = customContent == null ? "" : customContent.trim();
        dynamicContent = dynamicContent == null ? "" : dynamicContent.trim();
        if ((customContent + dynamicContent).length() == 0) {
            return "";
        }
        //计算当前行一共有几个模块
        int modulesLength = (customContent.length() == 0 ? dynamicContent :  customContent +"+" + dynamicContent).split("\\+").length;
        //创建存放各模块的数组，方便拼接
        String[] lineModulesArr = new String[modulesLength];
        /*
         * 当前行定制内容解析
         */
        if (StringUtils.isNotBlank(customContent)) {
            String[] split1Arr = customContent.split("\\+");
            /*
             * 处理单行的每一个模块
             */
            for (int i = 0; i < split1Arr.length; i++) {
                String part = split1Arr[i];
                String[] partArr = part.split("\\_");
                String p1 = partArr[0];
                //第X次出现
                int p2 = Integer.parseInt(partArr[1]);
                lineModulesArr[p2 - 1] = p1;
            }
        }
        /*
         * 当前行动态内容解析
         */
        if (StringUtils.isNotBlank(dynamicContent)) {
            String[] split1Arr = dynamicContent.split("\\+");
            /*
             * 处理单行的每一个模块
             */
            for (int i = 0; i < split1Arr.length; i++) {
                String part = split1Arr[i];
                String[] partArr = part.split("\\_");
                String p1 = partArr[0];
                //第X次出现
                int p2 = Integer.parseInt(partArr[1]);
                lineModulesArr[p2 - 1] = "{" + p1 + "}";
            }
        }
        //转换的结果，格式：静态 {动态类型}
        String content = "";
        for (String lineModules : lineModulesArr) {
            content += lineModules + " ";
        }
        return content.trim();
    }
}
