package com.icetech.park.service.lcd;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.Resource;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.icetech.cloudcenter.api.lcd.LcdService;
import com.icetech.park.dao.lcd.LcdBrightnessDao;
import com.icetech.park.dao.lcd.LcdConfigDao;
import com.icetech.park.dao.lcd.LcdShowDao;
import com.icetech.park.dao.lcd.LcdSoundDao;
import com.icetech.park.dao.lcd.LcdSoundcodeDao;
import com.icetech.park.dao.lcd.LcdTipsDao;
import com.icetech.park.domain.entity.lcd.LcdBrightness;
import com.icetech.park.domain.entity.lcd.LcdConfig;
import com.icetech.park.domain.entity.lcd.LcdShow;
import com.icetech.park.domain.entity.lcd.LcdSound;
import com.icetech.park.domain.entity.lcd.LcdSoundcode;
import com.icetech.park.domain.entity.lcd.LcdTips;
import com.icetech.cloudcenter.domain.response.LcdDto;
import com.icetech.cloudcenter.domain.response.LedSoundDto;
import com.icetech.redis.handle.RedisHandle;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.StringUtils;

import cn.hutool.core.collection.CollectionUtil;
import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
public class LcdServiceImpl implements LcdService {

    @Autowired
    private LcdShowDao lcdShowDao;
    @Autowired
    private LcdSoundDao lcdSoundDao;
    @Autowired
    private LcdConfigDao lcdConfigDao;
    @Autowired
    private LcdTipsDao lcdTipsDao;
    @Autowired
    private LcdBrightnessDao lcdBrightnessDao;
    @Autowired
    private LcdSoundcodeDao lcdSoundcodeDao;
    @Resource
    private RedisHandle redisHandle;

    @Override
    public ObjectResponse<List<LcdDto>> getLcdShowByChannel(Long channelId) {
        List<LcdDto> lcdDtoList = Stream.of(LcdShow.DisplayTypeEnum.values())
                .map(type -> getLcdShowByType(channelId, type.type).getData())
                .collect(Collectors.toList());
        return ObjectResponse.success(lcdDtoList);
    }

    @Override
    public ObjectResponse<LcdDto> getLcdShowByType(Long channelId, int type) {
        List<LcdShow> lcds = selectLcdShowByType(channelId, type);
        if (lcds == null || lcds.size() == 0){
            lcds = selectLcdShowByType(null, type);
        }
        LcdDto LcdDto = mergeLedContent(lcds, type);
        return ObjectResponse.success(LcdDto);
    }

    @Override
    public Map<Integer, LcdDto> getLcdShowDtoMapByChannel(Long channelId) {
        Map<String, List<LcdShow>> typeMap = getLcdShowMapByChannel(channelId);
        if (typeMap == Collections.EMPTY_MAP) typeMap = new LinkedHashMap<>();
        Map<String, List<LcdShow>> defaultTypeMap = getLcdShowMapByChannel(null);
        Map<Integer, LcdDto> dtoMap = new HashMap<>(defaultTypeMap.size());
        for (Map.Entry<String, List<LcdShow>> defaultTypeEntry : defaultTypeMap.entrySet()) {
            List<LcdShow> lcdShows = typeMap.get(defaultTypeEntry.getKey());
            if (CollectionUtils.isEmpty(lcdShows)) {
                lcdShows = defaultTypeEntry.getValue();
            }
            if (CollectionUtils.isNotEmpty(lcdShows)) {
                int type = Integer.parseInt(defaultTypeEntry.getKey());
                dtoMap.put(type, mergeLedContent(lcdShows, type));
            }
        }
        return dtoMap;
    }

    public List<LcdShow> selectLcdShowByType(@Nullable Long channelId, int type) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LCD_SHOW_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LCD_SHOW_CHANNEL + channelId;
        return redisHandle.getListFromMap(cacheKey, String.valueOf(type), LcdShow.class, cacheLcdShowMap(channelId), RedisKeyConstants.EXPIRE_LCD_SHOW);
    }

    @Override
    public Map<String, List<LcdShow>> getLcdShowMapByChannel(@Nullable Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LCD_SHOW_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LCD_SHOW_CHANNEL + channelId;
        return redisHandle.cacheListMap(cacheKey, LcdShow.class, cacheLcdShowMap(channelId), RedisKeyConstants.EXPIRE_LCD_SHOW);
    }

    private Supplier<Map<String, List<LcdShow>>> cacheLcdShowMap(@Nullable Long channelId) {
        return () -> {
            List<LcdShow> lcdShows = null;
            if (channelId != null)
                lcdShows = lcdShowDao.selectByChannelId(channelId);
            if (CollectionUtils.isEmpty(lcdShows))
                lcdShows = lcdShowDao.selectParkDefault();
            Map<String, List<LcdShow>> typeMap = lcdShows.stream().collect(Collectors.groupingBy(show -> show.getDisplayType().toString()));
            for (LcdShow.DisplayTypeEnum type : LcdShow.DisplayTypeEnum.values()) {
                typeMap.putIfAbsent(String.valueOf(type.type), null);
            }
            return typeMap;
        };
    }

    @Override
    public ObjectResponse<List<LedSoundDto>> getSoundConfigByChannel(Long channelId) {
        List<LcdSound> soundList = getLcdSoundsByChannel(channelId);
        if (soundList == null || soundList.size() == 0) {
            soundList = getLcdSoundsByChannel(null);
        }
        List<LedSoundDto> ledSoundDtoList = soundList.stream().map(sound -> {
            LedSoundDto ledSoundDto = new LedSoundDto();
            ledSoundDto.setSoundType(sound.getSoundType());
            ledSoundDto.setContent(sound.getContentPattern().replaceAll("\\+", " "));
            return ledSoundDto;
        }).collect(Collectors.toList());
        return ObjectResponse.success(ledSoundDtoList);
    }

    @Override
    public ObjectResponse<LedSoundDto> getSoundConfigByType(Long channelId, int type) {
        List<LcdSound> sounds = selectLcdSoundByType(channelId, type);
        if (CollectionUtils.isEmpty(sounds)) {
            sounds = selectLcdSoundByType(null, type);
        }
        if (CollectionUtils.isEmpty(sounds)) {
            LedSoundDto ledSoundDto = new LedSoundDto();
            ledSoundDto.setSoundType(type);
            ledSoundDto.setContent("");
            return ObjectResponse.success(ledSoundDto);
        } else {
            LcdSound sound = sounds.get(0);
            LedSoundDto ledSoundDto = new LedSoundDto();
            ledSoundDto.setSoundType(sound.getSoundType());
            ledSoundDto.setContent(sound.getContentPattern().replaceAll("\\+", " "));
            return ObjectResponse.success(ledSoundDto);
        }
    }

    @Override
    public Map<Integer, LedSoundDto> getSoundDtoMapByChannel(Long channelId) {
        Map<String, List<LcdSound>> typeMap = getLcdSoundMapByChannel(channelId);
        Map<String, List<LcdSound>> defaultTypeMap = getLcdSoundMapByChannel(null);
        Map<Integer, LedSoundDto> dtoMap = new HashMap<>(defaultTypeMap.size());
        for (Map.Entry<String, List<LcdSound>> defaultSoundEntry : defaultTypeMap.entrySet()) {
            List<LcdSound> sounds = typeMap.get(defaultSoundEntry.getKey());
            if (CollectionUtils.isEmpty(sounds)) sounds = defaultSoundEntry.getValue();
            LcdSound sound = CollectionUtil.get(sounds, 0);

            if (sound != null) {
                LedSoundDto ledSoundDto = new LedSoundDto();
                ledSoundDto.setSoundType(sound.getSoundType());
                ledSoundDto.setContent(sound.getContentPattern().replaceAll("\\+", " "));
                int type = Integer.parseInt(defaultSoundEntry.getKey());
                dtoMap.put(type, ledSoundDto);
            }
        }
        return dtoMap;
    }

    @Override
    public ObjectResponse<LcdConfig> getLcdConfigByChannel(Long channelId) {
        LcdConfig lcdConfig = selectChannelLcdConfig(channelId);
        if (lcdConfig == null) {
            lcdConfig = selectChannelLcdConfig(null);
        }
        if (lcdConfig == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        return ObjectResponse.success(lcdConfig);
    }

    @Override
    public ObjectResponse<LcdTips> getLcdTipsByChannel(Long channelId) {
        LcdTips lcdTips = cacheLcdTipsByChannel(channelId);
        if (lcdTips == null) {
            lcdTips = cacheLcdTipsByChannel(null);
        }
        if (lcdTips != null) {
            return ObjectResponse.success(lcdTips);
        } else {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    private LcdTips cacheLcdTipsByChannel(Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LCD_TIPS_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LCD_TIPS_CHANNEL + channelId;
        return redisHandle.cacheObject(cacheKey, LcdTips.class, () -> {
            if(channelId == null) {
                return lcdTipsDao.selectDefault();
            } else {
                return lcdTipsDao.selectByChannelId(channelId);
            }
        }, RedisKeyConstants.EXPIRE_LCD_TIPS);
    }

    @Override
    public ObjectResponse<LcdSoundcode> getLcdSoundCode(String soundCode) {
        Map<String, LcdSoundcode> lcdSoundCodeMap = cacheLcdSoundCodeMap();
        LcdSoundcode lcdSoundcode = lcdSoundCodeMap.get(soundCode);
        return ObjectResponse.returnNotFoundIfNull(lcdSoundcode);
    }

    @Override
    public ObjectResponse<List<LcdSoundcode>> getAllLcdSoundCodes() {
        Map<String, LcdSoundcode> lcdSoundCodeMap = cacheLcdSoundCodeMap();
        if (lcdSoundCodeMap.isEmpty()) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        List<LcdSoundcode> codes = new ArrayList<>(lcdSoundCodeMap.values());
        return ObjectResponse.success(codes);
    }

    private Map<String, LcdSoundcode> cacheLcdSoundCodeMap() {
        String cacheKey = RedisKeyConstants.KEY_LCD_SOUND_CODE_DEFAULT;
        return redisHandle.cacheObjectMap(cacheKey, LcdSoundcode.class, () -> lcdSoundcodeDao.selectAll().stream()
                        .collect(Collectors.toMap(LcdSoundcode::getSoundCode, Function.identity(), (older, newer) -> newer)),
                RedisKeyConstants.EXPIRE_LCD_SOUND_CODE);
    }

    public List<LcdSound> selectLcdSoundByType(@Nullable Long channelId, int type) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LCD_SOUND_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LCD_SOUND_CHANNEL + channelId;
        return redisHandle.getListFromMap(cacheKey, String.valueOf(type), LcdSound.class, cacheLcdSoundMap(channelId), RedisKeyConstants.EXPIRE_LCD_SOUND);
    }

    public List<LcdSound> getLcdSoundsByChannel(@Nullable Long channelId) {
        Map<String, List<LcdSound>> cacheMap = getLcdSoundMapByChannel(channelId);
        return cacheMap.values().stream().flatMap(Collection::stream).collect(Collectors.toList());
    }

    @Override
    public Map<String, List<LcdSound>> getLcdSoundMapByChannel(@Nullable Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LCD_SOUND_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LCD_SOUND_CHANNEL + channelId;
        return redisHandle.cacheListMap(cacheKey, LcdSound.class, cacheLcdSoundMap(channelId), RedisKeyConstants.EXPIRE_LCD_SOUND);
    }

    private Supplier<Map<String, List<LcdSound>>> cacheLcdSoundMap(@Nullable Long channelId) {
        return () -> {
            List<LcdSound> sounds = null;
            if (channelId != null)
                sounds = lcdSoundDao.selectByChannelId(channelId);
            if (CollectionUtils.isEmpty(sounds))
                sounds = lcdSoundDao.selectParkDefault();
            Map<String, List<LcdSound>> typeMap = sounds.stream().collect(Collectors.groupingBy(sound -> sound.getSoundType().toString()));
            for (LcdSound.SoundTypeEnum type : LcdSound.SoundTypeEnum.values()) {
                typeMap.putIfAbsent(String.valueOf(type.type), null);
            }
            return typeMap;
        };
    }

    public LcdConfig selectChannelLcdConfig(Long channelId) {
        String cacheKey = channelId == null ? RedisKeyConstants.KEY_LCD_CONFIG_PARK_DEFAULT : RedisKeyConstants.KEY_PREFIX_LCD_CONFIG_CHANNEL + channelId;
        return redisHandle.cacheObject(cacheKey, LcdConfig.class, () -> {
            LcdConfig lcdConfig = channelId == null ? lcdConfigDao.selectDefault() : lcdConfigDao.selectByChannelId(channelId);
            if (lcdConfig != null) {
                List<LcdBrightness> lcdBrightnesses = lcdBrightnessDao.findByLcdId(lcdConfig.getId());
                if (lcdBrightnesses == null || lcdBrightnesses.size() == 0) {
                    LcdBrightness lcdBrightness = new LcdBrightness();
                    lcdBrightness.setStartTimePoint("00:00");
                    lcdBrightness.setEndTimePoint("23:59");
                    lcdBrightness.setBrightnessVal(5);
                    lcdBrightnesses = new ArrayList<>();
                    lcdBrightnesses.add(lcdBrightness);
                }
                lcdConfig.setLcdBrightnessList(lcdBrightnesses);
            }
            return lcdConfig;
        }, RedisKeyConstants.EXPIRE_LCD_CONFIG);
    }

    /**
     * 合并屏显内容
     * @param lcds
     * @param type
     * @return
     */
    public LcdDto mergeLedContent(List<LcdShow> lcds, int type){
        LcdDto lcdDto = new LcdDto();
        lcdDto.setDisplayType(type);

        int lines = lcds.size();
        String[] lineContentArr = new String[lines];
        if (lcds != null && lcds.size() > 0){
            //当前显示类型的每行内容
            String content = "";
            //当前显示类型的每行屏显颜色
            String lcdColor = "";
            for (int l = 0;l < lines;l++){
                LcdShow lcdShow = lcds.get(l);

                lcdColor += "/" + lcdShow.getRowColor();
                String lineContent = transferFormat(lcdShow.getCustomContent(), lcdShow.getDynamicContent());
                lineContentArr[l] = lineContent;
            }
            for (String lc : lineContentArr){
                content += "/" + (StringUtils.isNotBlank(lc) ? lc : "");
            }
            if (content.startsWith("/")){
                lcdDto.setContent(content.substring(1, content.length()));
            }else{
                lcdDto.setContent(content);
            }
            lcdDto.setLedColor(lcdColor.substring(1, lcdColor.length()));
        }
        return lcdDto;
    }

    /**
     * 转换格式为：{变量类型} 一路顺风
     * @param customContent
     * @param dynamicContent
     * @return
     */
    private String transferFormat(String customContent, String dynamicContent) {
        customContent = customContent == null ? "" : customContent;
        dynamicContent = dynamicContent == null ? "" : dynamicContent;
        //计算当前行一共有几个模块
        int modulesLength = (customContent.length() == 0 ? dynamicContent :  customContent +"+" + dynamicContent).split("\\+").length;
        //创建存放各模块的数组，方便拼接
        String[] lineModulesArr = new String[modulesLength];
        /**
         * 当前行定制内容解析
         */
        if (StringUtils.isNotBlank(customContent)) {
            String[] split1Arr = customContent.split("\\+");
            /**
             * 处理单行的每一个模块
             */
            for (int i = 0; i < split1Arr.length; i++) {
                String part = split1Arr[i];
                String[] partArr = part.split("\\_");
                String p1 = partArr[0];
                //第X次出现
                int p2 = Integer.parseInt(partArr[1]);
                lineModulesArr[p2 - 1] = p1;
            }
        }
        /**
         * 当前行动态内容解析
         */
        if (StringUtils.isNotBlank(dynamicContent)) {
            String[] split1Arr = dynamicContent.split("\\+");
            /**
             * 处理单行的每一个模块
             */
            for (int i = 0; i < split1Arr.length; i++) {
                String part = split1Arr[i];
                String[] partArr = part.split("\\_");
                String p1 = partArr[0];
                //第X次出现
                int p2 = Integer.parseInt(partArr[1]);
                lineModulesArr[p2 - 1] = "{" + p1 + "}";
            }
        }
        //转换的结果，格式：静态 {动态类型}
        String content = "";
        for (String lineModules : lineModulesArr) {
            content += lineModules + " ";
        }
        return content.trim();
    }
}
