package com.icetech.park.service.impl.manage;

import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.request.ApplyNoPlateEnterRequest;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CloseBrakeRequest;
import com.icetech.cloudcenter.domain.request.DataEnterRequest;
import com.icetech.cloudcenter.domain.request.DataExitRequest;
import com.icetech.cloudcenter.domain.request.ModifyFeeRequest;
import com.icetech.cloudcenter.domain.request.NoplateExitRequest;
import com.icetech.cloudcenter.domain.request.OpenBrakeRequest;
import com.icetech.cloudcenter.domain.request.PullfeeRequest;
import com.icetech.cloudcenter.domain.request.QueryFeeRequest;
import com.icetech.cloudcenter.domain.request.QueryOrderFeeRequest;
import com.icetech.cloudcenter.domain.request.RemoteSwitchRequest;
import com.icetech.cloudcenter.domain.request.VoiceReportRequest;
import com.icetech.cloudcenter.domain.request.pnc.ExitPayCautionRequest;
import com.icetech.cloudcenter.domain.response.EnexCarInfoBaseResponse;
import com.icetech.cloudcenter.domain.response.EnterCarInfoResponse;
import com.icetech.cloudcenter.domain.response.PullfeeResponse;
import com.icetech.cloudcenter.domain.vo.UpdateFeeVo;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.AsyncNotifyInterface;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.UUIDTools;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.down.pnc.impl.ChannelDataServiceImpl;
import com.icetech.park.service.down.pnc.impl.ModifyFeeServiceImpl;
import com.icetech.park.service.down.pnc.impl.PncNoplateExitServiceImpl;
import com.icetech.park.service.down.pnc.impl.PncRemoteSwitchServiceImpl;
import com.icetech.park.service.down.pnc.impl.ReqEnterServiceImpl;
import com.icetech.park.service.down.pnc.impl.ShowAndSayServiceImpl;
import com.icetech.park.service.impl.base.ManageServiceBase;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

/**
 * 端网云管理接口实现
 */
@Slf4j
@Service
public class PnCloudManageImpl extends ManageServiceBase implements CloudManageService {
    @Autowired
    private ParkService parkService;
    @Autowired
    private PncNoplateExitServiceImpl pncNoplateExitService;
    @Autowired
    private ModifyFeeServiceImpl modifyFeeServiceImpl;
    @Autowired
    private PncRemoteSwitchServiceImpl pncRemoteSwitchService;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private P2CloudManageImpl p2CloudManage;
    @Autowired
    private ShowAndSayServiceImpl showAndSayService;
    @Autowired
    private ChannelDataServiceImpl channelDataService;
    @Autowired
    private ReqEnterServiceImpl reqEnterService;

    @Override
    public ObjectResponse<EnexCarInfoBaseResponse> getCarInfoFromRedis(String parkCode, String aisleCode, Integer type, String topic) {
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        //验证通道下是否存在相机
        ObjectResponse<List<ParkDevice>> deviceByChannel = parkDeviceService.getDeviceByChannel(park.getId(), aisleCode, 1);
        if (!ObjectResponse.isSuccess(deviceByChannel)){
            return ObjectResponse.failed(CodeConstants.ERROR_3003,"该通道尚未添加识别相机");
        }

        ObjectResponse<Void> objectResponse = channelDataService.execute(aisleCode, parkCode, park.getKey(), park.getId(), 1, topic, type);;
        return ObjectResponse.failed(objectResponse.getCode(), objectResponse.getMsg());
    }

    @Override
    public ObjectResponse<EnterCarInfoResponse> getCarEnterTraceFromRedis(String parkCode, String aisleCode) {
        CarEnterRequest enterTrace = cacheHandle.getEntrance(parkCode, aisleCode);
        if (Objects.isNull(enterTrace)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        return ObjectResponse.success(getEntranceCarInfoResponse(enterTrace));
    }

    @Override
    public ObjectResponse<EnexCarInfoBaseResponse> getCarInfoFromCamera(String parkCode, String aisleCode, Integer type, String topic) {
        log.info("[软触发调用] parkcode[{}], aisleCode[{}]", parkCode, aisleCode);
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        ObjectResponse<Void> objectResponse = channelDataService.execute(aisleCode, parkCode, park.getKey(), park.getId(), 2, topic, type);
        return ObjectResponse.instance(objectResponse.getCode(), objectResponse.getMsg(), null);
    }

    @Override
    public ObjectResponse<String> getCarInfoFromCameraAsync(String parkCode, String version, String sn,String aisleCode, Integer type) {
        log.info("[软触发调用] parkcode[{}], aisleCode[{}]", parkCode, aisleCode);
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        ObjectResponse<Void> objectResponse = channelDataService.execute(aisleCode, parkCode, park.getKey(), park.getId(), 2, AsyncNotifyInterface.getTopic(), type);
        if (objectResponse.getCode().equals(CodeConstants.ERROR_12002)) {
            return ObjectResponse.success(objectResponse.getMsg());
        }
        return ObjectResponse.failed();
    }

    @Override
    public ObjectResponse<Object> getCarExitInfoFromCacheByOrderNum(String parkCode, String orderNum) {
        ObjectResponse<OrderInfo> infoObjectResponse = orderService.findByOrderNum(orderNum);
        ObjectResponse.notError(infoObjectResponse);
        String key = RedisConstants.PNC_EXIT_PAY_ORDER_PROFILE + infoObjectResponse.getData().getParkId() + "_" + infoObjectResponse.getData().getLocalOrderNum();
        //如果是按场内预缴费流程支付，且有出口缴费信号时，则下发抬杆指令
        ExitPayCautionRequest exitPayCautionRequest = redisUtils.get(key, ExitPayCautionRequest.class);
        return exitPayCautionRequest == null ? ObjectResponse.failed(CodeConstants.ERROR_404) : ObjectResponse.success(exitPayCautionRequest);
    }

    @Override
    public ObjectResponse<Void> allowEnter(DataEnterRequest enterRequest) {
        return reqEnterService.execute(enterRequest);
    }

    @Override
    public ObjectResponse<Void> allowExit(DataExitRequest exitRequest) {
        String parkCode = exitRequest.getParkCode();
        NoplateExitRequest noplateExitRequest = new NoplateExitRequest();
        noplateExitRequest.setPlateNum(exitRequest.getPlateNum());
        noplateExitRequest.setExitTime(exitRequest.getExitTime().getTime() / 1000);
        noplateExitRequest.setChannelId(exitRequest.getAisleCode());
        noplateExitRequest.setParkCode(parkCode);
        noplateExitRequest.setCarType(exitRequest.getCarType());
        noplateExitRequest.setCarDesc(exitRequest.getSpecialCar());
        noplateExitRequest.setTopic(exitRequest.getTopic());
        noplateExitRequest.setExitWay(exitRequest.getExitWay());
        noplateExitRequest.setOperAccount(exitRequest.getOperAccount());
        noplateExitRequest.setExitTerminal(exitRequest.getExitTerminal());
        return pncNoplateExitService.noplateExit(noplateExitRequest);
    }

    @Override
    public ObjectResponse<PullfeeResponse> pullFee(PullfeeRequest pullfeeRequest) {
        QueryOrderFeeRequest queryOrderFeeRequest = new QueryOrderFeeRequest();
        queryOrderFeeRequest.setOrderNum(pullfeeRequest.getOrderNum());
        queryOrderFeeRequest.setPlateNum(pullfeeRequest.getPlateNum());
        queryOrderFeeRequest.setParkCode(pullfeeRequest.getParkCode());
        queryOrderFeeRequest.setChannelId(pullfeeRequest.getAisleCode());
        queryOrderFeeRequest.setCarType(pullfeeRequest.getCarType());
        queryOrderFeeRequest.setTopic(pullfeeRequest.getTopic());
        queryOrderFeeRequest.setExtraInfo(QueryFeeRequest.ExtraInfoEnum.PULL_FEE.val);
        queryOrderFeeRequest.setWithNotPay(true);

        String parkCode = pullfeeRequest.getParkCode();
        String channelId = pullfeeRequest.getAisleCode();
        if (NumberUtils.toPrimitive(pullfeeRequest.getSource()) == 1) {
            return p2CloudManage.p2cPullFee(pullfeeRequest, queryOrderFeeRequest, parkCode, channelId);
        }
        ObjectResponse<Void> objectResponse = orderService.pncQueryFee(queryOrderFeeRequest);
        ObjectResponse<PullfeeResponse> response = new ObjectResponse<>();
        response.setCode(objectResponse.getCode());
        response.setMsg(objectResponse.getMsg());
        response.setTraceId(objectResponse.getTraceId());
        return response;
    }

    @Override
    public ObjectResponse<Void> callVoiceReport(VoiceReportRequest voiceReportRequest) {
        return showAndSayService.execute(voiceReportRequest);
    }

    @Override
    public ObjectResponse<Void> requestOpenBrake(OpenBrakeRequest openBrakeRequest) {
        RemoteSwitchRequest remoteSwitchRequest = new RemoteSwitchRequest();
        remoteSwitchRequest.setParkCode(openBrakeRequest.getParkCode());
        remoteSwitchRequest.setChannelId(openBrakeRequest.getAisleCode());
        remoteSwitchRequest.setSwitchType(1);
        remoteSwitchRequest.setSequenceId(UUIDTools.getUuid());
        remoteSwitchRequest.setInoutEvent(openBrakeRequest.getExitWay());
        remoteSwitchRequest.setOperAccount(openBrakeRequest.getOperAccount());
        remoteSwitchRequest.setExTerminal(openBrakeRequest.getExitTerminal());
        remoteSwitchRequest.setPlateNum(openBrakeRequest.getPlateNum());
        remoteSwitchRequest.setReasonType(openBrakeRequest.getReasonType());
        remoteSwitchRequest.setRemark(openBrakeRequest.getRemark());
        return pncRemoteSwitchService.remoteSwitch(remoteSwitchRequest, openBrakeRequest.getTopic(),
                RemoteSwitchRequest.ExtraInfo.builder().biz(RemoteSwitchRequest.ExtraInfoEnum.OPEN.val)
                        .requestVO(openBrakeRequest).build());
    }

    @Override
    public ObjectResponse<Void> requestCloseBrake(CloseBrakeRequest closeBrakeRequest) {
        RemoteSwitchRequest remoteSwitchRequest = new RemoteSwitchRequest();
        remoteSwitchRequest.setParkCode(closeBrakeRequest.getParkCode());
        remoteSwitchRequest.setChannelId(closeBrakeRequest.getAisleCode());
        remoteSwitchRequest.setSwitchType(2);
        remoteSwitchRequest.setSequenceId(UUIDTools.getUuid());
        return pncRemoteSwitchService.remoteSwitch(remoteSwitchRequest, closeBrakeRequest.getTopic(),
                RemoteSwitchRequest.ExtraInfo.builder().biz(RemoteSwitchRequest.ExtraInfoEnum.CLOSE.val)
                        .requestVO(closeBrakeRequest).build());
    }

    @Override
    public ObjectResponse<Void> modifyCacheFee(ModifyFeeRequest modifyFeeRequest) {
            ObjectResponse<Park> byParkCode = parkService.findByParkCode(modifyFeeRequest.getParkCode());
            if (!ObjectResponse.isSuccess(byParkCode)) {
                return ObjectResponse.failed(CodeConstants.ERROR_404, "车场编号有误");
            }
            UpdateFeeVo updateFeeVo = new UpdateFeeVo();
            updateFeeVo.setParkId(byParkCode.getData().getId());
            updateFeeVo.setChannelId(modifyFeeRequest.getAisleCode());
            updateFeeVo.setOrderNum(modifyFeeRequest.getOrderNum());
            updateFeeVo.setPlateNum(modifyFeeRequest.getPlateNum());
            updateFeeVo.setNewUnpayPrice(modifyFeeRequest.getUpFee());
            updateFeeVo.setTopic(modifyFeeRequest.getTopic());
            return modifyFeeServiceImpl.execute(updateFeeVo);
    }

    @Override
    public ObjectResponse<Void> applyNoPlateEnter(ApplyNoPlateEnterRequest applyNoPlateEnterRequest) {
        //todo
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Void> fleetMode(String parkCode, Long parkId, String sn, Integer enable) {
        return ObjectResponse.failed("1000", "端网云模式不支持车队模式");
    }

}
