package com.icetech.park.service.impl.manage;

import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.api.fee.QueryOrderFeeService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.domain.enumeration.DataCollectionEnum;
import com.icetech.cloudcenter.domain.enumeration.EnexTypeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.enumeration.SwitchTypeEnum;
import com.icetech.cloudcenter.domain.pay.NoplateRecordStatus;
import com.icetech.cloudcenter.domain.pay.NoplateRecordType;
import com.icetech.cloudcenter.domain.request.ApplyNoPlateEnterRequest;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.request.CloseBrakeRequest;
import com.icetech.cloudcenter.domain.request.DataEnterRequest;
import com.icetech.cloudcenter.domain.request.DataExitRequest;
import com.icetech.cloudcenter.domain.request.ModifyFeeRequest;
import com.icetech.cloudcenter.domain.request.OpenBrakeRequest;
import com.icetech.cloudcenter.domain.request.PullfeeRequest;
import com.icetech.cloudcenter.domain.request.QueryFeeRequest;
import com.icetech.cloudcenter.domain.request.QueryOrderFeeRequest;
import com.icetech.cloudcenter.domain.request.VoiceReportRequest;
import com.icetech.cloudcenter.domain.request.p2c.FleetModeRequest;
import com.icetech.cloudcenter.domain.request.p2c.SoftTriggerRequest;
import com.icetech.cloudcenter.domain.response.EnexCarInfoBaseResponse;
import com.icetech.cloudcenter.domain.response.EnterCarInfoResponse;
import com.icetech.cloudcenter.domain.response.PullfeeResponse;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.cloudcenter.domain.response.p2c.RemoteSwitchResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderCarInfoConstant;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.AsyncNotifyInterface;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.DateTools;
import com.icetech.park.dao.catched.NoplateRecordDao;
import com.icetech.park.domain.entity.catched.NoplateRecord;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.down.p2c.impl.FleetModeServiceImpl;
import com.icetech.park.service.flow.p2c.FlowCondition;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 端云管理接口实现
 */
@Slf4j
@Service
public class P2CloudManageImpl extends CloudManageCommon implements CloudManageService {
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;
    @Autowired
    private QueryOrderFeeService queryOrderFeeService;
    @Autowired
    private NoplateRecordDao noplateRecordDao;
    @Autowired
    private FleetModeServiceImpl fleetModeService;

    @Override
    public ObjectResponse<EnexCarInfoBaseResponse> getCarInfoFromRedis(String parkCode, String aisleCode, Integer type, String topic) {
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        Long parkId = park.getId();
        //验证通道下是否存在相机
        ObjectResponse<List<ParkDevice>> deviceByChannel = parkDeviceService.getDeviceByChannel(parkId, aisleCode, 1);
        if (!ObjectResponse.isSuccess(deviceByChannel)) {
            return ObjectResponse.failed(CodeConstants.ERROR_3003, "该通道尚未添加识别相机");
        }

        EnexCarInfoBaseResponse enexCarInfoBaseResponse;
        if (type.equals(EnexTypeEnum.入场.getType())) {
            /*
             * 入场通道车辆信息获取
             */
            CarEnterRequest entrance = cacheHandle.getEntrance(parkCode, aisleCode);
            if (entrance == null) {
                ParkDevice parkDevice = deviceByChannel.getData().get(0);
                ObjectResponse<Void> objectResponse = getCurrChannelInfo(parkCode, aisleCode, parkDevice.getSerialNumber(), parkDevice.getProtocolVer(), type, topic, parkId);
                return ObjectResponse.instance(objectResponse.getCode(), objectResponse.getMsg(), null);
            } else {
                enexCarInfoBaseResponse = getEntranceCarInfoResponse(entrance);
            }
        } else {
            /*
             * 离场通道车辆信息获取
             */
            CarExitRequest exit = cacheHandle.getExit(parkCode, aisleCode);
            if (exit == null) {
                ParkDevice parkDevice = deviceByChannel.getData().get(0);
                ObjectResponse<Void> objectResponse = getCurrChannelInfo(parkCode, aisleCode, parkDevice.getSerialNumber(), parkDevice.getProtocolVer(), type, topic, parkId);
                return ObjectResponse.instance(objectResponse.getCode(), objectResponse.getMsg(), null);
            } else {
                //查询场内订单信息并设置
                enexCarInfoBaseResponse = getExitCarInfoResponse(parkId, parkCode, exit);
            }
        }
        enexCarInfoBaseResponse.setDataCollection(DataCollectionEnum.端云.getType());
        return ObjectResponse.success(enexCarInfoBaseResponse);

    }

    @Override
    public ObjectResponse<EnterCarInfoResponse> getCarEnterTraceFromRedis(String parkCode, String aisleCode) {
        CarEnterRequest enterTrace = cacheHandle.getEntrance(parkCode, aisleCode);
        if (Objects.isNull(enterTrace)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        return ObjectResponse.success(getEntranceCarInfoResponse(enterTrace));
    }

    @Override
    public ObjectResponse<EnexCarInfoBaseResponse> getCarInfoFromCamera(String parkCode, String aisleCode, Integer type, String topic) {
        log.info("[软触发调用] parkCode[{}], aisleCode[{}]", parkCode, aisleCode);
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        Long parkId = park.getId();
        //验证通道下是否存在相机
        ObjectResponse<List<ParkDevice>> deviceByChannel = parkDeviceService.getDeviceByChannel(parkId, aisleCode, 1);
        if (!ObjectResponse.isSuccess(deviceByChannel)) {
            return ObjectResponse.failed(CodeConstants.ERROR_3003, "该通道尚未添加识别相机");
        }
        ParkDevice parkDevice = deviceByChannel.getData().get(0);
        ObjectResponse<Void> objectResponse = getCurrChannelInfo(parkCode, aisleCode, parkDevice.getSerialNumber(), parkDevice.getProtocolVer(), type, topic, parkId);
        return ObjectResponse.instance(objectResponse.getCode(), objectResponse.getMsg(), null);
    }

    @Override
    public ObjectResponse<String> getCarInfoFromCameraAsync(String parkCode, String version, String sn, String aisleCode, Integer type) {
        log.info("[软触发调用] parkCode[{}], aisleCode[{}]", parkCode, aisleCode);
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        Long parkId = park.getId();
        ObjectResponse<String> objectResponse;
        //根据相机版本区分 1.5.5开始使用图片抓拍接口
        if (StringUtils.isNotEmpty(version) && P2cVersionEnum.getIndex(version) >= P2cVersionEnum.版本13.getIndex()
                && cacheHandle.judgeTakePicturePrivilege(sn)) {
            log.info("[调用新版本的图片抓拍接口] {},{},{},{}", parkCode, version, sn, aisleCode);
            objectResponse = takePicturesService.execute(parkId, parkCode, sn);
        } else {
            objectResponse = softTriggerService.executeAsync(parkId, parkCode, aisleCode);
        }
        return objectResponse;
    }

    @Override
    public ObjectResponse<Object> getCarExitInfoFromCacheByOrderNum(String parkCode, String orderNum) {
        CarExitRequest exitRequest = cacheHandle.getExitByOrderNum(parkCode, orderNum);
        return exitRequest == null ? ObjectResponse.failed(CodeConstants.ERROR_404) : ObjectResponse.success(exitRequest);
    }

    @Override
    public ObjectResponse<Void> allowEnter(DataEnterRequest enterRequest) {
        return allowEnter(remoteSwitchService, enterRequest);
    }

    @Override
    public ObjectResponse<Void> allowExit(DataExitRequest exitRequest) {
        String parkCode = exitRequest.getParkCode();
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        Long parkId = park.getId();
        String aisleCode = exitRequest.getAisleCode();
        String plateNum = exitRequest.getPlateNum();
        if (exitRequest.isOffLine()) {
            return exit(softTriggerService, exitRequest, parkCode, aisleCode, parkId, plateNum, exitRequest.getTopic());
        }
        String serialNumber;
        try {
            serialNumber = cacheHandle.getSerialNumber(parkCode, aisleCode);
        } catch (ResponseBodyException e) {
            return ObjectResponse.failed(CodeConstants.ERROR_3003);
        }

        // 开闸
        final String finalSerialNumber = serialNumber;
        asyncExecutor.execute(ThreadUtils.wrapTrace(() -> {
            ObjectResponse<RemoteSwitchResponse> objectResponse =
                    remoteSwitchService.execute(SwitchTypeEnum.开闸.getType(), parkId, exitRequest.getParkCode(),
                            finalSerialNumber, exitRequest.getPlateNum());
            if (ObjectResponse.isSuccess(objectResponse)) {
                if (objectResponse.getData().getResult() != 1) {
                    log.info("[二次下发开闸] 参数[{}]", exitRequest);
                    com.icetech.cloudcenter.domain.request.p2c.RemoteSwitchRequest request =
                            new com.icetech.cloudcenter.domain.request.p2c.RemoteSwitchRequest();
                    request.setSwitchType(SwitchTypeEnum.开闸.getType());
                    request.setSequenceId(RandomUtils.nextInt(100000, 1000000));
                    request.setPlateNum(exitRequest.getPlateNum());
                    long executeTime = DateTools.unixTimestamp();
                    Long cameraExecuteTime = objectResponse.getData().getExecuteTime();
                    if (cameraExecuteTime != null) {
                        if (cameraExecuteTime.toString().length() >= 13) {
                            executeTime = cameraExecuteTime / 1000 + 1;
                        } else {
                            executeTime = cameraExecuteTime + 1;
                        }
                    }
                    request.setTime(executeTime);
                    remoteSwitchService.secondOpen(request, parkId, exitRequest.getParkCode(), finalSerialNumber);
                }
            }
        }));

        // 语音显示
        Map<String, Object> paraMap = new HashMap<>();
        paraMap.put("parkTime", (exitRequest.getExitTime().getTime() - exitRequest.getEnterTime().getTime()) / 1000);
        paraMap.put("orderNum", exitRequest.getOrderNum());
        ObjectResponse<ParkInoutdevice> channelResp = parkService.getInOutDeviceByCode(parkId, aisleCode);
        ParkInoutdevice channel = channelResp.getData();
        downShowSayHandle.showSayExec(parkId, parkCode, channel.getId(), plateNum, exitRequest.getType(), FlowCondition.ResultCode.缴费后屏显播报, paraMap,
                serialNumber, 2, aisleCode, false);
        return exit(softTriggerService, exitRequest, parkCode, aisleCode, parkId, plateNum, exitRequest.getTopic());
    }

    @Override
    public ObjectResponse<PullfeeResponse> pullFee(PullfeeRequest pullfeeRequest) {
        QueryOrderFeeRequest queryOrderFeeRequest = new QueryOrderFeeRequest();
        queryOrderFeeRequest.setOrderNum(pullfeeRequest.getOrderNum());
        queryOrderFeeRequest.setPlateNum(pullfeeRequest.getPlateNum());
        queryOrderFeeRequest.setParkCode(pullfeeRequest.getParkCode());
        queryOrderFeeRequest.setChannelId(pullfeeRequest.getAisleCode());
        queryOrderFeeRequest.setCarType(pullfeeRequest.getCarType());
        queryOrderFeeRequest.setTopic(pullfeeRequest.getTopic());
        queryOrderFeeRequest.setExtraInfo(QueryFeeRequest.ExtraInfoEnum.PULL_FEE.val);
        queryOrderFeeRequest.setWithNotPay(true);

        String parkCode = pullfeeRequest.getParkCode();
        String channelId = pullfeeRequest.getAisleCode();
        return p2cPullFee(pullfeeRequest, queryOrderFeeRequest, parkCode, channelId);
    }

    @Override
    public ObjectResponse<Void> callVoiceReport(VoiceReportRequest voiceReportRequest) {
        //更新欠费订单费用
        ObjectResponse<QueryOrderFeeResponse> objectResponse = queryOrderFeeService.updateAndGetChannelFee(
                voiceReportRequest.getParkCode(), voiceReportRequest.getAisleCode(),
                voiceReportRequest.getPlateNum(), voiceReportRequest.getOrderNums());
        if (!ObjectResponse.isSuccess(objectResponse)) {
            return ObjectResponse.failed(objectResponse.getCode(), objectResponse.getMsg());
        }
        return callVoiceReport(softTriggerService, voiceReportRequest);
    }

    @Override
    public ObjectResponse<Void> requestOpenBrake(OpenBrakeRequest openBrakeRequest) {
        String parkCode = openBrakeRequest.getParkCode();
        String deviceNo;
        try {
            deviceNo = cacheHandle.getSerialNumber(parkCode, openBrakeRequest.getAisleCode());
        } catch (ResponseBodyException e) {
            log.info(e.getMessage());
            return ObjectResponse.failed(e.getErrCode(), e.getMessage());
        }
        //开闸
        return remoteSwitchService.execute(SwitchTypeEnum.开闸.getType(), parkCode, deviceNo,
                openBrakeRequest.getPlateNum(), openBrakeRequest.getTopic(),
                com.icetech.cloudcenter.domain.request.p2c.RemoteSwitchRequest.ExtraInfo.builder()
                        .biz(com.icetech.cloudcenter.domain.request.p2c.RemoteSwitchRequest.ExtraInfoEnum.OPEN.val)
                        .requestVO(openBrakeRequest).build());
    }

    @Override
    public ObjectResponse<Void> requestCloseBrake(CloseBrakeRequest closeBrakeRequest) {
        String parkCode = closeBrakeRequest.getParkCode();
        String deviceNo;
        try {
            deviceNo = cacheHandle.getSerialNumber(parkCode, closeBrakeRequest.getAisleCode());
        } catch (ResponseBodyException e) {
            log.info(e.getMessage());
            return ObjectResponse.failed(e.getErrCode(), e.getMessage());
        }
        return remoteSwitchService.execute(SwitchTypeEnum.关闸.getType(), parkCode, deviceNo, closeBrakeRequest.getPlateNum(),
                closeBrakeRequest.getTopic(),
                com.icetech.cloudcenter.domain.request.p2c.RemoteSwitchRequest.ExtraInfo.builder()
                        .biz(com.icetech.cloudcenter.domain.request.p2c.RemoteSwitchRequest.ExtraInfoEnum.CLOSE.val)
                        .requestVO(closeBrakeRequest).build());
    }

    @Override
    public ObjectResponse<Void> modifyCacheFee(ModifyFeeRequest modifyFeeRequest) {
        QueryOrderFeeResponse channelFee = cacheHandle.getChannelFee(modifyFeeRequest.getParkCode(), modifyFeeRequest.getAisleCode());
        if (channelFee != null) {
            String totalAmount = channelFee.getTotalAmount();
            String unPayPrice = channelFee.getUnpayPrice();
            //差值
            BigDecimal subtract = new BigDecimal(unPayPrice).subtract(new BigDecimal(modifyFeeRequest.getUpFee()));
            channelFee.setTotalAmount(new BigDecimal(totalAmount).subtract(subtract).toString());
            channelFee.setUnpayPrice(modifyFeeRequest.getUpFee());
            channelFee.setSumPrice(null);
            cacheHandle.setChannelFee(modifyFeeRequest.getParkCode(), modifyFeeRequest.getAisleCode(), channelFee);
        } else {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Void> applyNoPlateEnter(ApplyNoPlateEnterRequest applyNoPlateEnterRequest) {
        String parkCode = applyNoPlateEnterRequest.getParkCode();
        String plateNum = applyNoPlateEnterRequest.getPlateNum();
        String channelCode = applyNoPlateEnterRequest.getChannelCode();
        String unionId = applyNoPlateEnterRequest.getUnionId();

        //查询通道权限
        Integer isMaster = 1;
        ObjectResponse<ParkInoutdevice> channelInfo = parkService.getInOutDeviceByCode(applyNoPlateEnterRequest.getParkId(), channelCode);
        if (ObjectResponse.isSuccess(channelInfo)) {
            ParkInoutdevice parkInoutdevice = channelInfo.getData();
            isMaster = parkInoutdevice.getIsMaster();
        }
        DataEnterRequest enterRequest = new DataEnterRequest();
        BeanUtils.copyProperties(applyNoPlateEnterRequest, enterRequest);
        enterRequest.setAisleCode(channelCode);
        //临时车
        enterRequest.setType(PlateTypeEnum.临时车.getType());
        enterRequest.setEnterTime(DateTools.unixTimestamp());
        enterRequest.setTopic(AsyncNotifyInterface.getTopic());
        enterRequest.setEnterWay(OrderCarInfoConstant.IN_OUT_WAY_QR_CODE);

        ObjectResponse<Void> objectResponse = allowEnter(remoteSwitchService, enterRequest);
        if (ObjectResponse.isSuccess(objectResponse)) {
            NoplateRecord noplateRecord = new NoplateRecord();
            noplateRecord.setUnionId(unionId);
            noplateRecord.setType(NoplateRecordType.无牌车.getType());
            noplateRecord.setStatus(NoplateRecordStatus.已入场.getStatus());
            noplateRecord.setParkCode(parkCode);
            noplateRecord.setPlateNum(plateNum);
            noplateRecord.setEnterChannelId(channelCode);
            noplateRecord.setCreateTime(new Date());
            noplateRecord.setIsMaster(isMaster);
            noplateRecordDao.insert(noplateRecord);
        }
        return objectResponse;
    }

    @Override
    public ObjectResponse<Void> fleetMode(String parkCode, Long parkId, String sn, Integer enable) {
       return fleetModeService.send(FleetModeRequest.builder().enable(enable).build(), sn, parkId);
    }

    private ObjectResponse<Void> getCurrChannelInfo(String parkCode, String aisleCode, String sn, String version, Integer type, String topic, Long parkId) {
        ObjectResponse<Void> objectResponse;
        //根据相机版本区分 1.5.5开始使用图片抓拍接口
        if (StringUtils.isNotEmpty(version) && P2cVersionEnum.getIndex(version) >= P2cVersionEnum.版本13.getIndex()
                && cacheHandle.judgeTakePicturePrivilege(sn)) {
            log.info("[调用新版本的图片抓拍接口] {},{},{},{}", parkCode, version, sn, aisleCode);
            objectResponse = takePicturesService.executeAsync(parkId, parkCode, sn, aisleCode, type, topic);
        } else {
            objectResponse = softTriggerService.execute(parkId, parkCode, aisleCode, topic,
                    SoftTriggerRequest.GetCarInfo.builder().biz(SoftTriggerRequest.ExtraInfoEnum.GET_CAR_INFO.val)
                            .type(type).channelCode(aisleCode).build());
        }
        return objectResponse;
    }
}
