package com.icetech.park.service.impl.batch;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.api.IBatchSendService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.BatchSendConstants;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.request.BatchSendActionRequest;
import com.icetech.cloudcenter.domain.request.BatchSendProgressRequest;
import com.icetech.cloudcenter.domain.request.BatchSendRepeatRequest;
import com.icetech.cloudcenter.domain.response.BatchSendActionResponse;
import com.icetech.cloudcenter.domain.response.BatchSendProgressResponse;
import com.icetech.cloudcenter.domain.response.BatchSendRepeatResponse;
import com.icetech.cloudcenter.domain.vo.BatchSendRepeatVO;
import com.icetech.cloudcenter.domain.vo.BatchSendVO;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.StringUtils;
import com.icetech.common.utils.UUIDTools;
import com.icetech.park.dao.BatchsendTaskDao;
import com.icetech.park.dao.BatchsendTaskSubDao;
import com.icetech.park.domain.entity.BatchsendTask;
import com.icetech.park.domain.entity.BatchsendTaskSub;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.down.p2c.impl.BlacklistServiceImpl;
import com.icetech.park.service.down.p2c.impl.ChannelRulesServiceImpl;
import com.icetech.park.service.down.p2c.impl.ChargeRuleServiceImpl;
import com.icetech.park.service.down.p2c.impl.LcdConfigServiceImpl;
import com.icetech.park.service.down.p2c.impl.LedsoundConfigServiceImpl;
import com.icetech.park.service.down.p2c.impl.MonthCardServiceImpl;
import com.icetech.park.service.down.p2c.impl.OssConfigServiceImpl;
import com.icetech.park.service.down.p2c.impl.VipInfoServiceImpl;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

import static com.icetech.cloudcenter.domain.constants.BatchDownMsgConstants.DEVICE_TIMEOUT;

@Service
@Slf4j
public class BatchSendServiceImpl implements IBatchSendService {
    @Autowired
    private ChannelRulesServiceImpl channelRulesService;
    @Autowired
    private MonthCardServiceImpl monthCardService;
    @Autowired
    private VipInfoServiceImpl vipInfoService;
    @Autowired
    private BlacklistServiceImpl blacklistService;
    @Autowired
    private ChargeRuleServiceImpl chargeRuleService;
    @Autowired
    private LcdConfigServiceImpl lcdConfigService;
    @Autowired
    private LedsoundConfigServiceImpl ledsoundConfigService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private OssConfigServiceImpl ossConfigService;
    @Autowired
    private BatchsendTaskDao batchsendTaskDao;
    @Autowired
    private BatchsendTaskSubDao batchsendTaskSubDao;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;

    //任务自动超时时间，秒
    private int TIMEOUT = 600;

    @Override
    public ObjectResponse<BatchSendActionResponse> action(BatchSendActionRequest batchSendActionRequest) {
        Integer[] configTypes = null;
        Integer[] bizTypes = null;
        Long parkId = batchSendActionRequest.getParkId();
        if (StrUtil.isNotBlank(batchSendActionRequest.getConfigTypes())){
            configTypes = Arrays.stream(batchSendActionRequest.getConfigTypes().split(",")).map(Integer::valueOf).toArray(Integer[] :: new);
        }
        if (StrUtil.isNotBlank(batchSendActionRequest.getBizTypes())){
            bizTypes = Arrays.stream(batchSendActionRequest.getBizTypes().split(",")).map(Integer::valueOf).toArray(Integer[] :: new);
        }
        Long[] channelIds = Arrays.stream(batchSendActionRequest.getChannelIds().split(",")).map(Long::valueOf).toArray(Long[] :: new);
        if (parkId == null){
            return ObjectResponse.failed(CodeConstants.ERROR_400, "parkId不能为空");
        }
        ObjectResponse<ParkConfig> parkConfigObjectResponse = parkService.getParkConfig(parkId);
        ObjectResponse.notError(parkConfigObjectResponse);
        ParkConfig parkConfig = parkConfigObjectResponse.getData();
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(parkId);
        ObjectResponse.notError(parkObjectResponse);
        Park park = parkObjectResponse.getData();
        //如果是端云车场，必须要传通道
        if (parkConfig.getDataCollection() == 1 || parkConfig.getDataCollection() == 3){
            if (channelIds.length == 0){
                return ObjectResponse.failed(CodeConstants.ERROR_400, "channelIds不能为空");
            }
        }

        if ((bizTypes == null || bizTypes.length == 0)
                && (configTypes == null || configTypes.length == 0)){
            return ObjectResponse.failed(CodeConstants.ERROR_400, "bizTypes和configTypes不能同时为空");
        }
        BatchsendTask batchsendTask = new BatchsendTask();
        batchsendTask.setParkId(parkId);
        batchsendTask = batchsendTaskDao.selectOneByEntity(batchsendTask);
        if (batchsendTask != null){
            Date operTime = batchsendTask.getUpdateTime() == null ? batchsendTask.getCreateTime() : batchsendTask.getUpdateTime();
            Integer status = batchsendTask.getStatus();
            //有任务正在处理中，且没有超过时效
            if (status != 3 && (System.currentTimeMillis() - operTime.getTime())/1000 < TIMEOUT){
                return ObjectResponse.failed(CodeConstants.ERROR_405, "下发任务正在处理中");
            }else {
                log.info("[批量下发] 任务已经超时,可以开始新的下发任务,参数[{}]", batchSendActionRequest);
            }
        }

        ObjectResponse<List<ParkDevice>> deviceResponse = parkDeviceService.getDevicesByParkChannel(parkId, channelIds, 1);
        ObjectResponse<List<ParkInoutdevice>> inoutResponse = parkService.getChannelsByIds(channelIds);
        if (ObjectResponse.isSuccess(deviceResponse) && ObjectResponse.isSuccess(inoutResponse)){
            List<ParkDevice> deviceList = deviceResponse.getData();
            List<ParkInoutdevice> parkInoutdevices = inoutResponse.getData();
            Map<Integer, List<ParkDevice>> channelList = deviceList.stream().collect(Collectors.groupingBy(ParkDevice::getChannelId));

            Integer[] serviceTypes = ArrayUtils.addAll(configTypes, bizTypes);
            //保存主任务记录
            String taskId = UUIDTools.getUuid();
            BatchsendTask batchsendTaskNew = new BatchsendTask();
            batchsendTaskNew.setTaskId(taskId);
            batchsendTaskNew.setServiceTypes(Arrays.stream(serviceTypes).map(String::valueOf).collect(Collectors.joining(",")));
            batchsendTaskNew.setChannelIds(Arrays.stream(channelIds).map(String::valueOf).collect(Collectors.joining(",")));
            batchsendTaskNew.setParkId(batchSendActionRequest.getParkId());
            batchsendTaskNew.setStatus(2);
            batchsendTaskNew.setOperAccount(batchSendActionRequest.getOperAccount());
            batchsendTaskNew.setUpdateTime(new Date());
            batchsendTaskDao.insert(batchsendTaskNew);

            //初始化子任务
            initAndSendSubTask(parkId, park.getParkCode(), serviceTypes, taskId, parkInoutdevices, channelList, parkConfig);

            BatchSendActionResponse batchSendActionResponse = new BatchSendActionResponse();
            batchSendActionResponse.setTaskId(taskId);
            return ObjectResponse.success(batchSendActionResponse);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_400, "通道或相机设备不存在");
        }
    }

    private void initAndSendSubTask(Long parkId, String parkCode, Integer[] serviceTypes, String taskId, List<ParkInoutdevice> parkInoutdevices, Map<Integer, List<ParkDevice>> channelIdDeviceListMap, ParkConfig parkConfig) {

        Arrays.stream(serviceTypes).forEach(serviceType -> {
            BatchSendVO batchSendVO = new BatchSendVO();
            batchSendVO.setParkCode(parkCode);
            batchSendVO.setParkId(parkId);
            batchSendVO.setTaskId(taskId);
            List<BatchSendVO.SubTaskInfo> list = new ArrayList<>();
            parkInoutdevices.forEach(parkInoutdevice ->{
                Long channelId = parkInoutdevice.getId();
                List<ParkDevice> deviceList = channelIdDeviceListMap.get(channelId.intValue());

                if (deviceList == null || deviceList.size() == 0) {
                    createSubTask(parkId, taskId, serviceType, batchSendVO, list, parkInoutdevice, null);
                }else{
                    deviceList.forEach(parkDevice -> {
                        createSubTask(parkId, taskId, serviceType, batchSendVO, list, parkInoutdevice, parkDevice);
                    });
                }
            });
            batchSendVO.setSubTaskInfos(list);
            redisUtils.expire(RedisConstants.BATCH_DOWN_TASK_LIST + taskId, 60 * 60);
            //异步下发
            asyncExecutor.execute(ThreadUtils.wrapTrace(()-> down(batchSendVO, serviceType, parkConfig)));
        });
    }

    private void createSubTask(Long parkId, String taskId, Integer serviceType, BatchSendVO batchSendVO, List<BatchSendVO.SubTaskInfo> list, ParkInoutdevice parkInoutdevice, ParkDevice parkDevice) {
        BatchSendVO.SubTaskInfo subTaskInfo = new BatchSendVO.SubTaskInfo();
        subTaskInfo.setChannelId(parkInoutdevice.getId());

        String suffixChannelName = "";
        if (parkDevice != null && parkDevice.getIsDualcamera() != null && parkDevice.getIsDualcamera() == 1){
            if (parkDevice.getIsMaster() != null && parkDevice.getIsMaster() == 1){
                suffixChannelName = "-主相机";
            }else{
                suffixChannelName = "-辅相机";
            }
        }
        subTaskInfo.setChannelName(parkInoutdevice.getInandoutName() + suffixChannelName);
        subTaskInfo.setSn(parkDevice != null ? parkDevice.getSerialNumber() : null);

        //保存子任务
        BatchsendTaskSub batchsendTaskSub = new BatchsendTaskSub();
        batchsendTaskSub.setTaskId(taskId);
        batchsendTaskSub.setParkId(parkId);
        batchsendTaskSub.setChannelId(subTaskInfo.getChannelId());
        batchsendTaskSub.setChannelName(subTaskInfo.getChannelName());
        batchsendTaskSub.setSn(subTaskInfo.getSn());
        batchsendTaskSub.setServiceType(serviceType);
        batchsendTaskSub.setTotalNum(0);
        batchsendTaskSubDao.insert(batchsendTaskSub);
        log.info("[数据批量下发业务] 写入子任务表完成, 参数[{}]", batchsendTaskSub);

        //建立子任务和主任务的联系
        Integer subTaskId = batchsendTaskSub.getId();
        subTaskInfo.setSubTaskId(subTaskId);
        redisUtils.lRightPush(RedisConstants.BATCH_DOWN_TASK_LIST + taskId, subTaskId);

        //缓存子任务详情
        String subTaskKey = RedisConstants.BATCH_DOWN_SUBTASK_HASH + subTaskId;
        HashMap<String, Object> hashMap = new HashMap<>();
        hashMap.put("taskId", taskId);
        hashMap.put("channelId", subTaskInfo.getChannelId());
        hashMap.put("channelName", subTaskInfo.getChannelName());
        hashMap.put("sn", subTaskInfo.getSn() == null ? "NONE" : subTaskInfo.getSn());
        hashMap.put("serviceType", serviceType);
        hashMap.put("status", 1);
        hashMap.put("totalNum", 0);
        hashMap.put("successNum", 0);
        hashMap.put("failNum", 0);
        hashMap.put("startTime", DateTools.unixTimestamp());
        redisUtils.hPutAll(subTaskKey, hashMap);
        redisUtils.expire(subTaskKey, 60 * 60);
        log.info("[数据批量下发业务] 缓存子任务详情完成, 参数[{}]", batchsendTaskSub);

        list.add(subTaskInfo);

    }

    @Override
    public ObjectResponse<BatchSendProgressResponse> progress(BatchSendProgressRequest batchSendProgressRequest) {
        List<BatchSendProgressResponse.Progress> list = new ArrayList<>();
        BatchSendProgressResponse batchSendProgressResponse = new BatchSendProgressResponse();
        BatchsendTask batchsendTask = new BatchsendTask();
        batchsendTask.setTaskId(batchSendProgressRequest.getTaskId());
        batchsendTask = batchsendTaskDao.selectOneByEntity(batchsendTask);
        if (batchsendTask != null){
            String taskId = batchsendTask.getTaskId();
            String subTaskIds = batchSendProgressRequest.getSubTaskId();
            if (subTaskIds != null){
                Arrays.stream(subTaskIds.split(",")).forEach(subTaskId -> setSubTaskProgress(list, String.valueOf(subTaskId)));
                long count = list.stream().filter(progress -> progress.getStatus() != 3).count();
                if (count > 0){
                    batchSendProgressResponse.setIsFinish(0);
                }else {
                    batchSendProgressResponse.setIsFinish(1);
                }
            }else if (taskId != null){
                List<String> objects = redisUtils.lRange(RedisConstants.BATCH_DOWN_TASK_LIST + taskId, 0, -1, String.class);
                if (CollectionUtil.isNotEmpty(objects)) {
                    objects.forEach(subTaskId -> setSubTaskProgress(list, subTaskId));
                    long count = list.stream().filter(progress -> progress.getStatus() != 3).count();
                    if (count > 0){
                        batchSendProgressResponse.setIsFinish(0);
                    }else {
                        batchSendProgressResponse.setIsFinish(1);
                    }
                } else {
                    log.info("[任务进度查询]redis中没有主任务数据,从数据库查询[{}]", taskId);
                    List<BatchsendTaskSub> batchsendTaskSubs = batchsendTaskSubDao.selectListByTaskId(taskId);
                    if (CollectionUtil.isNotEmpty(batchsendTaskSubs)){
                        batchsendTaskSubs.forEach(batchsendTaskSub -> {
                            BatchSendProgressResponse.Progress progress = new BatchSendProgressResponse.Progress();
                            BeanUtils.copyProperties(batchsendTaskSub, progress);
                            progress.setSubTaskId(String.valueOf(batchsendTaskSub.getId()));
                            Integer successNum = batchsendTaskSub.getSuccessNum();
                            Integer totalNum = batchsendTaskSub.getTotalNum();
                            progress.setSuccessRatio(successNum == 0 ? 0 : (int) (successNum * 1.00 / totalNum * 100));
                            list.add(progress);
                        });
                        batchSendProgressResponse.setIsFinish(batchsendTask.getStatus() == 3 ? 1 : 0);
                    }else{
                        log.warn("[任务进度查询]根据任务ID查库未找到进度情况[{}]", taskId);
                        return ObjectResponse.failed(CodeConstants.ERROR_402);
                    }
                }
            }
            batchSendProgressResponse.setProgressList(list);

            Date operTime = batchsendTask.getUpdateTime() == null ? batchsendTask.getCreateTime() : batchsendTask.getUpdateTime();
            //数据库中的任务正在处理中，且超过时效或redis中已经全部完成时
            if (batchsendTask.getStatus() != 3
                    && ((System.currentTimeMillis() - operTime.getTime())/1000 > TIMEOUT || batchSendProgressResponse.getIsFinish() == 1)){
                //任务超时
                batchSendProgressResponse.setIsFinish(1);
                batchsendTask.setStatus(3);
                batchsendTaskDao.updateById(batchsendTask);
                log.info("[任务进度查询]任务已经超时,更新成已完成[{}]", taskId);
            }
            return ObjectResponse.success(batchSendProgressResponse);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }
    }

    @Override
    public ObjectResponse<BatchSendRepeatResponse> repeat(BatchSendRepeatRequest batchSendRepeatRequest) {
        Integer subTaskId = Integer.parseInt(batchSendRepeatRequest.getSubTaskId());
        BatchsendTaskSub batchsendTaskSub = batchsendTaskSubDao.selectById(subTaskId);
        if (batchsendTaskSub == null){
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }
        BatchSendRepeatVO batchSendRepeatVO = new BatchSendRepeatVO();
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(batchsendTaskSub.getParkId());
        batchSendRepeatVO.setParkCode(parkObjectResponse.getData().getParkCode());
        batchSendRepeatVO.setParkId(batchsendTaskSub.getParkId());
        batchSendRepeatVO.setTaskId(batchsendTaskSub.getTaskId());
        batchSendRepeatVO.setSubTaskId(subTaskId);
        batchSendRepeatVO.setOldTotalNum(batchsendTaskSub.getTotalNum());
        ObjectResponse<ParkDevice> deviceResponse = parkDeviceService.getDeviceBySerialNumber(batchsendTaskSub.getSn());
        ObjectResponse<ParkInoutdevice> inoutResponse = parkService.getInoutDeviceById(batchsendTaskSub.getChannelId());
        if (ObjectResponse.isSuccess(deviceResponse) && ObjectResponse.isSuccess(inoutResponse)){
            ParkInoutdevice parkInoutdevice = inoutResponse.getData();
            ParkDevice parkDevice = deviceResponse.getData();
            batchSendRepeatVO.setChannelId(parkInoutdevice.getId());
            batchSendRepeatVO.setChannelName(parkInoutdevice.getInandoutName());
            batchSendRepeatVO.setSn(parkDevice.getSerialNumber());

        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }

        String subTaskKey = RedisConstants.BATCH_DOWN_SUBTASK_HASH + subTaskId;
        if (!redisUtils.exists(subTaskKey)) {
            //补充REDIS 缓存子任务详情
            HashMap<String, Object> hashMap = new HashMap<>();
            hashMap.put("taskId", batchsendTaskSub.getTaskId());
            hashMap.put("channelId", batchsendTaskSub.getChannelId());
            hashMap.put("channelName", batchsendTaskSub.getChannelName());
            hashMap.put("sn", batchsendTaskSub.getSn());
            hashMap.put("serviceType", batchsendTaskSub.getServiceType());
            hashMap.put("status", 1);
            hashMap.put("totalNum", batchsendTaskSub.getTotalNum());
            hashMap.put("successNum", batchsendTaskSub.getSuccessNum());
            hashMap.put("failNum", batchsendTaskSub.getFailNum());
            hashMap.put("startTime", DateTools.unixTimestamp());
            redisUtils.hPutAll(subTaskKey, hashMap);
        }else{
            String failIds = redisUtils.hGet(subTaskKey, "failIds", String.class);
            if (StringUtils.isNotBlank(failIds)){
                batchSendRepeatVO.setFailIds(failIds);
            }
        }
        asyncExecutor.execute(ThreadUtils.wrapTrace(()->{
            //延长任务有效期
            batchsendTaskDao.updateTime(batchsendTaskSub.getTaskId(), new Date());

            Integer serviceType = batchsendTaskSub.getServiceType();
            downRepeat(batchSendRepeatVO, serviceType);
        }));
        return ObjectResponse.success();
    }

    private void down(BatchSendVO batchSendVO, Integer serviceType, ParkConfig parkConfig) {

        if (BatchSendConstants.PARK_CHANNEL_CONFIG == serviceType){
            channelRulesService.batchSend(batchSendVO);
        }
        if (BatchSendConstants.PARK_CHANNEL_CONFIG == serviceType){
            ossConfigService.batchSend(batchSendVO);
        }
        if (BatchSendConstants.CHARGE_CONFIG == serviceType){
            chargeRuleService.batchSend(batchSendVO);
        }
        if (BatchSendConstants.HINT_CONFIG == serviceType){
            for (BatchSendVO.SubTaskInfo subTaskInfo : batchSendVO.getSubTaskInfos()) {
                ObjectResponse<ParkInoutdevice> channelResp = parkService.getInoutDeviceById(subTaskInfo.getChannelId());
                ParkInoutdevice channel = channelResp.getData();
                if (channel.getLedLcdSource() == 2) {
                    lcdConfigService.batchSend(batchSendVO, subTaskInfo);
                } else {
                    ledsoundConfigService.batchSend(batchSendVO, subTaskInfo);
                }
            }
        }
        if (BatchSendConstants.MC_BIZ == serviceType){
            monthCardService.batchSend(batchSendVO);
        }
        if (BatchSendConstants.VIP_BIZ == serviceType){
            vipInfoService.batchSend(batchSendVO);
        }
        if (BatchSendConstants.BL_BIZ == serviceType){
            blacklistService.batchSend(batchSendVO);
        }
    }

    private void downRepeat(BatchSendRepeatVO batchSendRepeatVO, Integer serviceType) {
        //准备下发
        String configPrefix = "1";
        if (String.valueOf(serviceType).startsWith(configPrefix)){
            if (BatchSendConstants.PARK_CONFIG == serviceType || BatchSendConstants.CHANNEL_CONFIG == serviceType) {
                channelRulesService.batchSendRepeat(batchSendRepeatVO);
            }
            if (BatchSendConstants.PARK_CONFIG == serviceType) {
                ossConfigService.batchSendRepeat(batchSendRepeatVO);
            }
            if (BatchSendConstants.CHARGE_CONFIG == serviceType) {
                chargeRuleService.batchSendRepeat(batchSendRepeatVO);
            }
            if (BatchSendConstants.HINT_CONFIG == serviceType) {
                ObjectResponse<ParkInoutdevice> objectResponse = parkService.getInoutDeviceById(batchSendRepeatVO.getChannelId());
                if (ObjectResponse.isSuccess(objectResponse)) {
                    Integer ledLcdSource = objectResponse.getData().getLedLcdSource();
                    if (ledLcdSource == 2) {
                        lcdConfigService.batchSendRepeat(batchSendRepeatVO);
                    } else {
                        ledsoundConfigService.batchSendRepeat(batchSendRepeatVO);
                    }
                } else {
                    ledsoundConfigService.batchSendRepeat(batchSendRepeatVO);
                }
            }
        }else{
            if (BatchSendConstants.MC_BIZ == serviceType) {
                monthCardService.batchSendRepeat(batchSendRepeatVO);
            }
            if (BatchSendConstants.VIP_BIZ == serviceType) {
                vipInfoService.batchSendRepeat(batchSendRepeatVO);
            }
            if (BatchSendConstants.BL_BIZ == serviceType) {
                blacklistService.batchSendRepeat(batchSendRepeatVO);
            }
        }
    }


    private void setSubTaskProgress(List<BatchSendProgressResponse.Progress> list, String subTaskId) {
        BatchSendProgressResponse batchSendProgressResponse = new BatchSendProgressResponse();
        BatchSendProgressResponse.Progress progress = new BatchSendProgressResponse.Progress();
        String subTaskKey = RedisConstants.BATCH_DOWN_SUBTASK_HASH + subTaskId;
        Map<String, Object> objectMap = redisUtils.hGetAll(subTaskKey, Object.class);
        if (CollectionUtil.isNotEmpty(objectMap)) {
            progress.setSubTaskId(subTaskId);
            progress.setChannelId(Long.valueOf(objectMap.get("channelId").toString()));
            progress.setChannelName((String) objectMap.get("channelName"));
            progress.setSn((String) objectMap.get("sn"));
            Integer serviceType = (Integer) objectMap.get("serviceType");
            progress.setServiceType(serviceType);
            Integer status = (Integer) objectMap.get("status");
            progress.setStatus(status);
            Integer successNum = (Integer) objectMap.get("successNum");
            Integer totalNum = (Integer) objectMap.get("totalNum");
            progress.setTotalNum(totalNum);
            progress.setFailNum(totalNum - successNum);
            progress.setSuccessNum(successNum);
            progress.setReason((String) objectMap.get("reason"));
            Integer startTime = (Integer) objectMap.get("startTime");
            //配置下发的开头
            String configPrefix = "1";
            //配置业务，如果超过30秒仍然是正在处理中，则手动置为超时
            if (status != 3){
                if (String.valueOf(serviceType).startsWith(configPrefix)){
                    if (DateTools.unixTimestamp() - startTime > 30){
                        progress.setStatus(3);
                        progress.setReason(DEVICE_TIMEOUT);
                        redisUtils.hPut(subTaskKey, "failNum", 1);
                        redisUtils.hPut(subTaskKey, "status", 3);
                        redisUtils.hPut(subTaskKey, "reason", DEVICE_TIMEOUT);

                        finishErrorTask(1, 0, Integer.valueOf(subTaskId), DEVICE_TIMEOUT);
                    }
                }else{
                    if (DateTools.unixTimestamp() - startTime > 600){
                        progress.setStatus(3);
                        progress.setReason(DEVICE_TIMEOUT);
                        redisUtils.hPut(subTaskKey, "status", 3);
                        redisUtils.hPut(subTaskKey, "reason", DEVICE_TIMEOUT);

                        finishErrorTask(totalNum, successNum, Integer.valueOf(subTaskId), DEVICE_TIMEOUT);
                    }
                }
            }
            progress.setSuccessRatio(successNum == 0 ? 0 : (int) (successNum * 1.00 / totalNum * 100));
        } else {
            BatchsendTaskSub batchsendTaskSub = batchsendTaskSubDao.selectById(subTaskId);
            if (batchsendTaskSub == null){
                log.warn("[封装子任务进度]根据子任务ID未找到进度情况[{}]", subTaskId);
                return;
            }else{
                BeanUtils.copyProperties(batchsendTaskSub, progress);
                progress.setSubTaskId(String.valueOf(batchsendTaskSub.getId()));
                Integer successNum = batchsendTaskSub.getSuccessNum();
                Integer totalNum = batchsendTaskSub.getTotalNum();
                progress.setSuccessRatio(successNum == 0 ? 0 : (int) (successNum * 1.00 / totalNum * 100));
            }
        }
        list.add(progress);
    }
    private void finishErrorTask(Integer totalNum, Integer successNum, Integer subTaskId, String reason) {

        BatchsendTaskSub batchsendTaskSub = new BatchsendTaskSub();
        batchsendTaskSub.setId(subTaskId);
        batchsendTaskSub.setStatus(3);
        batchsendTaskSub.setTotalNum(totalNum);
        batchsendTaskSub.setFailNum(totalNum - successNum);
        batchsendTaskSub.setSuccessNum(successNum);
        batchsendTaskSub.setReason(reason);
        batchsendTaskSubDao.updateById(batchsendTaskSub);
    }
}
