package com.icetech.park.service.impl;

import com.icetech.cloudcenter.api.order.OrderPayService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.request.SumOrderPayRequest;
import com.icetech.cloudcenter.domain.order.SumOrderPayDto;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.*;
import com.icetech.cloudcenter.api.ReconciliationService;
import com.icetech.cloudcenter.domain.request.LocalReconciliationRequest;
import com.icetech.park.dao.ReconciliationDao;
import com.icetech.park.dao.ReconciliationInfoDao;
import com.icetech.park.domain.entity.Reconciliation;
import com.icetech.park.domain.entity.ReconciliationInfo;
import com.icetech.cloudcenter.domain.vo.common.ReconciliationVo;
import com.icetech.cloudcenter.domain.enumeration.ReconciliationInfoResultEnum;
import com.icetech.cloudcenter.domain.enumeration.ReconciliationResultEnum;
import com.icetech.park.service.AbstractService;
import com.icetech.oss.OssService;
import com.opencsv.CSVReader;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Service
@Slf4j
public class ReconciliationServiceImpl extends AbstractService implements ReconciliationService {

    @Autowired
    private ParkService parkService;
    @Autowired
    private ReconciliationDao reconciliationDao;
    @Autowired
    private ReconciliationInfoDao reconciliationInfoDao;
    @Autowired
    private OrderPayService orderPayService;
    @Autowired
    private OrderService orderService;
    @Autowired
    private OssService ossService;

    @Override
    public ObjectResponse execute(LocalReconciliationRequest localReconciliationRequest) {
        try {
            String parkCode = localReconciliationRequest.getParkCode();
            Date startTime = localReconciliationRequest.getStartTime();
            Date endTime = localReconciliationRequest.getEndTime();
            log.info("<对账明细> 对账车场 : {}, 对账时间, startTime : {}, endTime : {}", parkCode, startTime, endTime);
            // 判断是否有parkId
            if (parkCode == null) {
                // 查询所有端网云parkId
                ObjectResponse<List<Park>> listObjectResponse = parkService.parkListByType(2);
                if (!ObjectResponse.isSuccess(listObjectResponse)) {
                    return ObjectResponse.failed(CodeConstants.ERROR_404);
                } else {
                    // 遍历parkIdList
                    List<Park> parkList = listObjectResponse.getData();
                    for (Park park : parkList) {
                        Long parkId = park.getId();
                        ObjectResponse timeCheck = timeCheck(startTime, endTime, parkId, park.getParkCode());
                        if (ObjectResponse.isSuccess(timeCheck)) {
                            log.info("<对账明细> 车场 {} 对账成功，对账日期：{}至{}", parkId, startTime, endTime);
                        } else {
                            log.info("<对账明细> 车场 {} 对账失败，对账日期：{}至{}", parkId, startTime, endTime);
                        }
                    }
                }
            } else {
                ObjectResponse<Park> parkObjectResponse = parkService.findByParkCode(parkCode);
                if (!ObjectResponse.isSuccess(parkObjectResponse)) {
                    return ObjectResponse.failed(CodeConstants.ERROR_404);
                }
                Park park = parkObjectResponse.getData();
                ObjectResponse timeCheck = timeCheck(startTime, endTime, park.getId(), park.getParkCode());
                if (ObjectResponse.isSuccess(timeCheck)) {
                    return ObjectResponse.success();
                } else {
                    log.info("<对账明细> 对账失败，请求参数：{}", localReconciliationRequest);
                    return ObjectResponse.failed(CodeConstants.ERROR);
                }
            }
        } catch (Exception e) {
            log.info("<对账明细> 对账时异常: {}", e);
            return ObjectResponse.failed(CodeConstants.ERROR);
        }
        return ObjectResponse.success();

    }


    /**
     * 日期对账
     *
     * @param startTime
     * @param endTime
     * @param parkId
     * @return
     */
    public ObjectResponse timeCheck(Date startTime, Date endTime, Long parkId, String parkCode) {
        int times = 1;
        long startTimeNew = startTime.getTime()/1000;
        if (!startTime.equals(endTime)) {
            long endTimeNew = endTime.getTime()/1000;
            times = Math.toIntExact(endTimeNew - startTimeNew) / 60 / 60 / 24 + 1;
        }
        ObjectResponse objectResponse = null;
        for (int i = 0; i < times; i++) {
            objectResponse = checkBill(startTimeNew, parkId, parkCode);
            startTimeNew = startTimeNew + 60 * 60 * 24;
        }
        return objectResponse;
    }

    /**
     * 对账
     *
     * @param time
     * @param parkId
     * @return
     */
    public ObjectResponse checkBill(long time, long parkId, String parkCode) {
        try {
            long endTime = time + 60 * 60 * 24;
            String startTime = DateTools.secondTostringDate3(Math.toIntExact(time));
            // 查询是否重复对账
            int countReconciliationCount = reconciliationDao.countReconciliation(parkId, startTime);
            // 获取平台各项总金额
            SumOrderPayRequest sumOrderPayRequest = new SumOrderPayRequest();
            sumOrderPayRequest.setParkId(parkId);
            sumOrderPayRequest.setPayTimeStart(time);
            sumOrderPayRequest.setPayTimeEnd(endTime);
            sumOrderPayRequest.setPayStatus(PayStatusConstants.PAID);
            ObjectResponse<SumOrderPayDto> sumOrderPayDtoObjectResponse = orderPayService.sumOrderPay(sumOrderPayRequest);
            if (!ObjectResponse.isSuccess(sumOrderPayDtoObjectResponse)) {
                log.info("<对账接口>获取平台各项总金额失败;请求参数: {}", sumOrderPayRequest);
                return ObjectResponse.failed(CodeConstants.ERROR_404);
            }
            SumOrderPayDto sumOrderPayDto = sumOrderPayDtoObjectResponse.getData();
            BigDecimal pf_totalPrice = sumOrderPayDto.getTotalPrice();
            BigDecimal pf_paidPrice = sumOrderPayDto.getPaidPrice();
            BigDecimal pf_discountPrice = sumOrderPayDto.getDiscountPrice();
            BigDecimal pf_actualCash = sumOrderPayDto.getActualCash();
            BigDecimal pf_redpackRet = sumOrderPayDto.getRedpackRet();
            Integer pf_totalCount = sumOrderPayDto.getTotalCount();

            Reconciliation reconciliationUpdate = new Reconciliation();
            reconciliationUpdate.setParkId(parkId);
            reconciliationUpdate.setReportDate(startTime);
            reconciliationUpdate.setPlatformCount(pf_totalCount);
            reconciliationUpdate.setPlatformPriceTotal(pf_totalPrice);
            reconciliationUpdate.setPlatformPricePaid(pf_paidPrice);
            reconciliationUpdate.setPlatformPriceDiscount(pf_discountPrice);
            reconciliationUpdate.setPlatformActualCash(pf_actualCash);
            reconciliationUpdate.setPlatformRedpackRet(pf_redpackRet);
            if (countReconciliationCount == 0) {
                // 添加平台总对账数据
                reconciliationDao.insert(reconciliationUpdate);
            }
            List<String[]> csvList = new ArrayList<String[]>(); // 用来保存数据
            // 查账对账文件路径
            String csvFilePath = generateBillPath(parkCode, startTime);
            log.info("<对账接口> 下载OSS上文件的路径 csvFilePath : {}", csvFilePath);
            try {
                String imageUrl = ossService.getImageUrl(csvFilePath);
                File oss2File = FileTools.getFileFromUrl(imageUrl, getFileName(parkCode, startTime));
                try (FileInputStream fis = new FileInputStream(oss2File);
                     InputStreamReader isr = new InputStreamReader(fis);
                     CSVReader reader = new CSVReader(isr)) {
                    csvList = reader.readAll();
                    oss2File.delete();
                }
            } catch (Exception e) {
                log.info("<对账接口> 对账文件不存在 {}", csvFilePath);
                reconciliationUpdate.setResult(ReconciliationResultEnum.对账文件未上报.getResult());
                reconciliationDao.update(reconciliationUpdate);
                return ObjectResponse.failed(CodeConstants.ERROR_404);
            }
            // 获取文件头
            String[] strings = csvList.get(0);
            log.info("<对账接口> 获取csv文件头: {}，{}，{}，{}", strings[0], strings[1], strings[2], strings[3]);

            double local_price_total = Float.parseFloat(strings[2] == null ? "0" : strings[2]) / 100D;
            double local_price_paid = Float.parseFloat(strings[3] == null ? "0" : strings[3]) / 100D;
            double local_price_discount = Float.parseFloat(strings[4] == null ? "0" : strings[4]) / 100D;
            double local_actual_cash = Float.parseFloat(strings[5] == null ? "0" : strings[5]) / 100D;
            double local_redpack_ret = Float.parseFloat(strings[6] == null ? "0" : strings[6]) / 100D;

            reconciliationUpdate.setLocalCount(strings[1] == null ? 0 : Integer.parseInt(strings[1]));
            reconciliationUpdate.setLocalPriceTotal(new BigDecimal(local_price_total).setScale(2, BigDecimal.ROUND_HALF_DOWN));
            reconciliationUpdate.setLocalPricePaid(new BigDecimal(local_price_paid).setScale(2, BigDecimal.ROUND_HALF_DOWN));
            reconciliationUpdate.setLocalPriceDiscount(new BigDecimal(local_price_discount).setScale(2, BigDecimal.ROUND_HALF_DOWN));
            reconciliationUpdate.setLocalActualCash(new BigDecimal(local_actual_cash).setScale(2, BigDecimal.ROUND_HALF_DOWN));
            reconciliationUpdate.setLocalRedpackRet(new BigDecimal(local_redpack_ret).setScale(2, BigDecimal.ROUND_HALF_DOWN));
            log.info(
                    "<对账接口> 开始判断对账结果，参数: price_total:{},{}, price_paid:{},{}, price_discount:{},{}, count:{},{}",
                    local_price_total, pf_totalPrice, local_price_paid,
                    pf_paidPrice, local_price_discount, pf_discountPrice,
                    reconciliationUpdate.getLocalCount(), pf_totalCount);
            // 比较各项总金额
            if (local_price_total != NumberUtils.toDouble(pf_totalPrice)
                    || local_price_paid != NumberUtils.toDouble((pf_paidPrice))
                    || local_price_discount != NumberUtils.toDouble((pf_discountPrice))
                    || !ToolsUtil.parseFloat(reconciliationUpdate.getLocalCount()).equals(ToolsUtil.parseFloat(pf_totalCount))) {
                reconciliationUpdate.setResult(ReconciliationResultEnum.对账不一致.getResult());
            } else {
                reconciliationUpdate.setResult(ReconciliationResultEnum.对账一致.getResult());
            }
            // update总对账
            reconciliationDao.update(reconciliationUpdate);
            // 删除重复的对账明细
            reconciliationInfoDao.delRepeat(parkId, startTime);
            // 循环获取文件体
            for (int row = 1; row < csvList.size(); row++) {
                String[] oneRow = csvList.get(row);
                ReconciliationInfo reconciliationInfo = new ReconciliationInfo();
                reconciliationInfo.setParkId(parkId);
                reconciliationInfo.setReportDate(startTime);
                reconciliationInfo.setTradeNo(oneRow[0]);
                reconciliationInfo.setLocalOrderNum(oneRow[1]);
                reconciliationInfo.setPayChannel(Integer.parseInt(oneRow[2]));
                reconciliationInfo.setPayWay(Integer.parseInt(oneRow[3]));
                BigDecimal default_price = new BigDecimal(0);
                reconciliationInfo.setLocalPriceTotal(oneRow[4] == null ? default_price : new BigDecimal(Double.parseDouble(oneRow[4]) / 100D).setScale(2, BigDecimal.ROUND_HALF_DOWN));
                reconciliationInfo.setLocalPricePaid(oneRow[5] == null ? default_price : new BigDecimal(Double.parseDouble(oneRow[5]) / 100D).setScale(2, BigDecimal.ROUND_HALF_DOWN));
                reconciliationInfo.setLocalPriceDiscount(oneRow[6] == null ? default_price : new BigDecimal(Double.parseDouble(oneRow[6]) / 100D).setScale(2, BigDecimal.ROUND_HALF_DOWN));
                reconciliationInfo.setLocalActualCash(oneRow[7] == null ? default_price : new BigDecimal(Double.parseDouble(oneRow[7]) / 100D).setScale(2, BigDecimal.ROUND_HALF_DOWN));
                reconciliationInfo.setLocalRedpackRet(oneRow[8] == null ? default_price : new BigDecimal(Double.parseDouble(oneRow[8]) / 100D).setScale(2, BigDecimal.ROUND_HALF_DOWN));
                reconciliationInfo.setPayTime(oneRow[9] == null ? null : Integer.parseInt(oneRow[9]));
                if (reconciliationInfo.getOrderNum() != null) {
                    ObjectResponse<OrderInfo> orderInfoObjectResponse = orderService.findByOrderNum(reconciliationInfo.getOrderNum());
                    if (ObjectResponse.isSuccess(orderInfoObjectResponse)) {
                        OrderInfo orderInfo = orderInfoObjectResponse.getData();
                        reconciliationInfo.setPlateNum(orderInfo.getPlateNum());
                    }
                }
                reconciliationInfoDao.insert(reconciliationInfo);
            }
            List<ReconciliationVo> reconciliationVoList = reconciliationInfoDao.findBillList(parkId, time, endTime);
            log.info("<对账接口> 对账详情查询 请求参数: {}, {}, {}, 返回{} 条", parkId, time, endTime, reconciliationVoList.size());
            for (int i = 0; i < reconciliationVoList.size(); i++) {
                ReconciliationVo reconciliationVo = reconciliationVoList.get(i);
                ReconciliationInfo reconciliationInfo = new ReconciliationInfo();
                BeanUtils.copyProperties(reconciliationVo, reconciliationInfo);
                reconciliationInfo.setPlatformPriceTotal(reconciliationVo.getTotalPrice());
                reconciliationInfo.setPlatformPricePaid(reconciliationVo.getPaidPrice());
                reconciliationInfo.setPlatformPriceDiscount(reconciliationVo.getDiscountPrice());
                reconciliationInfo.setPlatformActualCash(reconciliationVo.getActualCash());
                reconciliationInfo.setPlatformRedpackRet(reconciliationVo.getRedpackRet());

                if (reconciliationInfo.getPlatformPriceTotal() == null && reconciliationInfo.getPlatformPricePaid() == null
                        && reconciliationInfo.getPlatformPriceDiscount() == null) {
                    reconciliationInfo.setResult(ReconciliationInfoResultEnum.本地有平台无.getResult());// 本地有平台无
                } else if (reconciliationInfo.getLocalPriceTotal() == null
                        && reconciliationInfo.getLocalPricePaid() == null
                        && reconciliationInfo.getLocalPriceDiscount() == null) {
                    reconciliationInfo.setParkId(parkId);
                    reconciliationInfo.setReportDate(startTime);
                    reconciliationInfo.setTradeNo(reconciliationVo.getPfTradeNo());
                    reconciliationInfo.setPayTime(reconciliationVo.getPfPayTime());
                    reconciliationInfo.setPayChannel(reconciliationVo.getPfPayChannel());
                    reconciliationInfo.setPayWay(reconciliationVo.getPfPayWay());
                    reconciliationInfo.setResult(ReconciliationInfoResultEnum.平台有本地无.getResult());// 平台有本地无
                } else {
                    if (!reconciliationInfo.getLocalPriceTotal()
                            .equals(reconciliationInfo.getPlatformPriceTotal())
                            || !reconciliationInfo.getLocalPricePaid()
                            .equals(reconciliationInfo.getLocalPricePaid())
                            || !reconciliationInfo.getLocalPriceDiscount()
                            .equals(reconciliationInfo.getPlatformPriceDiscount())) {
                        reconciliationInfo.setResult(ReconciliationInfoResultEnum.对账不一致.getResult());// 不一致
                    } else {
                        reconciliationInfo.setResult(ReconciliationInfoResultEnum.对账一致.getResult());// 一致
                    }
                }
                if (reconciliationInfo.getOrderNum() != null && reconciliationInfo.getPlateNum() == null) {
                    ObjectResponse<OrderInfo> orderInfoObjectResponse = orderService.findByOrderNum(reconciliationInfo.getOrderNum());
                    if (ObjectResponse.isSuccess(orderInfoObjectResponse)) {
                        OrderInfo orderInfo = orderInfoObjectResponse.getData();
                        reconciliationInfo.setPlateNum(orderInfo.getPlateNum());
                    }
                }
                if (reconciliationInfo.getResult().equals(ReconciliationInfoResultEnum.平台有本地无.getResult())) {
                    reconciliationInfoDao.insert(reconciliationInfo);
                } else {
                    reconciliationInfoDao.update(reconciliationInfo);
                }
            }
        } catch (Exception ex) {
            log.info("<对账接口> 异常: {}", ex);
            return ObjectResponse.failed(CodeConstants.ERROR);
        }

        return ObjectResponse.success();
    }

    private String generateBillPath(String parkCode, String startTime) {
        String[] strings = startTime.split("-");
        String path = parkCode + "/bill/" + strings[0] + strings[1] + "/" + getFileName(parkCode, startTime);
        return path;
    }

    private String getFileName(String parkCode, String startTime) {
        return startTime.replace("-", "") + "_" + parkCode + "_BILL_1_0.csv";
    }
}
