package com.icetech.park.service.impl;

import com.icetech.cloudcenter.api.QueryExitPlateNumService;
import com.icetech.cloudcenter.api.order.OrderDiscountService;
import com.icetech.cloudcenter.domain.constants.DataCommonConstants;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.response.QueryExitPlateNumResponse;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.order.domain.entity.OrderDiscount;
import com.icetech.park.service.AbstractService;
import com.icetech.park.handle.CacheHandle;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

@Service
@Slf4j
public class QueryExitPlateNumServiceImpl extends AbstractService implements QueryExitPlateNumService {
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private AlarmServiceImpl alarmService;
    @Autowired
    private OrderDiscountService orderDiscountService;

    @Override
    public ObjectResponse<QueryExitPlateNumResponse> query(String parkCode, String channelId) {
        if (StringUtils.isBlank(parkCode) || StringUtils.isBlank(channelId)){
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        CarExitRequest exit = cacheHandle.getExit(parkCode, channelId);
        if (exit == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_3008);
        }
        if (exit.getAllowExit() != null && !exit.getAllowExit()) {
            return ObjectResponse.failed(CodeConstants.ERROR_406, "当前通道禁止临时车出场");
        }
        String orderNum = exit.getOrderNum();
        String plateNum = exit.getPlateNum();
        if (DataCommonConstants.isNoPlate(plateNum)) {
            log.info("[查询出口当前车辆和费用信息] 未识别车牌返回, parkCode:{}, channelId:{}", parkCode, channelId);
            alarmService.queryFeeFailHandler(parkCode, exit.getParkId(), channelId);
            return ObjectResponse.failed(CodeConstants.ERROR_3008);
        }
        if (StringUtils.isNotBlank(orderNum) && StringUtils.isNotBlank(plateNum)) {
            QueryExitPlateNumResponse queryExitPlateNumResponse = new QueryExitPlateNumResponse();
            queryExitPlateNumResponse.setPlateNum(plateNum);
            queryExitPlateNumResponse.setOrderNum(orderNum);
            queryExitPlateNumResponse.setExitTime(exit.getExitTime());
            QueryOrderFeeResponse channelFee = cacheHandle.getChannelFee(parkCode, channelId);
            if (channelFee != null && channelFee.getPlateNum().equals(plateNum)) {
                queryExitPlateNumResponse.setCacheFee(channelFee);
                OrderDiscount orderDiscount = new OrderDiscount();
                orderDiscount.setOrderNum(orderNum);
                orderDiscount.setStatus(0);
                ObjectResponse<List<OrderDiscount>> list = orderDiscountService.findList(orderDiscount);
                if (list.getData() != null) {
                    boolean sendDiscountAfterIdentify = list.getData().stream().anyMatch(orderDiscount1 ->
                            orderDiscount1.getSendTime() != null && orderDiscount1.getSendTime().getTime() / 1000 > channelFee.getQueryTime());
                    if (sendDiscountAfterIdentify) {
                        queryExitPlateNumResponse.setCacheFee(null);
                    }
                }
            } else if (channelFee != null) {
                log.info("[查询出口当前车辆和费用信息] 出口车辆信息缓存中的车牌和出口费用信息缓存中的车牌号不同，前：{}，后：{}", plateNum, channelFee.getPlateNum());
            }
            return ObjectResponse.success(queryExitPlateNumResponse);
        }
        alarmService.queryFeeFailHandler(parkCode, exit.getParkId(), channelId);
        return ObjectResponse.failed(CodeConstants.ERROR_404);
    }
}
