package com.icetech.park.service.impl;

import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import com.icetech.cloudcenter.domain.response.DiscountUsedDto;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.order.dao.OrderDiscountDao;
import com.icetech.park.domain.vo.FindDiscountVo;
import com.icetech.cloudcenter.api.discount.MerchantDiscountService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.fee.dao.merchant.DiscountDayDao;
import com.icetech.fee.dao.merchant.MerchantDiscountDao;
import com.icetech.fee.dao.merchant.MerchantRechargeDao;
import com.icetech.fee.dao.merchant.ParkMerchantDao;
import com.icetech.basics.dao.park.ParkConfigDao;
import com.icetech.fee.domain.entity.merchant.DiscountDay;
import com.icetech.fee.domain.entity.merchant.MerchantDiscount;
import com.icetech.fee.domain.entity.merchant.ParkMerchant;
import com.icetech.order.domain.entity.OrderDiscount;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.cloudcenter.domain.response.MerchantDiscountDto;
import com.icetech.cloudcenter.domain.response.OrderDiscountDto;
import com.icetech.cloudcenter.domain.response.RechargeDetailDto;
import com.icetech.cloudcenter.domain.response.RechargeListDto;
import com.icetech.cloudcenter.domain.response.SubRechargeListDto;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author wangzw
 */
@Service("merchantDiscountService")
@Slf4j
public class MerchantDiscountServiceImpl implements MerchantDiscountService {
    @Autowired
    private MerchantDiscountDao merchantDiscountDao;
    @Autowired
    private OrderDiscountDao orderDiscountDao;
    @Autowired
    private ParkMerchantDao parkMerchantDao;
    @Autowired
    private MerchantRechargeDao merchantRechargeDao;
    @Autowired
    private ParkService parkService;
    @Autowired
    private DiscountDayDao discountDayDao;
    @Autowired
    private ParkConfigDao parkConfigDao;

    @Override
    public ObjectResponse<List<MerchantDiscountDto>> getDiscountTemplate(Long merchantId) {
        List<MerchantDiscountDto> dtos = Lists.newArrayList();
        //查询当前生效的优惠模板
        List<MerchantDiscount> discounts = merchantDiscountDao.selectByMerchantId(merchantId);
        ParkConfig parkConfig;
        if (!CollectionUtils.isEmpty(discounts)) {
            MerchantDiscount merchantDiscount1 = discounts.get(0);
            long parkId = merchantDiscount1.getParkId();
            parkConfig = parkConfigDao.selectByParkId(parkId);
            String units =parkConfig.getMoneyunitType();
            discounts.forEach(merchantDiscount -> {
                MerchantDiscountDto merchantDiscountDto = new MerchantDiscountDto();
                BeanUtils.copyProperties(merchantDiscount, merchantDiscountDto);
                merchantDiscountDto.setParkName(parkService.findByParkId(merchantDiscountDto.getParkId()).getData().getParkName());
                merchantDiscountDto.setUnits(units);
                dtos.add(merchantDiscountDto);
            });
        }
        return ObjectResponse.success(dtos);
    }

    @Override
    public ObjectResponse<MerchantDiscountDto> getDayDiscountTemplate(Long merchantId) {
        //查询当前生效的优惠模板
        MerchantDiscount discount = merchantDiscountDao.selectDayDisByMerchantId(merchantId);
        if (Objects.nonNull(discount)) {
            long parkId = discount.getParkId();
            ParkConfig parkConfig = parkConfigDao.selectByParkId(parkId);
            MerchantDiscountDto merchantDiscountDto = new MerchantDiscountDto();
            BeanUtils.copyProperties(discount, merchantDiscountDto);
            merchantDiscountDto.setParkName(parkService.findByParkId(merchantDiscountDto.getParkId()).getData().getParkName());
            merchantDiscountDto.setUnits(parkConfig.getMoneyunitType());
            merchantDiscountDto.setAmountDay(discount.getAmountday());
            return ObjectResponse.success(merchantDiscountDto);
        }
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
    }

    @Override
    public ObjectResponse<MerchantDiscountDto> findMerchantDiscountById(Long disId) {
        MerchantDiscount merchantDiscount = merchantDiscountDao.findById(disId);
        if (Objects.isNull(merchantDiscount)) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404.getCode(), CodeConstantsEnum.ERROR_404.getDesc());
        }
        MerchantDiscountDto merchantDiscountDto = new MerchantDiscountDto();
        BeanUtils.copyProperties(merchantDiscount, merchantDiscountDto);
        return ObjectResponse.success(merchantDiscountDto);
    }

    @Override
    public ObjectResponse<List<OrderDiscountDto>> findDiscountRecords(FindDiscountVo vo) {
        log.info("查询优惠参数[{}]", vo);
        String from = vo.getFrom();
        String to = vo.getTo();
        //分页查询
        PageHelper.startPage(NumberUtils.toPrimitive(vo.getPageNo(), 1), NumberUtils.toPrimitive(vo.getPageSize(), 10));

        if (StringUtils.isEmpty(from)) {
            from = LocalDateTime.now().minus(Duration.ofDays(1)).format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
            vo.setFrom(from);
        }
        if (StringUtils.isEmpty(to)) {
            to = DateTools.getFormat(new Date());
            vo.setTo(to);
        }
        List<OrderDiscountDto> orderDiscounts = orderDiscountDao.findDiscountRecords(vo);
        if (CollectionUtils.isEmpty(orderDiscounts)) {
            return ObjectResponse.success(Lists.newArrayList());
        }
        OrderDiscountDto orderDiscountDto1 = orderDiscounts.get(0);
        Long parkId = orderDiscountDto1.getParkId();
        ObjectResponse<ParkConfig> configObjectResponse = parkService.getParkConfig(parkId);
        ObjectResponse.notError(configObjectResponse);
        ParkConfig parkConfig = configObjectResponse.getData();
        String moneyunitType = parkConfig.getMoneyunitType();
        orderDiscounts.forEach(orderDiscountDto -> {
            Integer type = orderDiscountDto.getType();
            String typeName = "";
            switch (type) {
                case 1:
                    typeName = "全免券";
                    break;
                case 2:
                    typeName = "时长券";
                    break;
                case 3:
                    typeName = "代金券";
                    break;
                case 4:
                    typeName = "折扣券";
                    break;
                case 5:
                    typeName = "按天优免券";
                    break;
                default:
                    break;
            }
            orderDiscountDto.setTypeStr(typeName);
            orderDiscountDto.setUnits(moneyunitType);
        });
        return ObjectResponse.success(orderDiscounts);
    }

    @Override
    public ObjectResponse<DiscountUsedDto> findDiscountUsedRecords(FindDiscountVo vo) {
        DiscountUsedDto discountUsedDto = new DiscountUsedDto();
        discountUsedDto.setTotalDiscountPrice("0.0");
        ObjectResponse<List<OrderDiscountDto>> discountRecords = findDiscountRecords(vo);
        if (ObjectResponse.isSuccess(discountRecords)) {
            List<OrderDiscountDto> data = discountRecords.getData();
            discountUsedDto.setList(data);
            double sum = data.stream().mapToDouble(d -> NumberUtils.parseDouble(d.getGetAmount())).sum();
            discountUsedDto.setTotalDiscountPrice(String.valueOf(sum));
        }
        return ObjectResponse.success(discountUsedDto);
    }

    @Override
    public ObjectResponse<OrderDiscountDto> findDiscountRecord(Long id) {
        OrderDiscount orderDiscount = new OrderDiscount();
        orderDiscount.setId(id);
        orderDiscount = orderDiscountDao.selectLimitOne(orderDiscount);
        if (Objects.isNull(orderDiscount)) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404.getCode(), CodeConstantsEnum.ERROR_404.getDesc());
        }
        OrderDiscountDto orderDiscountDto = new OrderDiscountDto();
        BeanUtils.copyProperties(orderDiscount, orderDiscountDto);
        orderDiscountDto.setSendTime(DateTools.getFormat(orderDiscount.getSendTime()));
        //获取商户名称
        ParkMerchant parkMerchant = new ParkMerchant();
        parkMerchant.setId(Math.toIntExact(orderDiscount.getMerchantId()));
        parkMerchant = parkMerchantDao.selectById(parkMerchant);
        orderDiscountDto.setMerchantName(Objects.isNull(parkMerchant) ? "" : parkMerchant.getName());
        //获取车场信息
        Park park = parkService.findByParkId(orderDiscountDto.getParkId()).getData();
        orderDiscountDto.setParkName(Objects.isNull(park) ? "" : park.getParkName());
        Integer type = orderDiscountDto.getType();
        ParkConfig parkConfig = parkConfigDao.selectByParkId(orderDiscountDto.getParkId());
        String amount = orderDiscountDto.getAmount();
        switch (type) {
            case 2:
                amount = amount + "分钟";
                break;
            case 3:
                amount = amount + parkConfig.getMoneyunitType();
                break;
            case 4:
                amount = amount + "折";
                break;
            default:
                break;
        }
        orderDiscountDto.setAmount(amount);
        //按天优免
        if (type == 5){
            DiscountDay discountDay = discountDayDao.selectDiscountDay(orderDiscount.getDiscountNo(), (long) parkMerchant.getParkId());
            if (Objects.nonNull(discountDay)){
                orderDiscountDto.setStartTime(DateTools.getFormat(discountDay.getStartTime()));
                orderDiscountDto.setEndTime(DateTools.getFormat(discountDay.getEndTime()));
            }
        }
        return ObjectResponse.success(orderDiscountDto);
    }

    @Override
    public ObjectResponse<List<RechargeListDto>> findRechargeList(Long merchantId, Integer selectType,
                                                                  String yearMonth, String startDate, String endDate,List<Integer> rechargeTypes,List<Integer> payWays) {
        Map<String, Object> paraMap = new HashMap<>();
        paraMap.put("merchantId", merchantId);
        paraMap.put("yearMonth", yearMonth);
        paraMap.put("startDate", startDate);
        paraMap.put("endDate", endDate);
        paraMap.put("rechargeTypes", rechargeTypes);
        paraMap.put("payWays", payWays);
        ParkMerchant parkMerchant = new ParkMerchant();
        parkMerchant.setId(merchantId.intValue());
        ParkMerchant parkMerchant1 = parkMerchantDao.selectById(parkMerchant);
        int parkId = parkMerchant1.getParkId();
        String moneyunitType = parkConfigDao.selectByParkId(parkId).getMoneyunitType();
        if (selectType == null){
            List<RechargeListDto> rechargeListDtos = merchantRechargeDao.countMonth(paraMap);
            //遍历单月的结果集
            for (RechargeListDto rechargeListDto : rechargeListDtos){
                String yearMonth2 = rechargeListDto.getYearMonth();
                String newMonth = yearMonth2.replace("年", "-").replace("月", "");
                paraMap.put("yearMonth",newMonth);
                List<SubRechargeListDto> details = merchantRechargeDao.monthRecharge(paraMap);
                rechargeListDto.setDetails(details);
                rechargeListDto.setUnits(moneyunitType);
            }
            return ObjectResponse.success(rechargeListDtos);
        }else{
            //按月查询
            if (selectType == 1) {
                List<RechargeListDto> rechargeListDtos = merchantRechargeDao.countMonth(paraMap);
                if (rechargeListDtos != null && rechargeListDtos.size() > 0){
                    //按月查询一定只有一条记录
                    RechargeListDto rechargeListDto = rechargeListDtos.get(0);

                    List<SubRechargeListDto> details = merchantRechargeDao.monthRecharge(paraMap);
                    rechargeListDto.setDetails(details);
                    rechargeListDto.setUnits(moneyunitType);
                    return ObjectResponse.success(rechargeListDtos);
                }else{
                    return ObjectResponse.failed(CodeConstants.ERROR_404);
                }
                //按日期范围查询
            } else if (selectType == 2) {

                String total = merchantRechargeDao.countScope(merchantId, startDate, endDate);
                if (total == null){
                    return ObjectResponse.failed(CodeConstants.ERROR_404);
                }else{
                    RechargeListDto rechargeListDto = new RechargeListDto();
                    rechargeListDto.setYearMonth(startDate + "至" + endDate);
                    rechargeListDto.setMonthRechargeAmount(total);
                    List<SubRechargeListDto> details = merchantRechargeDao.monthRecharge(paraMap);
                    rechargeListDto.setDetails(details);
                    rechargeListDto.setUnits(moneyunitType);
                    List<RechargeListDto> list = new ArrayList<RechargeListDto>();
                    list.add(rechargeListDto);
                    return ObjectResponse.success(list);
                }
            }else{
                return ObjectResponse.failed(CodeConstants.ERROR_402);
            }
        }
    }

    @Override
    public ObjectResponse<RechargeDetailDto> findRechargeDetail(Integer recordId) {
        RechargeDetailDto rechargeDetail = merchantRechargeDao.findRechargeDetail(recordId);
        if (rechargeDetail == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        } else {
            Long parkId = rechargeDetail.getParkId();
            rechargeDetail.setUnits(parkConfigDao.selectByParkId(parkId).getMoneyunitType());
            return ObjectResponse.success(rechargeDetail);
        }
    }

    @Override
    public ObjectResponse<MerchantDiscount> findTemplateInfo(Long disId) {
        MerchantDiscount merchantDiscount = merchantDiscountDao.findById(disId);
        if (merchantDiscount == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        } else {
            return ObjectResponse.success(merchantDiscount);
        }
    }

    @Override
    public ObjectResponse saveDiscountPlateNum(DiscountDay discountDay) {
        int i = discountDayDao.insertSelective(discountDay);
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<DiscountDay> findValidDiscountDay(String plateNum, Long parkId, String payTime) {
        DiscountDay discountDay = discountDayDao.selectValidDiscountDay(plateNum, parkId, payTime);
        if (discountDay == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        } else {
            return ObjectResponse.success(discountDay);
        }
    }

    @Override
    public ObjectResponse<DiscountDay> findDiscountDay(String discountNo, Long parkId) {
        DiscountDay discountDay = discountDayDao.selectDiscountDay(discountNo, parkId);
        if (discountDay == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        } else {
            return ObjectResponse.success(discountDay);
        }
    }

    @Override
    public ObjectResponse findValidExistedDisCountDay(String plateNum,Long parkId) {
        DiscountDay discountDay = discountDayDao.findValidExistedDisCountDay(parkId,plateNum,new Date());
        if (Objects.isNull(discountDay)) return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        return ObjectResponse.success(discountDay);
    }

    @Override
    public ObjectResponse<DiscountDay> findLastDisDayByPlateAndDisId(String plateNum, Long parkId, Long disId) {
        DiscountDay discountDay = discountDayDao.findByPlateAndDisId(parkId, plateNum, disId);
        return ObjectResponse.returnNotFoundIfNull(discountDay);
    }

    @Override
    public ObjectResponse<DiscountDay> findLastDisDayByPlateAndOnceDay(String plateNum, Long parkId) {
        DiscountDay discountDay = discountDayDao.findLastDisDayByPlateAndOnceDay(parkId, plateNum);
        return ObjectResponse.returnNotFoundIfNull(discountDay);
    }

    @Override
    public ObjectResponse<OrderDiscount> findLastDiscountByPlateAndDisId(String plateNum, Long parkId, Long disId) {
        OrderDiscount orderDiscount = orderDiscountDao.findByPlateAndDisId(parkId, plateNum, disId);
        return ObjectResponse.returnNotFoundIfNull(orderDiscount);
    }

    @Override
    public ObjectResponse<OrderDiscount> findLastDiscountByPlateAndOnceDay(String plateNum, Long parkId) {
        OrderDiscount orderDiscount = orderDiscountDao.findLastDiscountByPlateAndOnceDay(parkId, plateNum);
        return ObjectResponse.returnNotFoundIfNull(orderDiscount);
    }

    @Override
    public ObjectResponse checkSpaceNum(Integer merchantId) {
        try {
            ParkMerchant parkMerchant = new ParkMerchant();
            parkMerchant.setId(merchantId);
            parkMerchant = parkMerchantDao.selectById(parkMerchant);
            //查询已经优惠的车位数
            Integer spaceNum = parkMerchant.getSpaceNum();
            if (Objects.isNull(spaceNum)){
                return ObjectResponse.success(spaceNum);
            }
            log.info("当前商户[{}]可优惠车位数[{}]",parkMerchant.getName(),spaceNum);
            if (spaceNum>0){
                return ObjectResponse.success(spaceNum);
            }else {
                return ObjectResponse.failed(CodeConstantsEnum.ERROR.getCode(),"当前可优惠车位数为0");
            }
        }catch (Exception e) {
            log.error("[优惠校验车位数]异常: {}. merchantId[{}]", e.getMessage(), merchantId, e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR);
        }
    }

    @Override
    public ObjectResponse subDiscountSpaceNum(Integer merchantId) {
        try {
            ParkMerchant parkMerchant = new ParkMerchant();
            parkMerchant.setId(merchantId);
            parkMerchant = parkMerchantDao.selectById(parkMerchant);
            Integer spaceNum = parkMerchant.getSpaceNum();
            if (Objects.isNull(spaceNum)){
                return ObjectResponse.success();
            }
            if(spaceNum>0){
                parkMerchant.setSpaceNum(spaceNum-1);
                log.info("当前商户[{}]扣减车位数[{}]",parkMerchant.getName(),spaceNum-1);
                parkMerchantDao.update(parkMerchant);
            }
            return ObjectResponse.success();
        }catch (Exception e){
            log.error("[优惠扣减车位数]异常: {}. merchantId[{}]", e.getMessage(), merchantId, e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR);
        }

    }

    @Override
    public ObjectResponse addDiscountDaySpaceNum() {
        //判断当前未释放有效期
        List<DiscountDay> discountDays = discountDayDao.selectDiscountDayByRelease(null, 590L);
        discountDays.forEach(discountDay -> {
            if (discountDay!=null&&System.currentTimeMillis()>=discountDay.getEndTime().getTime()){
                Integer merchantDisId = discountDay.getMerchantDisId();
                MerchantDiscount merchantDiscount = new MerchantDiscount();
                merchantDiscount.setId(merchantDisId);
                merchantDiscount = merchantDiscountDao.selectById(merchantDiscount);
                ParkMerchant parkMerchant = new ParkMerchant();
                parkMerchant.setId(Math.toIntExact(merchantDiscount.getMerchantId()));
                parkMerchant = parkMerchantDao.selectById(parkMerchant);
                Integer spaceNum = parkMerchant.getSpaceNum();
                if (spaceNum!=null){
                    parkMerchant.setSpaceNum(++spaceNum);
                    if (parkMerchant.getSpaceNum()>60){
                        parkMerchant.setSpaceNum(60);
                    }
                    log.info("当前商户[{}],优惠券编号[{}]按天优免释放后车位数[{}]",parkMerchant.getName(),discountDay.getDiscountNo(),parkMerchant.getSpaceNum());
                    parkMerchantDao.update(parkMerchant);
                    discountDay.setIsRelease(Boolean.TRUE);
                    discountDayDao.updateByPrimaryKey(discountDay);
                }
            }
        });
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<DiscountDay> selectLastByPlateNum(String plateNum, Long parkId) {
        DiscountDay discountDay = discountDayDao.selectLastByPlateNum(plateNum, parkId);
        if (Objects.nonNull(discountDay)){
            return ObjectResponse.success(discountDay);
        }
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
    }

    @Override
    public ObjectResponse updateDiscountDay(DiscountDay discountDay) {
        discountDayDao.updateByPrimaryKey(discountDay);
        return ObjectResponse.success();
    }
}
