package com.icetech.park.service.impl;

import com.icetech.cloudcenter.api.HeartbeatHandleService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.dao.device.HeartbeatDao;
import com.icetech.basics.dao.device.HeartbeatOfflineDao;
import com.icetech.basics.dao.device.ParkDeviceDao;
import com.icetech.cloudcenter.domain.constants.DCTimeOutConstants;
import com.icetech.basics.domain.entity.device.Heartbeat;
import com.icetech.basics.domain.entity.device.HeartbeatOffline;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.handle.PublicHandle;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

/**
 * 心跳定时处理实现
 * @author fangct
 */
@Slf4j
@Service
public class HeartbeatHandleV1ServiceImpl implements HeartbeatHandleService {
    @Autowired
    private HeartbeatDao heartbeatDao;
    @Autowired
    private HeartbeatOfflineDao heartbeatOfflineDao;
    @Autowired
    private ParkDeviceDao parkDeviceDao;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private PublicHandle publicHandle;
    @Autowired
    private ParkService parkService;
    @Autowired
    private CacheHandle cacheHandle;

    @Override
    public ObjectResponse execute(String param) {
        //只查询设备心跳断线的记录，不包括端网云的计费系统心跳，计费系统的断线记录在心跳接口parkStatus中处理
        List<Heartbeat> heartbeats = heartbeatDao.getOffline(DCTimeOutConstants.OFF_LINE_TIME);
        log.info("<心跳包定时处理> 心跳包断开设备个数：{}", heartbeats == null ? 0 : heartbeats.size());
        for (int i = 0; heartbeats != null && i< heartbeats.size(); i ++){
            Heartbeat heartbeat = heartbeats.get(i);
            Long parkId = heartbeat.getParkId();
            String deviceNo = heartbeat.getDeviceNo();

            ParkDevice parkDevice = parkDeviceDao.getByDeviceNo(parkId, deviceNo);
            if (parkDevice == null) {
                continue;
            }else if(parkDevice.getStatus() == 2) {
                //已经是离线状态
                continue;
            }else if(parkDevice.getStatus() == 1) {//未修改前为在线
                Long now = DateTools.unixTimestamp();
                Date endUpdatetime = parkDevice.getEndUpdatetime();
                if (endUpdatetime == null){//未连接的设备不写离线记录
                    continue;
                }else if (now - endUpdatetime.getTime() / 1000 < DCTimeOutConstants.OFF_LINE_TIME){
                    //如果设备最后通讯时间与当前时间相差不到掉线的时长，则可能是刚刚连接上
                    continue;
                }

                //修改为离线状态
                parkDeviceService.modifyStatus(parkId, deviceNo, 2, "断网", heartbeat.getServerTime());

                //增加掉线记录
                HeartbeatOffline heartbeatOffline = new HeartbeatOffline();
                heartbeatOffline.setParkId(parkId);
                heartbeatOffline.setLastConnectionTime(heartbeat.getServerTime());
                heartbeatOffline.setDeviceNo(deviceNo);
                heartbeatOfflineDao.insert(heartbeatOffline);

                /**
                 * 准备发送通知消息
                 */
                ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(parkId);
                ObjectResponse.notError(parkObjectResponse);
                Park park = parkObjectResponse.getData();
                if (parkDevice.getChannelId() != 0) {
                    ObjectResponse<ParkInoutdevice> objectResponse = parkService.getInoutDeviceById((long)parkDevice.getChannelId());
                    if (ObjectResponse.isSuccess(objectResponse)) {
                        ParkInoutdevice data = objectResponse.getData();
                        String content = "【设备离线提醒】车场名称：" + park.getParkName() + "，出入口名称：" + data.getInandoutName() + "，设备编号" + parkDevice.getDeviceNo() +
                                "，最后通讯时间：" + DateTools.getFormat(endUpdatetime);
                        publicHandle.sendSmgToOpsUser(content, parkId);
                        if (parkDevice.getType() == 1) {
                            /**
                             * 清除缓存中的端云相机连接信息
                             */
                            cacheHandle.closeForClearCache(parkDevice.getSerialNumber());
                        } else if (parkDevice.getType() == 5){
                            /**
                             * 清除缓存中的机器人连接信息
                             */
                            cacheHandle.closeForRobotClearCache(parkDevice.getSerialNumber());
                        }
                    }
                }

            }
        }
        return ObjectResponse.success();
    }
}
