package com.icetech.park.service.impl;

import com.icetech.basics.constants.HeartbeatOfflineConsts;
import com.icetech.basics.domain.dto.HeartbeatOfflineDto;
import com.icetech.cloudcenter.api.HeartbeatHandleService;
import com.icetech.basics.dao.device.ParkDeviceDao;
import com.icetech.cloudcenter.domain.constants.DCTimeOutConstants;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.basics.domain.entity.device.HeartbeatOffline;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.cloudcenter.domain.request.pnc.ChargeStatusRequest;
import com.icetech.park.domain.entity.park.ParkFreespace;
import com.icetech.basics.service.device.HeartbeatOfflineService;
import com.icetech.basics.service.device.impl.ParkDeviceDaoImpl;
import com.icetech.basics.service.device.impl.ParkDeviceServiceImpl;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.handle.ItcCacheHandle;
import com.icetech.park.service.handle.MorCacheHandle;
import com.icetech.park.service.park.impl.ParkFreeSpaceServiceImpl;
import com.icetech.third.service.third.MqPushService;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 心跳定时处理实现
 * @author fangct
 */
@Slf4j
@Service
public class HeartbeatHandleServiceImpl implements HeartbeatHandleService {
    @Autowired
    private ParkDeviceDao parkDeviceDao;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private ItcCacheHandle itcCacheHandle;
    @Autowired
    private MorCacheHandle morCacheHandle;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private ParkDeviceServiceImpl parkDeviceService;
    @Autowired
    private ParkDeviceDaoImpl parkDeviceDaoImpl;
    @Autowired
    private HeartbeatOfflineService heartbeatOfflineService;
    @Autowired
    private ParkFreeSpaceServiceImpl parkFreeSpaceService;
    @Autowired
    private MqPushService mqPushService;
    /**
     * 设备掉线数报警阈值
     */
    private static final int DEVICE_THRESHOLD = 100;
    /**
     * 端网云掉线数报警阈值
     */
    private static final int PNC_THRESHOLD = 10;
    /**
     * 分批处理时每批的大小
     */
    private static final int batchSize = 500;

    @Override
    public ObjectResponse execute(String param) {
        long now = DateTools.unixTimestamp();
        Long lastTaskTime = redisUtils.get(RedisConstants.HEARTBEAT_TASK_TIME, Long.class);
        if (lastTaskTime == null) {
            //默认一分钟前
            lastTaskTime = now - 60;
        }
        long diffSeconds = now - lastTaskTime;
        //相机设备检查
        deviceCheckAndUpdate(1, now);
        //机器人设备检查
        deviceCheckAndUpdate(5, now);
        //端网云检查
        pncCheck(now, lastTaskTime, diffSeconds);
        //批量更新端网云空车位
        //batchUpdateFreeSpace();
        //立柱更新最后通讯时间
        batchUpdateDeviceTime(8);
        //监控相机更新最后通讯时间
        batchUpdateDeviceTime(9);

        //更新上次任务时间
        redisUtils.set(RedisConstants.HEARTBEAT_TASK_TIME, now);
        return ObjectResponse.success();
    }

    /**
     * 设备心跳状态检查
     * @param deviceType
     * @param now
     * @param lastTaskTime
     * @param diffSeconds
     */
    private void deviceCheckAndUpdate(int deviceType, long now, Long lastTaskTime, long diffSeconds) {
        String key = deviceType ==1 ? RedisConstants.P2C_HEARTBEAT_TIME : RedisConstants.P2R_HEARTBEAT_TIME;
        Map<String, Long> listMap = redisUtils.hGetAll(key, Long.class);
        Set<String> sns = listMap.keySet();
        long timeEnd = now - DCTimeOutConstants.OFF_LINE_TIME;
        //上次心跳过去3分钟的设备集合
        List<String> offlineDeviceList = sns.stream()
                .filter(sn -> listMap.get(sn) <= timeEnd
                                && listMap.get(sn) > timeEnd - diffSeconds)
                .collect(Collectors.toList());
        if (!offlineDeviceList.isEmpty()) {
            if (offlineDeviceList.size() >= DEVICE_THRESHOLD) {
                log.warn("[监控埋点] 上次任务[{}]到当前任务的离线设备数为[{}], 超过了阈值[{}]",
                        DateTools.getFormat(new Date(lastTaskTime * 1000)), offlineDeviceList.size(), DEVICE_THRESHOLD);
            }
            parkDeviceService.batchModifyStatus(offlineDeviceList, listMap, deviceType, 2);
            offlineDeviceList.forEach(sn -> {
                if (deviceType == 1) {
                    //清除缓存中的端云相机连接信息
                    cacheHandle.closeForClearCache(sn);
                } else if (deviceType == 5) {
                    //清除缓存中的机器人连接信息
                    cacheHandle.closeForRobotClearCache(sn);
                } else if (deviceType == 8) {
                    //清除缓存中的机器人连接信息
                    itcCacheHandle.closeForClearCache(sn);
                } else if (deviceType == 9) {
                    //清除缓存中的机器人连接信息
                    morCacheHandle.closeForClearCache(sn);
                }
            });
        }

        //上次心跳时间在60秒内的设备都是在线
        List<String> onlineDeviceList = sns.stream()
                .filter(sn -> listMap.get(sn) >= now - 60)
                .collect(Collectors.toList());
        //查询库里的设备状态不在线，但实际在线的设备列表
        List<String> failStatusOnlineList = parkDeviceDao.selectFailStatusList(onlineDeviceList, deviceType, 1);
        if (!failStatusOnlineList.isEmpty()) {
            log.info("[设备状态纠正] 实际在线的设备[{}]", failStatusOnlineList);
            parkDeviceService.batchModifyStatus(failStatusOnlineList, listMap, deviceType, 1);
        }
    }

    /**
     * 端网云心跳状态检查
     * @param now
     * @param lastTaskTime
     * @param diffSeconds
     */
    private void pncCheck(long now, Long lastTaskTime, long diffSeconds) {
        String key = RedisConstants.PNC_HEARTBEAT_TIME;
        Map<String, Long> listMap = redisUtils.hGetAll(key, Long.class);
        Set<String> parkIds = listMap.keySet();
        long timeEnd = now - DCTimeOutConstants.OFF_LINE_TIME;
        //上次心跳过去3分钟的设备集合
        List<String> offlineParkList = parkIds.stream()
                .filter(sn -> listMap.get(sn) <= timeEnd
                                && listMap.get(sn) > timeEnd - diffSeconds)
                .collect(Collectors.toList());
        if (!offlineParkList.isEmpty()) {
            if (offlineParkList.size() >= PNC_THRESHOLD) {
                log.warn("[监控埋点] 上次任务[{}]到当前任务的离线设备数为[{}], 超过了阈值[{}]",
                        DateTools.getFormat(new Date(lastTaskTime * 1000)), offlineParkList.size(), PNC_THRESHOLD);
            }
            List<HeartbeatOffline> offlineList = offlineParkList.stream()
                    .map(parkId -> {
                        //增加掉线记录
                        HeartbeatOffline heartbeatOffline = new HeartbeatOffline();
                        heartbeatOffline.setParkId(Long.valueOf(parkId));
                        heartbeatOffline.setDeviceNo(HeartbeatOfflineConsts.DEVICE_NO_PNC_CENTER);
                        heartbeatOffline.setLastConnectionTime(listMap.get(parkId));
                        return heartbeatOffline;
                    }).collect(Collectors.toList());
            heartbeatOfflineService.saveBatch(offlineList);
            // 移除端网云协议缓存
            redisUtils.hDelete(RedisConstants.PNC_PROTOCOL, offlineParkList.toArray(new String[0]));
            for (HeartbeatOffline offline : offlineList) mqPushService.pushPncCenterOffline(offline);
        }

        //上次心跳时间在60秒内的设备都是在线
        List<Long> onlineDeviceList = parkIds.stream()
                .filter(parkId -> listMap.get(parkId) >= now - 60)
                .map(Long::new)
                .collect(Collectors.toList());
        //查询库里的设备状态不在线，但实际在线的设备列表
        if (!onlineDeviceList.isEmpty()) {
            log.info("[端网云状态更新] 实际在线的设备[{}]", onlineDeviceList);
            // 添加车场连接断开记录表
            List<HeartbeatOffline> heartbeatOfflines = heartbeatOfflineService.getNoFinishLastByParkId(onlineDeviceList);
            Iterator<HeartbeatOffline> iterator = heartbeatOfflines.iterator();
            while (iterator.hasNext()) {
                HeartbeatOffline heartbeatOffline = iterator.next();
                //最近心跳时间
                Long lastHeartbeatTime = listMap.get(String.valueOf(heartbeatOffline.getParkId()));
                if (lastHeartbeatTime != null) {
                    heartbeatOffline.setReconnectTime(lastHeartbeatTime);
                    heartbeatOffline.setOffTime(lastHeartbeatTime - heartbeatOffline.getLastConnectionTime());
                } else {
                    iterator.remove();
                }
            }
            if (!heartbeatOfflines.isEmpty()) {
                heartbeatOfflineService.updateBatchById(heartbeatOfflines);
            }
        }

        checkPncChargeHeartbeat(now, timeEnd, diffSeconds);
    }

    private void checkPncChargeHeartbeat(long now, long timeEnd, long diffSeconds) {
        if (log.isDebugEnabled()) log.debug("设备离线检测|端网云计费中心检测|检测时间|{}|{}|{}", now, timeEnd, diffSeconds);
        Set<String> chargeHeartbeatMapKeys = redisUtils.getStringRedisTemplate().keys(RedisConstants.PNC_HEARTBEAT_CHARGE_TIME_PREFIX + "*");
        if (CollectionUtils.isEmpty(chargeHeartbeatMapKeys)) return;
        List<HeartbeatOffline> offlineList = new LinkedList<>();
        List<HeartbeatOffline> onlineList = new LinkedList<>();
        for (String redisKey : chargeHeartbeatMapKeys) {
            Long parkId = Long.valueOf(redisKey.substring(RedisConstants.PNC_HEARTBEAT_CHARGE_TIME_PREFIX.length()));
            Map<String, ChargeStatusRequest> heartbeatMap = redisUtils.hGetAll(redisKey, ChargeStatusRequest.class);
            if (MapUtils.isEmpty(heartbeatMap)) continue;
            //上次心跳过去3分钟的设备集合
            heartbeatMap.values().stream()
                    .filter(heartbeat -> heartbeat.getTime() <= timeEnd && heartbeat.getTime() > timeEnd - diffSeconds)
                    .map(heartbeat -> {
                        //增加掉线记录
                        HeartbeatOfflineDto heartbeatOffline = new HeartbeatOfflineDto();
                        heartbeatOffline.setParkId(parkId);
                        heartbeatOffline.setDeviceNo(heartbeat.getId().toString());
                        heartbeatOffline.setRemark(heartbeat.getName());
                        heartbeatOffline.setLastConnectionTime(heartbeat.getTime());
                        return heartbeatOffline;
                    }).forEach(offlineList::add);
            List<String> onlineChargeIds = heartbeatMap.entrySet().stream()
                    .filter(entry -> entry.getValue().getTime() >= now - 60)
                    .map(Map.Entry::getKey)
                    .collect(Collectors.toCollection(LinkedList::new));
            if (!onlineChargeIds.isEmpty()) {
                List<HeartbeatOffline> offlines = heartbeatOfflineService.getNoFinishChargeLastByChargeId(parkId, onlineChargeIds);
                for (HeartbeatOffline offline : offlines) {
                    // 最近心跳时间
                    Long lastHeartbeatTime = heartbeatMap.get(offline.getDeviceNo()).getTime();
                    if (lastHeartbeatTime != null) {
                        offline.setReconnectTime(lastHeartbeatTime);
                        offline.setOffTime(lastHeartbeatTime - offline.getLastConnectionTime());
                        onlineList.add(offline);
                    }
                }
            }
            if (!offlineList.isEmpty()) {
                heartbeatOfflineService.saveBatch(offlineList);
                for (HeartbeatOffline offline : offlineList)
                    mqPushService.pushPncChargeOffline(offline);
            }
            if (!onlineList.isEmpty()) heartbeatOfflineService.updateBatchById(onlineList);
        }
    }

    /**
     * 设备心跳状态检查并更新时间
     * @param deviceType
     * @param now
     */
    private void deviceCheckAndUpdate(int deviceType, long now) {
        String key = "";
        if (deviceType == 1) {
            key = RedisConstants.P2C_HEARTBEAT_TIME;
        } else if (deviceType == 5) {
            key = RedisConstants.P2R_HEARTBEAT_TIME;
        } else if (deviceType == 8) {
            key = RedisConstants.ITC_HEARTBEAT_TIME;
        } else if (deviceType == 9) {
            key = RedisConstants.MOR_HEARTBEAT_TIME;
        }
        Map<String, Long> listMap = redisUtils.hGetAll(key, Long.class);
        Set<String> sns = listMap.keySet();
        if (CollectionUtils.isEmpty(sns)) {
            return;
        }
        List<String> allSnList = new ArrayList<>(sns);
        int offlineCount = 0;
        for(int i = 0; i < allSnList.size(); i += batchSize) {
            int end = i + batchSize;
            if(end > allSnList.size()) {
                end = allSnList.size();
            }
            List<String> snList = allSnList.subList(i, end);
            List<ParkDevice> parkDevices = parkDeviceDao.selectListBySns(snList, deviceType);
            //时间发生变化的设备
            List<ParkDevice> updateDeviceList = parkDevices.stream()
                    .filter(parkDevice -> {
                        Long redisTime = listMap.get(parkDevice.getSerialNumber());
                        return redisTime != null && (parkDevice.getEndUpdatetime() == null ||
                            parkDevice.getEndUpdatetime().getTime() / 1000 != redisTime);
                    }).collect(Collectors.toList());
            //实际在线设备，目前库里不是在线
            List<ParkDevice> onlineDeviceList = parkDevices.stream()
                    .filter(parkDevice -> listMap.get(parkDevice.getSerialNumber()) != null)
                    .filter(parkDevice -> listMap.get(parkDevice.getSerialNumber()) >= now - 60)
                    .filter(parkDevice -> !Integer.valueOf(1).equals(parkDevice.getStatus()))
                    .collect(Collectors.toList());

            //实际离线设备
            List<ParkDevice> offlineDeviceList = parkDevices.stream()
                    .filter(parkDevice -> listMap.get(parkDevice.getSerialNumber()) != null)
                    .filter(parkDevice -> now - listMap.get(parkDevice.getSerialNumber()) >= DCTimeOutConstants.OFF_LINE_TIME)
                    .filter(parkDevice -> !Integer.valueOf(2).equals(parkDevice.getStatus()))
                    .collect(Collectors.toList());
            offlineCount += offlineDeviceList.size();
            updateDeviceList.forEach(device -> {
                device.setEndUpdatetime(new Date(listMap.get(device.getSerialNumber()) * 1000));
                boolean offlineCondition = now - listMap.get(device.getSerialNumber()) >= DCTimeOutConstants.OFF_LINE_TIME;
                boolean onlineCondition = listMap.get(device.getSerialNumber()) >= now - 60;
                device.setStatus(offlineCondition ? 2 : (onlineCondition ? 1 : device.getStatus()));
                device.setDelFlag(null);
            });
            offlineDeviceList.forEach(device -> {
                device.setEndUpdatetime(new Date(listMap.get(device.getSerialNumber()) * 1000));
                boolean offlineCondition = now - listMap.get(device.getSerialNumber()) >= DCTimeOutConstants.OFF_LINE_TIME;
                boolean onlineCondition = listMap.get(device.getSerialNumber()) >= now - 60;
                device.setStatus(offlineCondition ? 2 : (onlineCondition ? 1 : device.getStatus()));
                device.setDelFlag(null);
            });
            parkDeviceDaoImpl.updateBatchById(updateDeviceList);
            parkDeviceDaoImpl.updateBatchById(offlineDeviceList);
            if (!onlineDeviceList.isEmpty()) {
                parkDeviceService.batchModifyStatus2(onlineDeviceList, listMap, 1);
            }
            if (!offlineDeviceList.isEmpty()) {
                parkDeviceService.batchModifyStatus2(offlineDeviceList, listMap, 2);
                offlineDeviceList.forEach(device -> {
                    if (deviceType == 1) {
                        //清除缓存中的端云相机连接信息
                        cacheHandle.closeForClearCache(device.getSerialNumber());
                    } else if (deviceType == 5) {
                        //清除缓存中的机器人连接信息
                        cacheHandle.closeForRobotClearCache(device.getSerialNumber());
                    } else if (deviceType == 8) {
                        //清除缓存中的立柱连接信息
                        itcCacheHandle.closeForClearCache(device.getSerialNumber());
                    } else if (deviceType == 9) {
                        //清除缓存中的立柱连接信息
                        morCacheHandle.closeForClearCache(device.getSerialNumber());
                    }
                });
            }
        }
        log.warn("[设备心跳检测] 本次检测到新增离线设备数为[{}]", offlineCount);
        if (offlineCount >= DEVICE_THRESHOLD) {
            log.warn("[监控埋点] 本次检测到新增离线设备数为[{}], 超过了阈值[{}]", offlineCount, DEVICE_THRESHOLD);
        }
    }

    private void batchUpdateDeviceTime(int deviceType) {
        String key = "";
        if (deviceType == 8) {
            key = RedisConstants.ITC_HEARTBEAT_TIME;
        } else if (deviceType == 9) {
            key = RedisConstants.MOR_HEARTBEAT_TIME;
        }
        Map<String, Long> listMap = redisUtils.hGetAll(key, Long.class);
        Set<String> sns = listMap.keySet();
        if (CollectionUtils.isEmpty(sns)) {
            return;
        }
        List<String> allSnList = new ArrayList<>(sns);
        int updateCount = 0;
        for(int i = 0; i < allSnList.size(); i += batchSize) {
            int end = i + batchSize;
            if(end > allSnList.size()) {
                end = allSnList.size();
            }
            List<String> snList = allSnList.subList(i, end);
            List<ParkDevice> parkDevices = parkDeviceDao.selectListBySns(snList, deviceType);
            //时间发生变化的设备
            List<ParkDevice> updateDeviceList = parkDevices.stream()
                    .filter(parkDevice -> {
                        Long redisTime = listMap.get(parkDevice.getSerialNumber());
                        return redisTime != null && (parkDevice.getEndUpdatetime() == null ||
                                parkDevice.getEndUpdatetime().getTime() / 1000 != redisTime);
                    }).collect(Collectors.toList());
            updateDeviceList = updateDeviceList.stream().map(parkDevice -> {
                ParkDevice parkDevice1 = new ParkDevice();
                parkDevice1.setId(parkDevice.getId());
                parkDevice1.setEndUpdatetime(new Date(listMap.get(parkDevice.getSerialNumber()) * 1000));
                return parkDevice1;
            }).collect(Collectors.toList());
            parkDeviceDaoImpl.updateBatchById(updateDeviceList);
            updateCount += updateDeviceList.size();
        }
        log.warn("[更新最后连接时间] total[{}], update[{}]", sns, updateCount);
    }
    /**
     * 批量更新空车位
     */
    private void batchUpdateFreeSpace() {
        Map<String, Integer> listMap = redisUtils.hGetAll(RedisConstants.PNC_FREESPACE, Integer.class);
        if (listMap.size() == 0) {
            return;
        }
        List<String> parkIds = new ArrayList<>(listMap.keySet());
        for(int i = 0; i < parkIds.size(); i += batchSize) {
            int end = i + batchSize;
            if(end > parkIds.size()) {
                end = parkIds.size();
            }
            List<String> parkList = parkIds.subList(i, end);
            List<ParkFreespace> freeSpaceList = parkList.stream().map(parkId -> {
                ParkFreespace parkFreespace = new ParkFreespace();
                parkFreespace.setParkId(Long.valueOf(parkId));
                Integer freeSpace = Math.max(NumberUtils.toPrimitive(listMap.get(parkId)), 0);
                parkFreespace.setFreeSpace(freeSpace);
                parkFreespace.setRealFreeSpace(NumberUtils.toPrimitive(listMap.get(parkId)));
                return parkFreespace;
            }).collect(Collectors.toList());
            parkFreeSpaceService.updateBatchByParkId(freeSpaceList);
        }
    }
}
