package com.icetech.park.service.impl;

import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.enumeration.SwitchTypeEnum;
import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.cloudcenter.domain.request.p2c.LcdHintRequest;
import com.icetech.cloudcenter.domain.response.p2c.CarEnexResponse;
import com.icetech.cloudcenter.domain.vo.p2c.ParkConnectedDeviceVo;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.service.down.p2c.impl.HintServiceImpl;
import com.icetech.park.service.down.p2c.impl.LcdHintServiceImpl;
import com.icetech.park.service.down.p2c.impl.P2cRemoteSwitchServiceImpl;
import com.icetech.park.handle.CacheHandle;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;

/**
 * 双摄像机数据整合业务
 * @author fangct
 */
@Service
@Slf4j
public class DualCameraV2ServiceImpl {
    @Resource
    private CacheHandle cacheHandle;
    @Resource
    private P2cRemoteSwitchServiceImpl remoteSwitchService;
    @Resource
    private HintServiceImpl hintService;
    @Resource
    private LcdHintServiceImpl lcdHintService;
    @Resource
    private ParkService parkService;

    /**
     * 后置处理
     * @param carEnexResponse 响应参数
     * @param parkId 车场ID
     * @param parkCode 车场编号
     * @param serialNumber 当前上报事件的相机SN
     * @return 响应给从相机的结果
     */
    @Async
    public void afterHandle(CarEnexResponse carEnexResponse, Long parkId, String parkCode,
                            String channelCode, Integer enexType, String plateNum,
                            Integer openFlagRequest, String serialNumber) {
        if (carEnexResponse == null) {
            return;
        }
        //只对入场上报和离场上报做处理，其他接口直接返回
        List<ParkConnectedDeviceVo> connectedList = cacheHandle.getConnectedList(parkCode, channelCode);
        if (CollectionUtils.isEmpty(connectedList) || connectedList.size() < 2) {
            return;
        }
        ParkConnectedDeviceVo vo = connectedList.stream()
                .filter(parkConnectedDeviceVo -> enexType.equals(parkConnectedDeviceVo.getInandoutType()))
                .filter(parkConnectedDeviceVo -> !serialNumber.equals(parkConnectedDeviceVo.getDeviceNo()))
                .findFirst().orElse(null);
        if (vo == null) {
            return;
        }
        //当前通道的另一个SN
        String anotherSn = vo.getDeviceNo();
        log.info("[双摄业务] 通道[{}]上报了进出事件, 另一个设备SN[{}]", channelCode, anotherSn);
        Integer openFlag = carEnexResponse.getOpenFlag();
        Integer showType = carEnexResponse.getShowType();
        String show = carEnexResponse.getShow();
        String say = carEnexResponse.getSay();
        TokenDeviceVo deviceInfo = cacheHandle.getDeviceInfo(anotherSn);
        if (deviceInfo == null) {
            log.info("[双摄业务] 设备不在线, anotherSn[{}]", anotherSn);
            return;
        }
        if (NumberUtils.toPrimitive(openFlagRequest) == 1 || NumberUtils.toPrimitive(openFlag) == 1){
            //开闸
            String msgId = remoteSwitchService.open(SwitchTypeEnum.开闸.getType(), parkId, parkCode, anotherSn, plateNum);
            if (msgId == null) {
                log.info("[双摄业务] 开闸指令下发失败, plateNum[{}]", plateNum);
            }
        }
        if (StringUtils.isNotBlank(show) || StringUtils.isNotBlank(say)){
            Integer showDeviceType = carEnexResponse.getShowDeviceType();
            if (showDeviceType == null) {
                ObjectResponse<ParkInoutdevice> channelResp = parkService.getInOutDeviceByCode(parkId, channelCode);
                ParkInoutdevice channel = channelResp.getData();
                showDeviceType = channel.getLedLcdSource();
            }
            if (NumberUtils.toPrimitive(showDeviceType) == 2) {
                //LCD语音显示
                LcdHintRequest hintRequest = new LcdHintRequest();
                hintRequest.setShow(show);
                hintRequest.setSay(say);
                hintRequest.setPlateNum(plateNum);
                hintRequest.setQrCodeUrl(carEnexResponse.getQrCodeUrl());
                String msgId = lcdHintService.showAndSay(parkId, parkCode, anotherSn, hintRequest);
                if (msgId == null) {
                    log.info("[双摄业务] 语音显示指令下发失败, 参数[{}]", hintRequest);
                }
            } else {
                //语音显示
                HintRequest hintRequest = new HintRequest();
                hintRequest.setShow(show);
                hintRequest.setSay(say);
                hintRequest.setPlateNum(plateNum);
                hintRequest.setShowType(showType);
                hintRequest.setExtendShow(carEnexResponse.getExtendShow());
                String msgId = hintService.showAndSay(parkId, parkCode, anotherSn, hintRequest);
                if (msgId == null) {
                    log.info("[双摄业务] 语音显示指令下发失败, 参数[{}]", hintRequest);
                }
            }
        }
    }

    @Async
    public void lcdHint(Long parkId, String parkCode, String channelCode,
                        String serialNumber, LcdHintRequest hintRequest) {
        ParkConnectedDeviceVo vo = anotherCamera(parkCode, channelCode, serialNumber);
        if (vo == null) {
            return;
        }
        lcdHintService.showAndSay(parkId, parkCode, vo.getDeviceNo(), hintRequest);
    }

    @Async
    public void ledHint(Long parkId, String parkCode, String channelCode,
                        String serialNumber, HintRequest hintRequest) {
        ParkConnectedDeviceVo vo = anotherCamera(parkCode, channelCode, serialNumber);
        if (vo == null) {
            return;
        }
        hintService.showAndSay(parkId, parkCode, vo.getDeviceNo(), hintRequest);
    }

    @Async
    public void remoteSwitch(Long parkId, String parkCode, String channelCode,
                        String serialNumber, HintRequest hintRequest) {
        ParkConnectedDeviceVo vo = anotherCamera(parkCode, channelCode, serialNumber);
        if (vo == null) {
            return;
        }
        hintService.showAndSay(parkId, parkCode, vo.getDeviceNo(), hintRequest);
    }

    private ParkConnectedDeviceVo anotherCamera(String parkCode, String channelCode, String serialNumber) {
        //只对入场上报和离场上报做处理，其他接口直接返回
        List<ParkConnectedDeviceVo> connectedList = cacheHandle.getConnectedList(parkCode, channelCode);
        if (CollectionUtils.isEmpty(connectedList) || connectedList.size() < 2) {
            return null;
        }
        return connectedList.stream()
                .filter(parkConnectedDeviceVo -> !serialNumber.equals(parkConnectedDeviceVo.getDeviceNo()))
                .findFirst().orElse(null);
    }

}
