package com.icetech.park.service.impl;

import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.common.utils.FixSizeLinkedList;
import com.icetech.cloudcenter.domain.response.p2c.CarEnexResponse;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.dao.ChannelDualcameraDao;
import com.icetech.park.domain.entity.ChannelDualcamera;
import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.cloudcenter.domain.vo.p2c.ChannelRecentPlateNumsVo;
import com.icetech.cloudcenter.domain.enumeration.SwitchTypeEnum;
import com.icetech.park.service.down.p2c.impl.HintServiceImpl;
import com.icetech.park.service.down.p2c.impl.P2cRemoteSwitchServiceImpl;
import com.icetech.park.handle.CacheHandle;
import com.icetech.third.utils.RedisUtils;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 双摄像机数据整合业务
 * @author fangct
 */
@Service
@Slf4j
public class DualCameraServiceImpl {
    @Autowired
    private ChannelDualcameraDao channelDualcameraDao;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private P2cRemoteSwitchServiceImpl remoteSwitchService;
    @Autowired
    private HintServiceImpl hintService;
    @Autowired
    private ParkService parkService;
    /**
     * 间隔查询时间，50ms
     */
    private static final int INTERVAL = 50;

    /**
     * 前置处理
     * @param p2CBaseRequest
     * @param parkId
     * @param parkCode
     * @param deviceNo
     * @param inandoutCode
     * @param inandoutType
     * @param isMaster
     * @return
     */
    public ResultCode preHandle(P2cBaseRequest p2CBaseRequest, Long parkId, String parkCode, String deviceNo,
                                String inandoutCode, Integer inandoutType, boolean isMaster) {
        /**
         * 接下来，处理进出场的双摄像机数据处理问题
         *  前提：1、当前通道有两个相机在线；
         */
        //查询缓存是不是存在，有则直接返回已经处理
        //保存多相机上报数据
        saveChannelDualCamera(p2CBaseRequest, parkId, inandoutCode, inandoutType, deviceNo, isMaster);
        //判断车牌是否已经处理过
        ChannelRecentPlateNumsVo channelRecentPlateNumsVo = resolveParamAndGetChannelPlateNumByinandoutType(p2CBaseRequest, inandoutType);
        FixSizeLinkedList<ChannelRecentPlateNumsVo> msPlateNums = cacheHandle.getMSPlateNums(parkCode, inandoutCode);
        if (msPlateNums != null && msPlateNums.contains(channelRecentPlateNumsVo)){
            log.info("<双摄相机前置处理> 该通道最近上报车牌中已处理过该车牌，车牌号：{}", channelRecentPlateNumsVo.getPlateNum());
            throw new ResponseBodyException(CodeConstants.ERROR_405, "此车牌已经处理过");
        }

        int WAIT_TIME = 500;
        int SHIBIE_BETWEEN_TIME = 2000;
        ObjectResponse<ParkConfig> configObjectResponse = parkService.getParkConfig(parkId);
        if (ObjectResponse.isSuccess(configObjectResponse)){
            if (configObjectResponse.getData().getDualcameraTime() != null){
                WAIT_TIME = configObjectResponse.getData().getDualcameraTime();
            }
            if (configObjectResponse.getData().getDualcameraEnextime() != null){
                SHIBIE_BETWEEN_TIME = configObjectResponse.getData().getDualcameraEnextime();
            }
        }
        SHIBIE_BETWEEN_TIME = SHIBIE_BETWEEN_TIME / 1000;
        String slave_key = RedisConstants.SLAVE_CHANNEL_PROFILE + parkCode + inandoutCode;
        String master_key = RedisConstants.MASTER_CHANNEL_PROFILE + parkCode + inandoutCode;
        if (!isMaster) {//从相机时，放入当前通道待处理缓存，等待主相机循环获取
            redisUtils.setExpireMilliSeconds(slave_key, p2CBaseRequest, (long)(WAIT_TIME));
            if (redisUtils.exists(master_key)){
                log.info("<双摄相机前置处理> 主相机已上报，从相机无需处理，车牌号：{}", channelRecentPlateNumsVo.getPlateNum());
                throw new ResponseBodyException(CodeConstants.ERROR_406, "主相机已上报，从相机无需处理");
            }
        }else{
            redisUtils.setExpireMilliSeconds(master_key, p2CBaseRequest, (long)(WAIT_TIME));
        }
        /**
         * 等待另一相机上报的条件
         * 1 当前相机是主相机
         * 2 当前相机是从相机，且此时主相机未上报
         */
        P2cBaseRequest otherCameraData = getOtherCameraData(parkCode, inandoutCode, isMaster, WAIT_TIME);
        /**
         * 判断另一相机是否上报，判断完后，保存到主从缓存，超过时效后另一相机再上报，直接返回 ERROR_405 重复请求
         */
        cacheHandle.addMSPlateNums(parkCode, inandoutCode, channelRecentPlateNumsVo);
        if (otherCameraData == null){
            //单位时间内，只有一个相机上报
            if (isMaster){
                //只有主相机上报时，则主相机正常处理
                return ResultCode.主相机识别_从相机未识别;
            }else{
                //只有从相机上报时，判断主相机是否在线，如果不在线，则由从相机处理，否则由主相机处理
                //此处只处理主相机在线的，因为程序能到此处，一定是该通道有两个或以上相机在线
                return ResultCode.从相机识别_主相机在线;
            }
        }else{//单位时间内，主从相机都上报了
            if (!isMaster){
                throw new ResponseBodyException(CodeConstants.ERROR_406, "主从相机都上报，从相机无需处理");
            }
            ResultCode resultCode = ResultCode.主从都识别_未开闸;
            if (inandoutType == 1){//入场处理
                //当前相机上报
                CarEnterRequest enterRequest = (CarEnterRequest) p2CBaseRequest.getBizContent();
                //另一相机上报
                CarEnterRequest otherEnterRequest = JsonUtils.convert2bean(otherCameraData.getBizContent(), CarEnterRequest.class);

                if (enterRequest.getOpenFlag() == 1 || otherEnterRequest.getOpenFlag() == 1){
                    resultCode = ResultCode.主从都识别_已开闸;
                }
                /**
                 * 车牌号相同时，取置信度高的
                 */
                if (enterRequest.getPlateNum().equals(otherEnterRequest.getPlateNum())){
                    if (enterRequest.getReliability() < otherEnterRequest.getReliability()) {
                        p2CBaseRequest.setBizContent(otherEnterRequest);
                    } else if (enterRequest.getReliability().equals(otherEnterRequest.getReliability())) {
                        //取最新入场的
                        if (enterRequest.getEnterTime() < otherEnterRequest.getEnterTime()) {
                            p2CBaseRequest.setBizContent(otherEnterRequest);
                        }
                    }
                }else{
                    //以开闸的为准
                    if (enterRequest.getCarType().equals(PlateTypeEnum.临时车.getType()) && !otherEnterRequest.getCarType().equals(PlateTypeEnum.临时车.getType())){
                        p2CBaseRequest.setBizContent(otherEnterRequest);
                    } else {
                        //两次数据的入场时间是否在设置范围内
                        if (Math.abs(enterRequest.getEnterTime() - otherEnterRequest.getEnterTime()) <= SHIBIE_BETWEEN_TIME) {
                            //取置信度高的
                            if (enterRequest.getReliability() < otherEnterRequest.getReliability()) {
                                p2CBaseRequest.setBizContent(otherEnterRequest);
                            } else if (enterRequest.getReliability().equals(otherEnterRequest.getReliability())) {
                                //取最新入场的
                                if (enterRequest.getEnterTime() < otherEnterRequest.getEnterTime()) {
                                    p2CBaseRequest.setBizContent(otherEnterRequest);
                                }
                            }
                        } else {
                            //取最新入场的
                            if (enterRequest.getEnterTime() < otherEnterRequest.getEnterTime()) {
                                p2CBaseRequest.setBizContent(otherEnterRequest);
                            }
                        }
                    }
                }
            }else{//出场处理
                CarExitRequest exitRequest = (CarExitRequest) p2CBaseRequest.getBizContent();
                CarExitRequest otherExitRequest = JsonUtils.convert2bean(otherCameraData.getBizContent(), CarExitRequest.class);

                if (exitRequest.getOpenFlag() == 1 || otherExitRequest.getOpenFlag() == 1){
                    resultCode = ResultCode.主从都识别_已开闸;
                }
                /**
                 * 车牌号相同时，取置信度高的
                 */
                if (exitRequest.getPlateNum().equals(otherExitRequest.getPlateNum())){
                    if (exitRequest.getReliability() < otherExitRequest.getReliability()) {
                        p2CBaseRequest.setBizContent(otherExitRequest);
                    } else if (exitRequest.getReliability().equals(otherExitRequest.getReliability())) {
                        //取最新离场的
                        if (exitRequest.getExitTime() < otherExitRequest.getExitTime()) {
                            p2CBaseRequest.setBizContent(otherExitRequest);
                        }
                    }
                }else{
                    //以非临时车的为准
                    if (exitRequest.getCarType().equals(PlateTypeEnum.临时车.getType()) && !otherExitRequest.getCarType().equals(PlateTypeEnum.临时车.getType())){
                        p2CBaseRequest.setBizContent(otherExitRequest);
                    } else {
                        //两次数据的离场时间是否在设置范围内
                        if (Math.abs(exitRequest.getExitTime() - otherExitRequest.getExitTime()) <= SHIBIE_BETWEEN_TIME) {
                            //取置信度高的
                            if (exitRequest.getReliability() < otherExitRequest.getReliability()) {
                                p2CBaseRequest.setBizContent(otherExitRequest);
                            } else if (exitRequest.getReliability().equals(otherExitRequest.getReliability())) {
                                //取最新离场的
                                if (exitRequest.getExitTime() < otherExitRequest.getExitTime()) {
                                    p2CBaseRequest.setBizContent(otherExitRequest);
                                }
                            }
                        } else {
                            //取最新离场的
                            if (exitRequest.getExitTime() < otherExitRequest.getExitTime()) {
                                p2CBaseRequest.setBizContent(otherExitRequest);
                            }
                        }
                    }
                }
            }
            return resultCode;
        }
    }
    /**
     * 后置处理
     * @param p2cBaseResponse
     * @param parkId
     * @param parkCode
     * @param inandoutCode
     * @param p2cBaseRequest 请求参数
     * @param resultCode
     * @return 响应给从相机的结果
     */
    public P2cBaseResponse afterHandle(P2cBaseResponse p2cBaseResponse, Long parkId, String parkCode, String inandoutCode, Integer inandoutType, P2cBaseRequest p2cBaseRequest, ResultCode resultCode) {
        CarEnexResponse carEnexResponse = (CarEnexResponse) p2cBaseResponse.getData();
        Integer openFlag = carEnexResponse.getOpenFlag();
        Integer showType = carEnexResponse.getShowType();
        String show = carEnexResponse.getShow();
        String say = carEnexResponse.getSay();
        if (resultCode.equals(ResultCode.主从都识别_已开闸)){
            CarEnexResponse response = (CarEnexResponse) p2cBaseResponse.getData();
            response.setOpenFlag(0);
            response.setShow(null);
            response.setSay(null);
            p2cBaseResponse.setData(response);
            return p2cBaseResponse;
        }
        String serialNumber = cacheHandle.getSerialNumber(parkCode, inandoutCode);
        String plateNum = null;
        if (inandoutType == 1){
            CarEnterRequest enterRequest = JsonUtils.convert2bean(p2cBaseRequest.getBizContent(), CarEnterRequest.class);
            plateNum = enterRequest.getPlateNum();
        }else{
            CarExitRequest exitRequest = JsonUtils.convert2bean(p2cBaseRequest.getBizContent(), CarExitRequest.class);
            plateNum = exitRequest.getPlateNum();
        }
        //主相机是否处理过，默认为没有处理
        boolean isMasterDone = false;
        if (openFlag != null && openFlag == 1){
            /**
             * 开闸
             */
            String msgId = remoteSwitchService.open(SwitchTypeEnum.开闸.getType(), parkId, parkCode, serialNumber, plateNum);
            if (msgId == null) {
                log.info("<平台离场> 开闸指令下发失败");
            }else{
                isMasterDone = true;
            }
        }
        if (StringUtils.isNotBlank(show) || StringUtils.isNotBlank(say)){
            /**
             * 语音显示
             */
            HintRequest hintRequest = new HintRequest();
            hintRequest.setShow(show);
            hintRequest.setSay(say);
            hintRequest.setPlateNum(plateNum);
            hintRequest.setShowType(showType);
            String msgId = hintService.showAndSay(parkId, parkCode, serialNumber, hintRequest);
            if (msgId == null) {
                log.info("<平台离场> 语音显示指令下发失败");
            }else{
                isMasterDone = true;
            }
        }
        if (isMasterDone){
            /**
             * 返回从相机默认的结果
             */
            CarEnexResponse response = (CarEnexResponse) p2cBaseResponse.getData();
            response.setOpenFlag(0);
            response.setShow(null);
            response.setSay(null);
            p2cBaseResponse.setData(response);
            return p2cBaseResponse;
        }else{
            return p2cBaseResponse;
        }
    }
    /**
     * 保存主从相机上报的数据
     * @param p2CBaseRequest
     * @param parkId
     * @param inandoutCode
     * @param inandoutType
     * @param serialNumber
     * @param isMaster
     */
    private void saveChannelDualCamera(P2cBaseRequest p2CBaseRequest, Long parkId, String inandoutCode, int inandoutType, String serialNumber, boolean isMaster) {
        ChannelDualcamera channelDualcamera = new ChannelDualcamera();
        channelDualcamera.setParkId(parkId);
        channelDualcamera.setEnexType(inandoutType);
        channelDualcamera.setChannelCode(inandoutCode);
        channelDualcamera.setMsType(isMaster);
        channelDualcamera.setSerialNumber(serialNumber);
        channelDualcamera.setParameter(JsonUtils.toString(p2CBaseRequest));
        channelDualcameraDao.insert(channelDualcamera);
    }

    /**
     * 解析bizContent参数并赋值，并返回入口出口上报的公共参数
     * @param p2CBaseRequest
     * @param inandoutType
     * @return
     */
    private ChannelRecentPlateNumsVo resolveParamAndGetChannelPlateNumByinandoutType(P2cBaseRequest p2CBaseRequest, int inandoutType) {
        ChannelRecentPlateNumsVo channelRecentPlateNumsVo = new ChannelRecentPlateNumsVo();
        if (inandoutType == 1){
            CarEnterRequest enterRequest = JsonUtils.convert2bean(p2CBaseRequest.getBizContent(), CarEnterRequest.class);
            channelRecentPlateNumsVo.setEnexType(inandoutType);
            channelRecentPlateNumsVo.setEnexTime(enterRequest.getEnterTime());
            channelRecentPlateNumsVo.setPlateNum(enterRequest.getPlateNum());
            p2CBaseRequest.setBizContent(enterRequest);
        }else{
            CarExitRequest exitRequest = JsonUtils.convert2bean(p2CBaseRequest.getBizContent(), CarExitRequest.class);
            channelRecentPlateNumsVo.setEnexType(inandoutType);
            channelRecentPlateNumsVo.setEnexTime(exitRequest.getExitTime());
            channelRecentPlateNumsVo.setPlateNum(exitRequest.getPlateNum());
            p2CBaseRequest.setBizContent(exitRequest);
        }
        return channelRecentPlateNumsVo;
    }

    /**
     * 等待另一相机上报的数据
     * @param parkCode
     * @param inandoutCode
     * @param isMaster
     * @return
     */
    public P2cBaseRequest getOtherCameraData(String parkCode, String inandoutCode, boolean isMaster, int WAIT_TIME){
        String key = null;
        if (isMaster){
            key = RedisConstants.SLAVE_CHANNEL_PROFILE + parkCode + inandoutCode;
        }else{
            key = RedisConstants.MASTER_CHANNEL_PROFILE + parkCode + inandoutCode;
        }
        Long currentTimeMillis = System.currentTimeMillis();
        Long lastTime = currentTimeMillis + WAIT_TIME;
        int n = 1;
        //查询redis中是否返回
        while (lastTime > currentTimeMillis) {
            if (redisUtils.exists(key)){
                P2cBaseRequest data = redisUtils.get(key, P2cBaseRequest.class);
                log.info("第{}次从redis中读取到了{}相机的上报信息，车场编号：{}，通道编号：{}", n , isMaster ? "主" : "从", parkCode, inandoutCode);
                return data;
            }
            try {
                Thread.sleep(INTERVAL);
            } catch (InterruptedException e) {
                log.warn(String.valueOf(e.getMessage()), e);
            }
            currentTimeMillis = System.currentTimeMillis();
            n ++;
        }
        log.info("{}相机未等到另一相机的上报信息，车场编号：{}，通道编号：{}", isMaster ? "主" : "从", parkCode, inandoutCode);
        return null;
    }
    @Setter
    @Getter
    @ToString
    public static class FlowRet{
        private ResultCode resultCode;
        private Object para;
    }
    @ToString
    public enum ResultCode{
        无需处理(0),
        主相机识别_从相机未识别(1),
        主从都识别_未开闸(2),
        主从都识别_已开闸(3),
        从相机识别_主相机在线(4),
        从相机识别_主相机不在线(5),
        ;
        private Integer code;
        private ResultCode(int code){
            this.code = code;
        }
    }
}
