package com.icetech.park.service.impl;

import com.fasterxml.jackson.core.type.TypeReference;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.domain.SendMessage;
import com.icetech.basics.constants.TextConstant;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.third.utils.JsonUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.validator.Validator;
import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.enumeration.DataCollectionEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.factory.SendServiceFactory;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;

import static com.icetech.basics.constants.TextConstant.*;

/**
 * 统一下发实现
 *
 * @author fangct
 */
@Slf4j
@Service
public class DownSendServiceImpl implements DownSendService {
    @Autowired
    private ParkService parkService;
    @Autowired
    private RedisUtils redisUtils;

    @Override
    public ObjectResponse<?> send(SendRequest sendRequest) {
        try {
            if (!Validator.validate(sendRequest)) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage("", "下发参数校验失败"));
            }
        } catch (Exception e) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage("", "下发参数校验失败"));
        }
        Long parkId = sendRequest.getParkId();
        ObjectResponse<ParkConfig> parkConfigResponse = parkService.getParkConfig(parkId);
        if (!ObjectResponse.isSuccess(parkConfigResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage("", "下发失败，车场配置未找到"));
        }
        ParkConfig parkConfig = parkConfigResponse.getData();
        if (parkConfig.getDataCollection() == null) {
            log.warn("<下发业务> 车场数据采集方式未配置，参数：{}", sendRequest);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage("", "车场数据采集方式未配置"));
        }
        /*
         * 端云架构
         */
        if (parkConfig.getDataCollection().equals(DataCollectionEnum.端云.getType())) {
            return p2cDown(sendRequest, parkId);
        } else if (parkConfig.getDataCollection().equals(DataCollectionEnum.纯云.getType())) {
            return fullCloudDown(sendRequest, parkId);
        } else {
            return pncDown(sendRequest);
        }
    }

    private static ObjectResponse<?> pncDown(SendRequest sendRequest) {
        // 端网云车场下发
        DownSendService downSendService;
        try {
            downSendService = SendServiceFactory.createSendService(sendRequest.getServiceType());
        }catch (Exception e) {
            log.warn("端网云未找到具体的serviceBean信息" );
            return ObjectResponse.failed(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(TWO, "无需下发"));
        }
        ObjectResponse<?> response;
        try {
            response = downSendService.send(sendRequest);
            if (!CodeConstants.SUCCESS.equals(response.getCode())
                    && !CodeConstants.ERROR_12002.equals(response.getCode())) {
                SendMessage sendMessage;
                try {
                    JsonUtils.fromJson(response.getMsg(), new TypeReference<SendMessage>() {
                    });
                    return response;
                } catch (Exception e) {
                    sendMessage = SendMessage.builder()
                            .type(DataCollectionEnum.端网云.getType().toString())
                            .describes(Collections.singletonList(SendMessage.Describe.builder()
                                    .failType(DEFAULT)
                                    .fixedDisplay(response.getMsg())
                                    .build()))
                            .build();
                    String msg = JsonUtils.toJson(sendMessage);
                    response.setMsg(msg);
                    return response;
                }
            }
        } catch (ResponseBodyException e) {
            log.warn("端网云下发失败信息-{}", e.getMessage());
            return ObjectResponse.failed(e.getErrCode(), TextConstant.getDefaultMessage(TWO, e.getMessage() == null ? "下发失败" : e.getMessage()));
        }catch (Exception e) {
            log.error("端网云下发失败信息, e=", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, e.getMessage() == null ? "下发失败" : e.getMessage()));
        }
        return response;
    }

    private ObjectResponse<?> fullCloudDown(SendRequest sendRequest, Long parkId) {
        //纯云车场下发
        ObjectResponse<Park> parkResponse = parkService.findByParkId(parkId);
        if (!ObjectResponse.isSuccess(parkResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, "车场数据未找到"));
        }
        Park park = parkResponse.getData();
        DownService<?, ?> sendService;
        try {
            sendService = SendServiceFactory.createFullCloudSendService(sendRequest.getServiceType());
        }catch (Exception e) {
            return ObjectResponse.failed(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "无需下发"));
        }
        Integer batchServiceType = sendRequest.getServiceType();
        if (redisUtils.exists(RedisConstants.BATCH_DOWN_OPEN_FLAG + parkId + ":" + batchServiceType)) {
            return ObjectResponse.failed(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "批量下发中"));
        }
        ObjectResponse<?> response;
        try {
            response = sendService.send(sendRequest, park.getParkCode());
            if (!CodeConstants.SUCCESS.equals(response.getCode())
                    && !CodeConstants.ERROR_12002.equals(response.getCode())) {
                SendMessage sendMessage;
                try {
                    JsonUtils.fromJson(response.getMsg(), new TypeReference<SendMessage>() {
                    });
                    return response;
                } catch (Exception e) {
                    sendMessage = SendMessage.builder()
                            .type(DataCollectionEnum.纯云.getType().toString())
                            .describes(Collections.singletonList(SendMessage.Describe.builder()
                                    .failType(DEFAULT)
                                    .fixedDisplay(response.getMsg() == null ? "下发失败" : response.getMsg())
                                    .build()))
                            .build();
                    String msg = JsonUtils.toJson(sendMessage);
                    response.setMsg(msg);
                }
            }
        } catch (ResponseBodyException e) {
            log.warn("纯云下发失败信息-{}", e.getMessage());
            return ObjectResponse.failed(e.getErrCode(), TextConstant.getDefaultMessage(THREE, e.getMessage() == null ? "下发失败" : e.getMessage()));
        }catch (Exception e) {
            log.error("纯云下发失败信息", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, e.getMessage() == null ? "下发失败" : e.getMessage()));
        }
        return response;
    }

    private ObjectResponse<?> p2cDown(SendRequest sendRequest, Long parkId) {
        //端云车场下发
        ObjectResponse<Park> parkResponse = parkService.findByParkId(parkId);
        if (!ObjectResponse.isSuccess(parkResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(ONE, "车场数据未找到"));
        }
        Park park = parkResponse.getData();
        DownService<?, ?> sendService;
        try {
            sendService = SendServiceFactory.createP2cSendService(sendRequest.getServiceType());
        }catch (Exception e) {
            return ObjectResponse.failed(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(ONE, "无需下发"));
        }
        Integer batchServiceType = sendRequest.getServiceType();
        if (P2cDownCmdEnum.通道权限下发.getCmdType().equals(sendRequest.getServiceType())) {
            batchServiceType = P2cDownCmdEnum.车场权限下发.getCmdType();
        }
        if (redisUtils.exists(RedisConstants.BATCH_DOWN_OPEN_FLAG + parkId + ":" + batchServiceType)) {
            return ObjectResponse.failed(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(ONE, "批量下发中"));
        }
        ObjectResponse<?> response;
        try {
            response = sendService.send(sendRequest, park.getParkCode());
            if (!CodeConstants.SUCCESS.equals(response.getCode())
                    && !CodeConstants.ERROR_12002.equals(response.getCode())) {
                SendMessage sendMessage;
                try {
                    JsonUtils.fromJson(response.getMsg(), new TypeReference<SendMessage>() {
                    });
                    return response;
                } catch (Exception e) {
                    sendMessage = SendMessage.builder()
                            .type(DataCollectionEnum.端云.getType().toString())
                            .describes(Collections.singletonList(SendMessage.Describe.builder()
                                    .failType(DEFAULT)
                                    .fixedDisplay(response.getMsg() == null ? "下发失败" : response.getMsg())
                                    .build()))
                            .build();
                    String msg = JsonUtils.toJson(sendMessage);
                    response.setMsg(msg);
                }
            }
        } catch (ResponseBodyException e) {
            log.warn("端云下发失败信息-{}", e.getMessage());
            return ObjectResponse.failed(e.getErrCode(), TextConstant.getDefaultMessage(ONE, e.getMessage() == null ? "下发失败" : e.getMessage()));
        }catch (Exception e) {
            log.error("端云下发失败信息", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(ONE, e.getMessage() == null ? "下发失败" : e.getMessage()));
        }
        return response;
    }
}
