package com.icetech.park.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.icetech.cloudcenter.api.IDeviceManageService;
import com.icetech.basics.api.DictionaryItemService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.park.dao.FileDao;
import com.icetech.park.dao.FirmwareDao;
import com.icetech.park.dao.FirmwareUpgradeDao;
import com.icetech.park.domain.entity.File;
import com.icetech.park.domain.entity.Firmware;
import com.icetech.park.domain.entity.FirmwareUpgrade;
import com.icetech.park.domain.dto.FirmwareDTO;
import com.icetech.park.domain.dto.FirmwareListDTO;
import com.icetech.park.domain.dto.FirmwareUpgradeListDTO;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.request.BatchUpgradeRequest;
import com.icetech.cloudcenter.domain.request.DeviceUpgradeRequest;
import com.icetech.cloudcenter.domain.request.FirmwareListRequest;
import com.icetech.cloudcenter.domain.request.SaveFirmwareRequest;
import com.icetech.cloudcenter.domain.request.UpgradeLogListRequest;
import com.icetech.cloudcenter.domain.request.p2c.UpgradeRequest;
import com.icetech.cloudcenter.domain.response.BatchUpgradeValidateResponse;
import com.icetech.cloudcenter.domain.response.DeviceListDTO;
import com.icetech.basics.domain.dto.DictionaryItemDto;
import com.icetech.cloudcenter.domain.response.FirmwareListResponse;
import com.icetech.cloudcenter.domain.response.SupportUpgradeDeviceResponse;
import com.icetech.cloudcenter.domain.response.UpgradeLogListResponse;
import com.icetech.cloudcenter.domain.response.UpgradeProgressResponse;
import com.icetech.cloudcenter.domain.response.ValidateUpgradeResponse;
import com.icetech.cloudcenter.domain.response.ValidateVersionResponse;
import com.icetech.park.service.down.p2c.impl.UpgradeServiceImpl;
import com.icetech.park.service.handle.DeviceUpgradeHandle;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.Page;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.UUIDTools;
import com.icetech.common.validator.Validator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 为外部程序提供接口服务
 */
@Slf4j
@Service
public class DeviceManageServiceImpl implements IDeviceManageService {
    @Autowired
    private UpgradeServiceImpl upgradeService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private FirmwareDao firmwareDao;
    @Autowired
    private FirmwareUpgradeDao firmwareUpgradeDao;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private DictionaryItemService dictionaryItemService;
    @Autowired
    private DeviceUpgradeHandle deviceUpgradeHandle;
    @Autowired
    private FileDao fileDao;
    //数据字典表的typeId
    private int DICTIONARY_TYPEID = 8;

    @Override
    public ObjectResponse upgrade(DeviceUpgradeRequest deviceUpgradeRequest) {
        //校验参数
        boolean validator = Validator.validate(deviceUpgradeRequest);
        if (!validator){
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        ObjectResponse<ParkDevice> deviceObjectResponse = parkDeviceService.getDeviceBySerialNumber(deviceUpgradeRequest.getSn());
        if (!ObjectResponse.isSuccess(deviceObjectResponse)){
            return ObjectResponse.failed(CodeConstants.ERROR_404, "设备不存在");
        }
        ParkDevice parkDevice = deviceObjectResponse.getData();
        Long parkId = (long)parkDevice.getParkId();

        ObjectResponse<ParkInoutdevice> inoutdeviceObjectResponse = parkService.getInoutDeviceById((long) parkDevice.getChannelId());
        if (!ObjectResponse.isSuccess(inoutdeviceObjectResponse)){
            return ObjectResponse.failed(CodeConstants.ERROR_404, "通道不存在");
        }
        ParkInoutdevice parkInoutdevice = inoutdeviceObjectResponse.getData();
        ObjectResponse<List<DictionaryItemDto>> dictionaryItems = dictionaryItemService.findListByTypeId(DICTIONARY_TYPEID);
        if (!ObjectResponse.isSuccess(dictionaryItems)){
            return ObjectResponse.failed(CodeConstants.ERROR_402, "设备型号未配置");
        }
        List<DictionaryItemDto> itemDtos = dictionaryItems.getData().stream().filter(item -> item.getText().equals(parkDevice.getProductModel())).collect(Collectors.toList());
        if (itemDtos == null || itemDtos.size() == 0) {
            return ObjectResponse.failed(CodeConstants.ERROR_402, "不支持升级该型号设备");
        }
        DictionaryItemDto dictionaryItemDto = itemDtos.get(0);
        FirmwareDTO firmwareDTO = firmwareDao.selectByModelAndVer(dictionaryItemDto.getValue(), deviceUpgradeRequest.getNewFirmwareVer());
        if (firmwareDTO == null || firmwareDTO.getFilePath() == null){
            return ObjectResponse.failed(CodeConstants.ERROR_402, "不存在该版本的升级包");
        }

        //判断是否有执行中的任务
        FirmwareUpgrade runningTask = firmwareUpgradeDao.selectTaskBySnAndStatus(deviceUpgradeRequest.getSn(), FirmwareUpgrade.StatusEnum.升级中.status);
        if (runningTask != null){
            Date actionTime = runningTask.getActionTime();
            long now = DateTools.unixTimestamp();
            //判断是否超时响应，超时时间为10分钟
            if (now - actionTime.getTime() / 1000 > 600) {
                runningTask.setStatus(FirmwareUpgrade.StatusEnum.升级失败.status);
                runningTask.setReason("设备响应超时");
                runningTask.setUpdateTime(new Date());
                firmwareUpgradeDao.updateById(runningTask);
            } else {
                return ObjectResponse.failed(CodeConstants.ERROR_402, "当前设备正在升级，请勿重复操作");
            }
        }
        //判断是否有待执行任务
        FirmwareUpgrade waitTask = firmwareUpgradeDao.selectTaskBySnAndStatus(deviceUpgradeRequest.getSn(), FirmwareUpgrade.StatusEnum.待升级.status);
        if (waitTask != null){
            waitTask.setStatus(FirmwareUpgrade.StatusEnum.升级失败.status);
            waitTask.setReason("被新任务取消");
            waitTask.setUpdateTime(new Date());
            firmwareUpgradeDao.updateById(waitTask);
        }

        DateTime actionTime = DateUtil.parseDateTime(deviceUpgradeRequest.getActionTime());
        long now = System.currentTimeMillis();
        //是否立即升级
        boolean isNowAction = true;
        if (actionTime.getTime() > now){
            isNowAction = false;
        }
        //记录升级记录
        FirmwareUpgrade firmwareUpgrade = new FirmwareUpgrade();
        firmwareUpgrade.setParkId(parkId);
        firmwareUpgrade.setSn(deviceUpgradeRequest.getSn());
        firmwareUpgrade.setDeviceType(deviceUpgradeRequest.getDeviceType());
        firmwareUpgrade.setChannelName(parkInoutdevice.getInandoutName());
        firmwareUpgrade.setActionTime(isNowAction ? new Date() : actionTime);
        firmwareUpgrade.setOperAccount(deviceUpgradeRequest.getOperAccount());
        firmwareUpgrade.setProductModel(parkDevice.getProductModel());
        firmwareUpgrade.setSourceVer(parkDevice.getFirmwareVersion());
        firmwareUpgrade.setTargetVer(deviceUpgradeRequest.getNewFirmwareVer());
        firmwareUpgrade.setFileId(firmwareDTO.getFileId());
        firmwareUpgrade.setUpgradeChannel(parkDevice.getStatus() == 3 ? 2 : 1);

        String taskId = UUIDTools.generateShortUuid();
        firmwareUpgrade.setTaskId(taskId);
        firmwareUpgradeDao.insert(firmwareUpgrade);

        Map<String, String> ret = new HashMap<>();
        ret.put("taskId", taskId);
        if (isNowAction){
            UpgradeRequest upgradeRequest = new UpgradeRequest();
            upgradeRequest.setNewFirmwareVer(deviceUpgradeRequest.getNewFirmwareVer());
            upgradeRequest.setSendTime(DateTools.unixTimestamp());
            upgradeRequest.setFirmwarePath(firmwareDTO.getFilePath());
            upgradeRequest.setFirmwareSize(String.valueOf(firmwareDTO.getFileSize()));
            upgradeRequest.setTaskId(taskId);
            upgradeRequest.setUpgradeChannel(firmwareUpgrade.getUpgradeChannel());
            upgradeService.execute(upgradeRequest, deviceUpgradeRequest.getSn(), parkId);
        }
        return ObjectResponse.success(ret);
    }

    @Override
    public ObjectResponse batchUpgrade(BatchUpgradeRequest batchUpgradeRequest) {
        //校验参数
        boolean validator = Validator.validate(batchUpgradeRequest);
        if (!validator){
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        String sns = batchUpgradeRequest.getSns();
        List<String> msgList = new ArrayList<>();
        Arrays.stream(sns.split(",")).forEach(sn ->{
            DeviceUpgradeRequest deviceUpgradeRequest = new DeviceUpgradeRequest();
            BeanUtil.copyProperties(batchUpgradeRequest, deviceUpgradeRequest);
            deviceUpgradeRequest.setSn(sn);
            ObjectResponse objectResponse = upgrade(deviceUpgradeRequest);
            if (!ObjectResponse.isSuccess(objectResponse)){
                msgList.add(sn + ":" + objectResponse.getMsg());
            }
        });
        if (msgList.size() != 0) {
            String msg = StringUtils.join(msgList.toArray(), ";");
            return ObjectResponse.failed(CodeConstants.ERROR_402, msg);
        }else {
            return ObjectResponse.success();
        }
    }

    @Override
    public ObjectResponse<BatchUpgradeValidateResponse> batchUpgradeValidate(String sns) {
        if (sns == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        BatchUpgradeValidateResponse batchUpgradeValidateResponse = new BatchUpgradeValidateResponse();
        int upgradingCount = 0, upgradableCount = 0;
        for (String sn : sns.split(",")) {
            ObjectResponse<ValidateUpgradeResponse> objectResponse = validateUpgrade(sn);
            if (ObjectResponse.isSuccess(objectResponse)){
                upgradableCount++;
            }else{
                upgradingCount++;
            }
        }
        batchUpgradeValidateResponse.setUpgradableCount(upgradableCount);
        batchUpgradeValidateResponse.setUpgradingCount(upgradingCount);
        return ObjectResponse.success(batchUpgradeValidateResponse);
    }

    @Override
    public ObjectResponse<Page<FirmwareListResponse>> firmwareList(FirmwareListRequest firmwareListRequest) {
        PageHelper.startPage(firmwareListRequest.getPageNo(),firmwareListRequest.getPageSize());
        List<FirmwareListDTO> firmwareListDTOS = firmwareDao.selectListByParam(firmwareListRequest.getProductModelId(), firmwareListRequest.getFirmwareVersion(), firmwareListRequest.getStartTime(), firmwareListRequest.getEndTime());
        if (CollectionUtil.isEmpty(firmwareListDTOS)){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        PageInfo pageInfo = new PageInfo(firmwareListDTOS);
        Page<FirmwareListResponse> page = new Page<>();
        List<FirmwareListResponse> respList = new ArrayList<>();
        firmwareListDTOS.forEach(dto -> {
            FirmwareListResponse firmwareListResponse = new FirmwareListResponse();
            BeanUtil.copyProperties(dto, firmwareListResponse, CopyOptions.create().setIgnoreNullValue(true));
            respList.add(firmwareListResponse);
        });
        page.setRows(respList);
        page.setTotal(pageInfo.getTotal());
        page.setTotalPage(pageInfo.getPages());

        return ObjectResponse.success(page);
    }

    @Override
    public ObjectResponse saveFirmware(SaveFirmwareRequest saveFirmwareRequest) {
        //校验参数
        boolean validator = Validator.validate(saveFirmwareRequest);
        if (!validator){
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        Integer firmwareId = saveFirmwareRequest.getFirmwareId();
        Firmware firmware = new Firmware();
        firmware.setId(saveFirmwareRequest.getFirmwareId());
        firmware.setFirmwareVersion(saveFirmwareRequest.getFirmwareVersion());
        firmware.setProductModelId(saveFirmwareRequest.getProductModelId());
        firmware.setFileId(saveFirmwareRequest.getFileId());
        firmware.setOperAccount(saveFirmwareRequest.getOperAccount());

        if (firmwareId != null){
            firmwareDao.updateById(firmware);
        }else{
            firmwareDao.insert(firmware);
        }
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<List<String>> findByProductModelId(Integer productModelId) {
        List<Firmware> firmwares = firmwareDao.selectListByProductModelId(productModelId);
        if (CollectionUtil.isEmpty(firmwares)){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        List<String> firmwareVersionList = new ArrayList<>();
        firmwares.forEach(firmware -> firmwareVersionList.add(firmware.getFirmwareVersion()));
        return ObjectResponse.success(firmwareVersionList);
    }

    @Override
    public ObjectResponse upgradeProgress(String taskId) {
        UpgradeProgressResponse progress = deviceUpgradeHandle.progress(taskId);
        if (progress == null){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        return ObjectResponse.success(progress);
    }

    @Override
    public ObjectResponse<ValidateVersionResponse> validateVersion(Integer productModelId, String firmwareVersion) {
        List<Firmware> firmwares = firmwareDao.selectListByVersion(productModelId, firmwareVersion);
        ValidateVersionResponse validateVersionResponse = new ValidateVersionResponse();
        if (CollectionUtil.isEmpty(firmwares)){
            validateVersionResponse.setIsExist(0);
        }else{
            validateVersionResponse.setIsExist(1);
        }
        return ObjectResponse.success(validateVersionResponse);
    }

    @Override
    public ObjectResponse<ValidateUpgradeResponse> validateUpgrade(String sn) {
        if (sn == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        ObjectResponse<ParkDevice> deviceObjectResponse = parkDeviceService.getDeviceBySerialNumber(sn);
        ObjectResponse.notError(deviceObjectResponse);

        ValidateUpgradeResponse validateUpgradeResponse = new ValidateUpgradeResponse();
        validateUpgradeResponse.setIsUpgrade(0);

        ParkDevice parkDevice = deviceObjectResponse.getData();
        String protocolVer = parkDevice.getProtocolVer();
        String productModel = parkDevice.getProductModel();
        if (parkDevice.getStatus() != 1 && parkDevice.getStatus() != 3) {
            validateUpgradeResponse.setReason("设备不在线");
            return ObjectResponse.success(validateUpgradeResponse);
        }
        if (protocolVer == null || productModel == null) {
            validateUpgradeResponse.setReason("未知的协议版本号或产品型号");
            return ObjectResponse.success(validateUpgradeResponse);
        }
        if (P2cVersionEnum.getIndex(protocolVer) < P2cVersionEnum.版本7.getIndex()) {
            validateUpgradeResponse.setReason("协议版本号过低");
            return ObjectResponse.success(validateUpgradeResponse);
        }
        ObjectResponse<DictionaryItemDto> objectResponse = dictionaryItemService.findOneItem(8, productModel, null);
        if (!ObjectResponse.isSuccess(objectResponse)) {
            validateUpgradeResponse.setReason("设备型号未定义");
            return ObjectResponse.success(validateUpgradeResponse);
        }
        DictionaryItemDto dictionaryItemDto = objectResponse.getData();
        List<Firmware> firmwares = firmwareDao.selectListByProductModelId(dictionaryItemDto.getValue());
        if (CollectionUtil.isEmpty(firmwares)){
            validateUpgradeResponse.setReason("该设备型号的固件包未添加");
            return ObjectResponse.success(validateUpgradeResponse);
        }
        FirmwareUpgrade runningTask = firmwareUpgradeDao.selectTaskBySnAndStatus(sn, FirmwareUpgrade.StatusEnum.升级中.status);
        //判断是否超时
        if (runningTask != null){
            Date actionTime = runningTask.getActionTime();
            long now = DateTools.unixTimestamp();
            //上一个任务是执行中时，判断是否超时响应，超时时间为10分钟
            if (now - actionTime.getTime() / 1000 < 600) {
                return ObjectResponse.failed(CodeConstants.ERROR_402, "当前设备正在升级，请勿重复操作");
            }
        }
        validateUpgradeResponse.setIsUpgrade(1);
        return ObjectResponse.success(validateUpgradeResponse);
    }

    @Override
    public ObjectResponse<List<SupportUpgradeDeviceResponse>> supportUpgradeDevice(List<DeviceListDTO> deviceListDTO) {
        ObjectResponse<List<DictionaryItemDto>> listByTypeId = dictionaryItemService.findListByTypeId(8);
        if (!ObjectResponse.isSuccess(listByTypeId)) {
            return ObjectResponse.failed(CodeConstants.ERROR_402, "未定义产品型号");
        }
        //所有的设备型号列表
        List<DictionaryItemDto> dictionaryItemDtos = listByTypeId.getData();
        Map<String, Integer> integerMap = dictionaryItemDtos.stream().collect(Collectors.toMap(DictionaryItemDto::getText, DictionaryItemDto::getValue));

        List<Integer> hasFirmwareModels = firmwareDao.selectAllProductModel();

        List<SupportUpgradeDeviceResponse> list = new ArrayList<>();
        deviceListDTO.forEach(deviceDTO ->{
            SupportUpgradeDeviceResponse supportUpgradeDeviceResponse = new SupportUpgradeDeviceResponse();
            supportUpgradeDeviceResponse.setDeviceId(deviceDTO.getDeviceId());
            supportUpgradeDeviceResponse.setIsSupportUpgrade(1);
            list.add(supportUpgradeDeviceResponse);
            String protocolVer = deviceDTO.getProtocolVer();
            String productModel = deviceDTO.getProductModel();
            if (deviceDTO.getDeviceStatus() != 1 && deviceDTO.getDeviceStatus() != 3) {
                supportUpgradeDeviceResponse.setIsSupportUpgrade(0);
                supportUpgradeDeviceResponse.setReason("设备不在线");
                return;
            }
            if (protocolVer == null || productModel == null) {
                supportUpgradeDeviceResponse.setIsSupportUpgrade(0);
                supportUpgradeDeviceResponse.setReason("未知的协议版本号或产品型号");
                return;
            }
            if (P2cVersionEnum.getIndex(protocolVer) < P2cVersionEnum.版本7.getIndex()) {
                supportUpgradeDeviceResponse.setIsSupportUpgrade(0);
                supportUpgradeDeviceResponse.setReason("协议版本号过低");
                return;
            }
            if (CollectionUtil.isEmpty(hasFirmwareModels) || !hasFirmwareModels.contains(integerMap.get(productModel))){
                supportUpgradeDeviceResponse.setIsSupportUpgrade(0);
                supportUpgradeDeviceResponse.setReason("该设备型号的固件包未添加");
                return;
            }
        });
        return ObjectResponse.success(list);
    }

    @Override
    public ObjectResponse reUpgrade(String taskId) {
        FirmwareUpgrade firmwareUpgrade = firmwareUpgradeDao.selectByTaskId(taskId);
        if (firmwareUpgrade == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        UpgradeRequest upgradeRequest = new UpgradeRequest();
        upgradeRequest.setTaskId(taskId);
        upgradeRequest.setNewFirmwareVer(firmwareUpgrade.getTargetVer());
        upgradeRequest.setSendTime(DateTools.unixTimestamp());
        File file = fileDao.selectById(firmwareUpgrade.getFileId());
        upgradeRequest.setFirmwarePath(file.getFilePath());
        upgradeRequest.setFirmwareSize(String.valueOf(file.getFileSize()));
        upgradeRequest.setUpgradeChannel(firmwareUpgrade.getUpgradeChannel());
        try {
            upgradeService.execute(upgradeRequest, firmwareUpgrade.getSn(), firmwareUpgrade.getParkId());
        }catch (ResponseBodyException re){
            return ObjectResponse.failed(re.getErrCode(), re.getMessage());
        }
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Page<UpgradeLogListResponse>> upgradeLogList(UpgradeLogListRequest upgradeLogListRequest) {

        PageHelper.startPage(upgradeLogListRequest.getPageNo(),upgradeLogListRequest.getPageSize());

        List<FirmwareUpgradeListDTO> firmwareUpgradeListDTOS = firmwareUpgradeDao.selectListByParam(upgradeLogListRequest.getParkIdList(), upgradeLogListRequest.getSn(), upgradeLogListRequest.getTaskStatus(), upgradeLogListRequest.getStartActionTime(), upgradeLogListRequest.getEndActionTime());
        if (CollectionUtil.isEmpty(firmwareUpgradeListDTOS)){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        PageInfo pageInfo = new PageInfo(firmwareUpgradeListDTOS);
        List<UpgradeLogListResponse> list = new ArrayList<>();
        firmwareUpgradeListDTOS.forEach(dto ->{
            UpgradeLogListResponse upgradeLogListResponse = new UpgradeLogListResponse();
            BeanUtil.copyProperties(dto, upgradeLogListResponse, CopyOptions.create().setIgnoreNullValue(true));
            list.add(upgradeLogListResponse);
        });
        Page<UpgradeLogListResponse> page = new Page<>();
        page.setTotalPage(pageInfo.getPages());
        page.setTotal(pageInfo.getTotal());
        page.setRows(list);
        return ObjectResponse.success(page);
    }

}
