package com.icetech.park.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.*;
import com.icetech.common.validator.Validator;
import com.icetech.cloudcenter.api.IDataReportService;
import com.icetech.cloudcenter.domain.request.pnc.DataCenterBaseRequest;
import com.icetech.cloudcenter.domain.enumeration.DataCollectionEnum;
import com.icetech.basics.utils.Spring;
import com.icetech.park.service.report.pnc.ReportService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

@Service
@Slf4j
public class DataReportServiceImpl implements IDataReportService {
    @Autowired
    private ParkService parkService;
    private Map<String, Integer> FAIL_CS = new HashMap<>();
    private static int FAIL_LIMIT = 5;
    /**
     * service实现bean名的后缀
     */
    private final String SERVICE_SUFFIX = "ServiceImpl";
    //黑名单车场
    private static String Blacklist_ParkCode = "P12345678";
    //错误车场，防止频繁处理
    private static Map<String, Long> ERROR_PARKS = new HashMap<>();

    @Override
    public String report(String args) {
        DataCenterBaseRequest baseRequest = JsonUtils.parseObject(args, DataCenterBaseRequest.class);
        String parkCode = baseRequest.getParkCode();
        if (Blacklist_ParkCode.contains(parkCode)){
            return ResponseUtils.setResponse(CodeConstants.ERROR_400, CodeConstants.getName(CodeConstants.ERROR_400));
        }
        if (ERROR_PARKS.get(parkCode) != null){
            Long lastTime = ERROR_PARKS.get(parkCode);
            long windowTime = DateTools.unixTimestamp() - lastTime;
            if (windowTime < 600){
                log.info("{}-{}", parkCode, windowTime);
                //return ResponseUtils.setResponse(CodeConstants.ERROR_400, CodeConstants.getName(CodeConstants.ERROR_400));
            }else{
                ERROR_PARKS.remove(parkCode);
            }

        }

        if (StringUtils.isBlank(parkCode)){
            log.info("车场编号为空");
            return ResponseUtils.setResponse(CodeConstants.ERROR_400, CodeConstants.getName(CodeConstants.ERROR_400));
        }

        ObjectResponse<Park> objectResponse = parkService.findByParkCode(baseRequest.getParkCode());
        if (!ObjectResponse.isSuccess(objectResponse)){
            return JsonUtils.toString(dealFailCode(objectResponse, baseRequest));
        }
        Park park = objectResponse.getData();

        ObjectResponse<ParkConfig> parkConfigResponse = parkService.getParkConfig(parkCode);
        ObjectResponse.notError(parkConfigResponse);
        ParkConfig parkConfig = parkConfigResponse.getData();

        if (DataCollectionEnum.端网云.getType().equals(parkConfig.getDataCollection())){
            try{
                //签名校验，测试车场跳过签名
                if (StringUtils.isNotBlank(baseRequest.getVersionNum())){
                    verify(baseRequest, park.getKey());
                }
                String prefix = baseRequest.getServiceName();
                ReportService reportService = getServiceBean(prefix);
                ObjectResponse response = reportService.report(baseRequest, park.getId());
                if (ObjectResponse.isSuccess(response)){
                    return JsonUtils.toString(response);
                }else{
                    return JsonUtils.toString(dealFailCode(response, baseRequest));
                }
            }catch (ResponseBodyException e){
                ObjectResponse objectResponse1 = new ObjectResponse(e.getErrCode(), CodeConstants.getName(e.getErrCode()) + "," + e.getMessage());
                return JsonUtils.toString(dealFailCode(objectResponse1, baseRequest));
            }catch (Exception e){
                log.error("端网云上报处理异常", e);
                return JsonUtils.toString(dealFailCode(ObjectResponse.failed(), baseRequest));
            }
        }else{
            ERROR_PARKS.put(parkCode, DateTools.unixTimestamp());
            return ResponseUtils.setResponse(CodeConstants.ERROR_401, CodeConstants.getName(CodeConstants.ERROR_401) + "，连接模式不正确");
        }

    }

    /**
     * 获取service实现类
     * @param prefix
     * @return
     */
    private ReportService getServiceBean(String prefix){
        String serviceName = prefix + SERVICE_SUFFIX;
        try {
            return Spring.getBean(serviceName);
        }catch(RuntimeException e){
            throw new ResponseBodyException(CodeConstants.ERROR_404, CodeConstants.getName(CodeConstants.ERROR_404));
        }
    }

    /**
     * 验证参数和签名
     * @param baseRequest
     * @param secretKey
     */
    private void verify(DataCenterBaseRequest baseRequest, String secretKey) {
        if (Validator.validate(baseRequest)) {

            boolean verifyResult = SignTools.verifyMD5Sign(baseRequest, secretKey);
            if (!verifyResult) {
                throw new ResponseBodyException(CodeConstants.ERROR_401, "验签未通过");
            }
        } else {
            throw new ResponseBodyException(CodeConstants.ERROR_400, "必填参数为空");
        }
    }

    private ObjectResponse dealFailCode(ObjectResponse objectResponse, DataCenterBaseRequest baseRequest){
        Object bizContent = baseRequest.getBizContent();
        try {
            if (bizContent == null) {
                return ObjectResponse.failed(CodeConstants.ERROR_402);
            }
            String md5 = DigestUtils.md5Hex(JsonUtils.toString(bizContent)).toUpperCase();
            Integer cs = FAIL_CS.get(md5);
            if (cs == null){
                FAIL_CS.put(md5, 1);
            }else if (cs < FAIL_LIMIT){
                FAIL_CS.put(md5, cs + 1);
            }else{
                FAIL_CS.remove(md5);
                log.info("<通用数据接口> 上传失败次数已超过[{}]次，返回成功，参数：{}，应返回：{}", FAIL_LIMIT, JsonUtils.replaceJsonContent(JsonUtils.toString(baseRequest), "base64Str"), objectResponse);

                if (baseRequest.getServiceName().equals("enter") || baseRequest.getServiceName().equals("complateOrder")) {
                    Map<String, String> data = new HashMap<String, String>();
                    data.put("orderId", (String) ((JSONObject)bizContent).get("orderId"));
                    return ObjectResponse.success(data);
                }else{
                    return new ObjectResponse(CodeConstants.SUCCESS, "限制访问");
                }
            }
        } catch (Exception e) {
            log.error(String.valueOf(e.getMessage()), e);
        }
        return objectResponse;
    }
}
