package com.icetech.park.service.handle.showsay;

import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.api.lcd.LcdService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.store.StoreCardService;
import com.icetech.cloudcenter.domain.response.LcdDto;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.domain.entity.lcd.LcdConfig;
import com.icetech.park.domain.entity.lcd.LcdShow;
import com.icetech.park.domain.entity.lcd.LcdTips;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.IShowVoiceHandle;
import com.icetech.park.service.flow.p2c.FlowCondition;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * LCD显示屏处理
 *
 * @author fangct
 */
@Component
@Slf4j
public class LcdShowHandle implements IShowVoiceHandle {

    @Autowired
    private LcdService lcdService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private StoreCardService storeCardService;
    @Autowired
    private ShowSayBaseHandle showSayBaseHandle;
    @Autowired
    private ParkRegionDao parkRegionDao;

    @Override
    public String enterHandle(Long parkId, Long channelId, String plateNum, Integer type, FlowCondition.ResultCode resultCode, Map<String, Object> para) {
        return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdShow.DisplayTypeEnum.入场显示.type);
    }

    @Override
    public String exitHandle(Long parkId, Long channelId, String plateNum, Integer type, FlowCondition.ResultCode resultCode, Map<String, Object> para) {
        return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdShow.DisplayTypeEnum.出场显示.type);
    }

    /**
     * 生成最终要显示的内容
     *
     * @param parkId 车场ID
     * @param plateNum 车牌号
     * @param type 车辆类型
     * @param resultCode 处理结果
     * @param para 参数MAP
     * @param displayType 显示类型
     * @return 显示内容
     */
    private String generateContent(Long parkId, Long channelId, String plateNum, Integer type, FlowCondition.ResultCode resultCode, Map<String, Object> para, int displayType) {
        ObjectResponse<LcdDto> objectResponse = lcdService.getLcdShowByType(channelId, displayType);
        String content = objectResponse.getData().getContent();
        String[] dynamicValues = StringUtils.findBraceValue(content);
        if (para == null) {
            para = new HashMap<>();
        }
        for (String value : dynamicValues) {
            int dynamicValue = Integer.parseInt(value);
            if (dynamicValue == LcdShow.VariateEnum.停车场名称.type) {
                Object parkName = para.get("parkName");
                if (parkName == null) {
                    ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(parkId);
                    para.put("parkName", parkObjectResponse.getData().getParkName());
                }
            } else if (dynamicValue == LcdShow.VariateEnum.月卡剩余天数.type && PlateTypeEnum.月卡车.getType().equals(type)) {
                Integer days = showSayBaseHandle.getMonthCarRemainDays(parkId, channelId, plateNum, para);
                if (days == null) {
                    content = content.replace("{" + dynamicValue + "}", "");
                    if (NumberUtils.toPrimitive((Integer) para.get("days")) == -2) {
                        type = PlateTypeEnum.临时车.getType();
                    }
                    continue;
                }
            }
            content = content.replace("{" + dynamicValue + "}", getReplaceContent(dynamicValue, parkId, channelId, plateNum, type, resultCode, para));
        }
        return content.trim();
    }

    private String getReplaceContent(int dynamicValue, Long parkId, Long channelId, String plateNum, Integer type,
                                     FlowCondition.ResultCode resultCode, Map<String, Object> para) {
        String replaceContent = "";
        switch (dynamicValue) {
            case 1://当前时间
                Date date = new Date();
                replaceContent = DateTools.getFormat(DateTools.DF_CN, date);
                break;
            case 2://今日限号
                ObjectResponse<LcdConfig> lcdConfigObjectResponse = lcdService.getLcdConfigByChannel(channelId);
                LcdConfig lcdConfig = lcdConfigObjectResponse.getData();
                Integer limitType = lcdConfig.getLimitType();
                int week = Integer.parseInt(DateTools.getWeek());
                if (limitType == 0) {//尾号限行
                    if (week == 1) {
                        replaceContent = "今日限号" + lcdConfig.getLimitDriveNum1();
                    } else if (week == 2) {
                        replaceContent = "今日限号" + lcdConfig.getLimitDriveNum2();
                    } else if (week == 3) {
                        replaceContent = "今日限号" + lcdConfig.getLimitDriveNum3();
                    } else if (week == 4) {
                        replaceContent = "今日限号" + lcdConfig.getLimitDriveNum4();
                    } else if (week == 5) {
                        replaceContent = "今日限号" + lcdConfig.getLimitDriveNum5();
                    } else {//周六、周日
                        replaceContent = "不限行";
                    }
                } else {//单双号限行
                    if (week == 0 || week == 6) {
                        replaceContent = "不限行";
                    } else {
                        int day = Integer.parseInt(DateTools.getDay());
                        if (day % 2 == 0) {
                            replaceContent = "单号限行";
                        } else {
                            replaceContent = "双号限行";
                        }
                    }
                }
                break;
            case 3:
                int freeSpace = showSayBaseHandle.getFreeSpace(parkId, channelId, para);
                replaceContent = showSayBaseHandle.getFreeSpaceName(parkId, channelId, para) + "剩余车位" + freeSpace;
                break;
            case 4:
                replaceContent = plateNum;
                break;
            case 5:
                showSayBaseHandle.setAndGetPlateType(resultCode, parkId, channelId, plateNum, type, para);
                replaceContent = (String) para.get("showTypeStr");
                break;
            case 6:
                if (type.equals(PlateTypeEnum.月卡车.getType()) || resultCode.equals(FlowCondition.ResultCode.过期卡)) {
                    Integer days = showSayBaseHandle.getMonthCarRemainDays(parkId, channelId, plateNum, para);;
                    if (days != null) {
                        replaceContent = "剩余" + days + "天";
                    }
                }
                break;
            case 7:
                Long parkTime = (Long) para.get("parkTime");
                if (parkTime != null && parkTime != 0) {
                    replaceContent = "停车" + DateTools.secondToSecondsTime(parkTime.intValue());
                }
                break;
            case 8:
                if (resultCode.equals(FlowCondition.ResultCode.过期卡)) {
                    Boolean isMonth = (Boolean) para.get("isMonth");
                    replaceContent = isMonth != null && isMonth ? enterTips(parkId, channelId, FlowCondition.ResultCode.月卡车) : enterTips(parkId, channelId, FlowCondition.ResultCode.有牌车允许临时车进入);
                } else {
                    replaceContent = enterTips(parkId, channelId, resultCode);
                }
                break;
            case 9:
                String prefix = "";
                if (Boolean.TRUE.equals(para.get("hasInnerAreaFee"))
                        && (FlowCondition.ResultCode.需缴费.equals(resultCode)
                        || FlowCondition.ResultCode.固定费用.equals(resultCode)
                        || FlowCondition.ResultCode.欠费补缴.equals(resultCode))) {
                    showSayBaseHandle.setAndGetPlateType(resultCode, parkId, channelId, plateNum, type, para);
                    if (para.get("regionId") != null && (PlateTypeEnum.月卡车.getType().equals(para.get("type"))
                            || PlateTypeEnum.VIP车辆.getType().equals(para.get("type")))) {
                        Long regionId = (Long) para.get("regionId");
                        ParkRegion parkRegion = parkRegionDao.selectById(regionId);
                        prefix = parkRegion != null ? parkRegion.getRegionName() : prefix;
                    }
                }
                replaceContent = prefix + exitTips(parkId, channelId, resultCode, para);
                break;
            case 10:
                Object fee = para.get("fee");
                if (fee == null || Float.parseFloat((String) fee) <= 0) {
                    if (ShowSayConstants.NO_FEE_RESULT_LIST.contains(resultCode)) {
                        replaceContent = "无需缴费";
                    }
                    break;
                }
                replaceContent = fee + "元";
                break;
            case 11:
                replaceContent = (String) para.get("parkName");
                break;
            case 12:
                if (PlateTypeEnum.储值卡车.getType().equals(type)) {
                    ObjectResponse<BigDecimal> storeCardBalance = storeCardService.getStoreCardBalance(parkId, plateNum);
                    if (ObjectResponse.isSuccess(storeCardBalance)) {
                        BigDecimal balance = storeCardBalance.getData();
                        replaceContent = "储值卡余额" + balance + "元";
                    }
                }
                break;
            default:
                break;
        }
        return replaceContent;
    }

    /**
     * 入场提示
     *
     * @param parkId 车场ID
     * @param resultCode 处理结果
     * @return 提示内容
     */
    public String enterTips(Long parkId, Long channelId, FlowCondition.ResultCode resultCode) {
        ObjectResponse<LcdTips> objectResponse = lcdService.getLcdTipsByChannel(channelId);
        LcdTips lcdTips = objectResponse.getData();
        switch (resultCode) {
            case 月卡车:
            case 内部车辆:
            case VIP车辆:
                return lcdTips.getEnterMonthVip();
            case 有牌车允许临时车进入:
            case 储值卡车允许进入:
                return lcdTips.getEnterTmpSpecial();
            case 有牌车不允许临时车进入:
            case 储值卡车禁止进入:
                return lcdTips.getEnterNotAllowTmpRun();
            case 黑名单车:
                return lcdTips.getEnterBlackCar();
            case 无牌车允许进出:
                return lcdTips.getEnterNotAllowNocarRun();
            case 无牌车不允许进入:
                return lcdTips.getEnterNotAllowNocarRun();
            case 车位已满禁止入场:
                return lcdTips.getEnterNoParkingSpace();
            case 黄牌车禁止入场:
                return lcdTips.getEnterNotAllowTmpRun();
            case 过期卡:
                return lcdTips.getEnterMonthExpire();
            case 车位已占:
                return lcdTips.getEnterMultiParkCar();
            case 欠费补缴:
                return lcdTips.getEnterUnpaid();
            case 虚假车牌:
                return lcdTips.getEnterShamPlate() == null ? "" : lcdTips.getEnterShamPlate();
            default:
                break;
        }
        return "";
    }

    /**
     * 出场提示
     *
     * @param parkId 车场ID
     * @param resultCode 处理结果
     * @return 响应内容
     */
    public String exitTips(Long parkId, Long channelId, FlowCondition.ResultCode resultCode, Map<String, Object> para) {
        ObjectResponse<LcdTips> objectResponse = lcdService.getLcdTipsByChannel(channelId);
        LcdTips lcdTips = objectResponse.getData();
        switch (resultCode) {
            case 月卡车:
            case 内部车辆:
            case 无入场有全免优惠:
            case 免费时间内:
            case 无需缴费:
            case 特殊车辆免费:
            case 预缴费未超时:
            case 无入场VIP全免车:
                String prefix = "";
                String payWay = showSayBaseHandle.getPayWay(para);
                if (payWay != null) {
                    prefix = payWay + "支付成功 ";
                }
                return prefix + lcdTips.getExitOpenGate();
            case 黑名单车:
            case 不允许临时车出:
                return lcdTips.getExitNotAllowRun();
            case 需缴费:
            case 固定费用:
                return lcdTips.getExitPay();
            case 无入场记录:
                return lcdTips.getExitNoEnterinfo();
            case 无牌车允许进出:
                return lcdTips.getExitNoCarplate();
            case 缴费后屏显播报:
                String prefix2 = "";
                String payWay2 = showSayBaseHandle.getPayWay(para);
                if (payWay2 != null) {
                    prefix2 = payWay2 + "支付成功 ";
                }
                return prefix2 + lcdTips.getExitOpenGate();
            case 欠费补缴:
                return lcdTips.getExitUnpaid();
            case 虚假车牌:
                return lcdTips.getExitShamPlate() == null ? "" : lcdTips.getExitShamPlate();
            default:
                break;
        }
        return "";
    }
}
