package com.icetech.park.service.handle.showsay;

import com.icetech.cloudcenter.api.lcd.LcdService;
import com.icetech.cloudcenter.domain.response.LedSoundDto;
import com.icetech.park.domain.entity.lcd.LcdSound;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.park.service.IShowVoiceHandle;
import com.icetech.park.service.flow.p2c.FlowCondition;
import com.icetech.common.utils.NumberUtils;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * 语音播报处理
 * @author fangct
 */
@Component
@Slf4j
public class LcdSayHandle implements IShowVoiceHandle {

    @Autowired
    private LcdService lcdService;
    @Autowired
    private ShowSayBaseHandle showSayBaseHandle;

    @Override
    public String enterHandle(Long parkId, Long channelId, String plateNum, Integer type, FlowCondition.ResultCode resultCode, Map<String, Object> para) {
        switch (resultCode) {
            case 月卡车:
            case VIP车辆:
            case 访客车:
            case 内部车辆:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.入场月卡车_内部车辆.type);
            case 过期卡:
                if (type.equals(PlateTypeEnum.月卡车.getType())){
                    return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.入场月卡车_内部车辆.type);
                }else{
                    return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.入场临时车.type);
                }
            case 有牌车允许临时车进入:
            case 储值卡车允许进入:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.入场临时车.type);
            case 有牌车不允许临时车进入:
            case 储值卡车禁止进入:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.临时车禁止通行.type);
            case 黑名单车:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.黑名单车辆进出场.type);
            case 无牌车允许进出:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.入场无牌车.type);
            case 无牌车不允许进入:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.临时车禁止通行.type);
            case 车位已满禁止入场:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.车位已满.type);
            case 黄牌车禁止入场:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.临时车禁止通行.type);
            case 车位已占:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.多位多车车位占用.type);
            case 欠费补缴:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出场收费车辆.type);
            case 虚假车牌:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.入口虚假车牌.type);
            default:
                break;
        }
        return "";
    }

    @Override
    public String exitHandle(Long parkId, Long channelId, String plateNum, Integer type, FlowCondition.ResultCode resultCode, Map<String, Object> para) {

        switch (resultCode) {
            case 月卡车:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出场月卡车.type);
            case 无入场有全免优惠:
            case 免费时间内:
            case 无需缴费:
            case 特殊车辆免费:
            case 预缴费未超时:
            case 内部车辆:
            case VIP车辆:
            case 无入场VIP全免车:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出场免费车辆_临时特殊.type);
            case 黑名单车:
            case 不允许临时车出:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.黑名单车辆进出场.type);
            case 需缴费:
            case 固定费用:
            case 欠费补缴:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出场收费车辆.type);
            case 无入场记录:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出场无入场记录.type);
            case 无牌车允许进出:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出场无牌车.type);
            case 缴费后屏显播报:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出场支付成功.type);
            case 虚假车牌:
                return generateContent(parkId, channelId, plateNum, type, resultCode, para, LcdSound.SoundTypeEnum.出口虚假车牌.type);
            default:
                break;
        }
        return "";
    }

    /**
     * 生成最终要显示的内容
     * @param parkId 车场ID
     * @param plateNum 车牌号
     * @param type 车辆类型
     * @param resultCode 处理结果
     * @param para 参数MAP
     * @param soundType 语音类型
     * @return 要播出的语音内容
     */
    private String generateContent(Long parkId, Long channelId, String plateNum, Integer type, FlowCondition.ResultCode resultCode, Map<String, Object> para, int soundType) {
        if (para == null){
            para = new HashMap<>();
        }
        ObjectResponse<LedSoundDto> soundDtoObjectResponse = lcdService.getSoundConfigByType(channelId, soundType);
        LedSoundDto soundVo = soundDtoObjectResponse.getData();
        String content = soundVo.getContent();
        String[] dynamicValues = content.split(" ");

        String mcDays = "D5";
        for (String dynamicValue : dynamicValues) {
            if (dynamicValue.equals(mcDays) && PlateTypeEnum.月卡车.getType().equals(type)) {
                Integer days = showSayBaseHandle.getMonthCarRemainDays(parkId, channelId, plateNum, para);
                if (days == null) {
                    content = content.replace(dynamicValue, "");
                    if (NumberUtils.toPrimitive((Integer) para.get("days")) == -2) {
                        type = PlateTypeEnum.临时车.getType();
                    }
                    continue;
                }
            }
            content = content.replace(dynamicValue, getNoTtsReplaceContent(dynamicValue, parkId, channelId, plateNum, type, resultCode, para));
        }
        return content;
    }

    private String getNoTtsReplaceContent(String dynamicValue, Long parkId, Long channelId, String plateNum, Integer type,
                                          FlowCondition.ResultCode resultCode, Map<String, Object> para) {
        String replaceContent = "";
        switch (dynamicValue) {
            case "D2":
                int freeSpace = showSayBaseHandle.getFreeSpace(parkId, channelId, para);
                replaceContent = "<D2>" + freeSpace + "</D2>";
                break;
            case "D3":
                replaceContent = "<D3>" + plateNum + "</D3>";
                break;
            case "D4":
                showSayBaseHandle.setAndGetPlateType(resultCode, parkId, channelId, plateNum, type, para);
                Integer realType = (Integer) para.getOrDefault("type", 1);
                String newType;
                if (realType == 2){
                    newType = PlateTypeEnum.月卡车.name();
                } else {
                    newType = PlateTypeEnum.临时车.name();
                }
                replaceContent = "<D4>" + newType +"</D4>";
                break;
            case "D5":
                if (PlateTypeEnum.月卡车.getType().equals(type) || resultCode.equals(FlowCondition.ResultCode.月卡车) || resultCode.equals(FlowCondition.ResultCode.过期卡)){
                    Integer days = showSayBaseHandle.getMonthCarRemainDays(parkId, channelId, plateNum, para);;
                    if (days != null){
                        replaceContent = "<D5>" + days + "</D5>";
                    }
                }
                break;
            case "D6":
                Long parkTime = (Long) para.get("parkTime");
                if(parkTime != null && parkTime != 0){
                    replaceContent = "<D6>" + parkTime + "</D6>";
                }
                break;
            case "D7":
                String feeStr = (String) para.get("fee");
                if (feeStr != null){
                    int fee = (int) (Float.parseFloat(feeStr) * 10);
                    replaceContent = "<D7>" + fee + "</D7>";
                }
                break;
            case "D8":
                replaceContent = "<D8>"+ "小型车" + "</D8>";
                break;
            default:
                replaceContent = "<"+dynamicValue+"/>";
                break;
        }
        return replaceContent;
    }
}
