package com.icetech.park.service.handle;

import com.alibaba.fastjson.JSON;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.third.domain.entity.third.SendInfoRecord;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.enumeration.SendOperTypeEnum;
import com.icetech.cloudcenter.domain.request.pnc.DownBaseRequest;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.constants.TimeOutConstants;
import com.icetech.common.domain.WebSocketMessage;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.SignTools;
import com.icetech.common.utils.Slf4jUtils;
import com.icetech.common.utils.SpringUtils;
import com.icetech.common.utils.UUIDTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.icetech.cloudcenter.domain.constants.RedisConstants.SEND_MSGID_PROFILE;

@Slf4j
@Component
public class PncDownHandle {
    @Autowired
    private ParkService parkService;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private StringRedisTemplate redisTemplate;

    /**
     * 同步返回的服务
     */
    private static final List<String> SYNC_SERVICE_NAME_LIST = new ArrayList<>();
    static {
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.缴费查询.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.实时通道数据.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.无牌车入场.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.无牌车离场.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.语音显示屏内容.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.请求入场.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.修改停车费用.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.远程开关闸.getServiceName());
        SYNC_SERVICE_NAME_LIST.add(DownServiceEnum.预缴费.getServiceName());
    }

    /**
     * 签名并下发消息
     * @param parkCode
     * @param serviceName
     * @param t
     * @return
     */
    public <T> String signAndSend(String parkCode, String serviceName, T t){
        return signAndSend(parkCode, serviceName, t, null);
    }

    public <T> String signAndSend(
            String parkCode, String serviceName, T t, Long serviceId){
        ObjectResponse<Park> objectResponse = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(objectResponse);
        Park park = objectResponse.getData();
        Long parkId = park.getId();
        String key = park.getKey();
        return signAndSend(parkId, parkCode, key, serviceName, t, serviceId);
    }
    public <T> String signAndSend(
            Long parkId, String serviceName, T t){
        return signAndSend(parkId, serviceName, t, null);
    }
    public <T> String signAndSend(
            Long parkId, String serviceName, T t, Long serviceId){
        return signAndSend(parkId, serviceName, t, serviceId, null);
    }
    public <T> String signAndSend(Long parkId, String serviceName, T t, Long serviceId, String channelId){
        return signAndSend(parkId, serviceName, t, serviceId, channelId, null);
    }

    public <T> String signAndSend(Long parkId, String serviceName, T t, Long serviceId, String channelId, Long recordId) {
        ObjectResponse<Park> objectResponse = parkService.findByParkId(parkId);
        ObjectResponse.notError(objectResponse);
        Park park = objectResponse.getData();
        String parkCode = park.getParkCode();
        String key = park.getKey();
        return signAndSend(parkId, parkCode, key, serviceName, t, serviceId, channelId, recordId, null, null);
    }

    /**签名并下发消息
     * @param parkId
     * @param parkCode
     * @param key
     * @param serviceName
     * @param t
     * @param serviceId
     * @param <T>
     * @return
     */
    public <T> String signAndSend(Long parkId, String parkCode, String key, String serviceName, T t, Long serviceId) {
        return signAndSend(parkId, parkCode, key, serviceName, t, serviceId, null);
    }

    public <T> String signAndSend(Long parkId, String parkCode, String key, String serviceName, T t, Long serviceId, String channelId) {
        return signAndSend(parkId, parkCode, key, serviceName, t, serviceId, channelId, null, null, null);
    }

    /**
     * 异步结果
     * @param parkId 车场id
     * @param parkCode 车场编号
     * @param key 车场key
     * @param serviceName 业务名
     * @param channelId 通道编号
     * @param topic 响应的topic
     * @param <T> 参数
     * @return 结果
     */
    public <T> String signAndSendAsyncResult(Long parkId, String parkCode, String key, String serviceName, T t,
                                             String channelId, String topic, String extraInfo) {
        return signAndSend(parkId, parkCode, key, serviceName, t, null, channelId, null, topic, extraInfo);
    }
    public <T> String signAndSendAsyncResult(Long parkId, String parkCode, String key, String serviceName, T t, String topic, String extraInfo) {
        return signAndSend(parkId, parkCode, key, serviceName, t, null, null, null, topic, extraInfo);
    }
    public <T> String signAndSendAsyncResult(Long parkId, String serviceName, T t,
                                             Long serviceId, String channelId, Long recordId, String topic, String extraInfo) {
        ObjectResponse<Park> objectResponse = parkService.findByParkId(parkId);
        ObjectResponse.notError(objectResponse);
        Park park = objectResponse.getData();
        String parkCode = park.getParkCode();
        String key = park.getKey();
        return signAndSend(parkId, parkCode, key, serviceName, t, serviceId, channelId, recordId, topic, extraInfo);
    }
    public <T> String signAndSendAsyncResult(String parkCode, String serviceName, T t, String topic) {
        ObjectResponse<Park> objectResponse = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(objectResponse);
        Park park = objectResponse.getData();
        Long parkId = park.getId();
        String key = park.getKey();
        return signAndSend(parkId, parkCode, key, serviceName, t, null, null, null, topic, null);
    }
    public <T> String signAndSendAsyncResult(Long parkId, String serviceName, T t, String topic) {
        ObjectResponse<Park> objectResponse = parkService.findByParkId(parkId);
        ObjectResponse.notError(objectResponse);
        Park park = objectResponse.getData();
        String parkCode = park.getParkCode();
        String key = park.getKey();
        return signAndSend(parkId, parkCode, key, serviceName, t, null, null, null, topic, null);
    }
    public <T> String signAndSendAsyncResult(String parkCode, String serviceName, T t, String topic, String extraInfo) {
        ObjectResponse<Park> objectResponse = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(objectResponse);
        Park park = objectResponse.getData();
        Long parkId = park.getId();
        String key = park.getKey();
        return signAndSend(parkId, parkCode, key, serviceName, t, null, null, null, topic, extraInfo);
    }
    public <T> String signAndSend(Long parkId, String parkCode, String key, String serviceName, T t,
                                  Long serviceId, String channelId, Long recordId, String topic, String extraInfo) {
        DownBaseRequest<T> baseRequest = new DownBaseRequest<>();
        baseRequest.setParkCode(parkCode);
        baseRequest.setTimestamp(DateTools.unixTimestamp());
        baseRequest.setServiceName(serviceName);
        boolean isSdk = redisUtils.exists(RedisConstants.PNC_ONLINE_CHANNEL_PROFILE + parkCode);
        if (channelId == null && isSdk){
            channelId = redisUtils.get(RedisConstants.PNC_ONLINE_CHANNEL_PROFILE + parkCode, String.class);
        }
        String messageId = UUIDTools.getUuid();
        if(isSdk){
            String messageIdPrefix = channelId == null ? "" : channelId + "_";
            messageId = messageIdPrefix + messageId;
        }
        baseRequest.setMessageId(messageId);
        baseRequest.setBizContent(t);
        boolean success = false;
        try {
            String protocolValue = redisUtils.hGet(RedisConstants.PNC_PROTOCOL, parkId.toString(), String.class);
            if (protocolValue == null) {
                log.warn("消息无法下发,无法确认端网云连接类型|{}|{}|{}|{}", parkId, parkCode, serviceName, serviceId);
                return null;
            }
            int protocolType = Integer.parseInt(protocolValue);
            Map<String, Object> map = SignTools.convertMap(baseRequest);
            String sign = SignTools.getMySign(map, key);
            baseRequest.setSign(sign);
            String json = JsonUtils.toString(baseRequest);
            if (protocolType == 1) {
                WebSocketMessage message = WebSocketMessage.buildMessage("pnc", parkCode, json)
                        .setProtocolType(1).setTrace(Slf4jUtils.getTraceId());
                redisTemplate.convertAndSend(RedisKeyConstants.SOCKET_MSG_TOPIC, JSON.toJSONString(message));
                log.info("[端网云ws下发] 参数[{}]", json);
            } else {
                log.warn("消息无法下发,无法确认端网云连接类型|{}|{}|{}|{}", parkId, parkCode, serviceName, serviceId);
                return null;
            }
            //记录
            addRecord(parkId, parkCode, serviceName, serviceId, messageId, t, recordId, topic, extraInfo);
            //redis缓存，为超时重发提供支持
            redisUtils.set(SEND_MSGID_PROFILE + messageId, json, 8L);
            success = true;
            return messageId;
        } catch (Exception e) {
            log.warn(String.valueOf(e.getMessage()), e);
        }
        return null;
    }

    /**
     * 新增下发记录
     * @param parkId
     * @param serviceName
     * @param serviceId
     * @param messageId
     * @param t
     * @param topic
     * @param extraInfo
     */
    private <T> void addRecord(Long parkId, String parkCode, String serviceName, Long serviceId, String messageId,
                           T t, Long recordId, String topic, String extraInfo) {
        SendInfoRecord<T> sendinfoRecord = new SendInfoRecord<>();
        sendinfoRecord.setMessageId(messageId);
        sendinfoRecord.setParkId(parkId);
        sendinfoRecord.setParkCode(parkCode);
        sendinfoRecord.setParams(t);
        sendinfoRecord.setServiceId(serviceId);
        sendinfoRecord.setServiceType(DownServiceEnum.getServiceType(serviceName));
        sendinfoRecord.setOperType(SendOperTypeEnum.请求.getOperType());
        sendinfoRecord.setRecordId(recordId);
        sendinfoRecord.setTopic(topic);
        sendinfoRecord.setEnv(SpringUtils.getActiveProfile());
        sendinfoRecord.setExtraInfo(extraInfo);
        redisUtils.set(RedisKeyConstants.MQ_RECORD_PREFIX + messageId, sendinfoRecord, TimeOutConstants.REDIS_TIMEOUT);
    }
}
