package com.icetech.park.service.handle;

import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.impl.TaskCenterServiceImpl;
import com.icetech.third.domain.entity.third.SendInfoRecord;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2rCmdEnum;
import com.icetech.cloudcenter.domain.enumeration.SendOperTypeEnum;
import com.icetech.cloudcenter.domain.request.p2c.P2rBaseRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2rBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.service.AbstractService;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.constants.TimeOutConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.UUIDTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 端云机器人架构的发送消息
 */
@Component
@Slf4j
public class P2rMsgHandle {

    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private TaskCenterServiceImpl taskCenterService;
    @Autowired
    private CacheHandle cacheHandle;

    public <T> String send(Long parkId, String serialNumber, Integer cmdType, T t, Long serviceId) {
        return send(parkId, serialNumber, cmdType, cmdType,t, serviceId);
    }

    /**
     * 给某一个设备发送消息
     *
     * @param reqCmdType
     * @param reqCmdType
     * @param t
     * @param <T>
     * @return
     */
    public <T> String send(Long parkId, String serialNumber, Integer reqCmdType,
                           Integer retCmdType, T t, Long serviceId) {
        P2rBaseRequest p2rBaseRequest = new P2rBaseRequest();
        String messageId = UUIDTools.getUuid();
        p2rBaseRequest.setMessageId(messageId);
        String key = RedisConstants.P2R_SN_PROFILE + serialNumber;
        TokenDeviceVo tokenDeviceVo = redisUtils.get(key, TokenDeviceVo.class);
        if (tokenDeviceVo == null) {
            log.info("[机器人消息下发] 设备不在线, serialNumber[{}]", serialNumber);
            return null;
        }
        p2rBaseRequest.setToken(tokenDeviceVo.getToken());
        p2rBaseRequest.setBizContent(t);

        String paraJson = JsonUtils.toString(p2rBaseRequest);
        ObjectResponse objectResponse = ObjectResponse.success();
        boolean success = ObjectResponse.isSuccess(objectResponse);
        if (success) {
            addRequestRecord(parkId, retCmdType, serviceId, messageId, t);
        }
        return success ? messageId : null;
    }
    /**
     * 给某一个设备发送消息
     * @param cmdType
     * @param t
     * @param <T>
     * @return
     */
    public <T> String send(Long parkId, String serialNumber, Integer cmdType, T t){
        return send(parkId, serialNumber, cmdType, cmdType, t, null);
    }
    /**
     * 响应处理
     *
     * @param p2rBaseResponse
     * @param cmd
     * @param serialNumber
     * @return
     */
    public void dealResponse(P2rBaseResponse p2rBaseResponse, String cmd, String serialNumber) {
        TokenDeviceVo tokenInfo = cacheHandle.getRobotDeviceInfo(serialNumber);
        cmd = cmd.substring(0, cmd.indexOf(AbstractService.CMD_SUFFIX));
        Integer cmdType = P2rCmdEnum.getCmdType(cmd);
        String messageId = p2rBaseResponse.getMessageId();
        SendInfoRecord sendinfoRecord = redisUtils.get(RedisKeyConstants.MQ_RECORD_PREFIX + messageId,
                SendInfoRecord.class);
        if (sendinfoRecord == null) {
            sendinfoRecord = new SendInfoRecord();
            sendinfoRecord.setServiceType(cmdType);
            sendinfoRecord.setMessageId(messageId);
            sendinfoRecord.setParkId(tokenInfo.getParkId());
        }
        //serviceType以下发时的为准，主要是解决通道权限下发和车场权限下发的cmd一样的问题
        cmdType = sendinfoRecord.getServiceType();

        Integer code = p2rBaseResponse.getCode();
        String msg = p2rBaseResponse.getMsg();
        String data = JsonUtils.toString(p2rBaseResponse.getData());
        String key = RedisConstants.RESP_MSG_PROFILE + messageId;

        ObjectResponse retResponse = new ObjectResponse();
        retResponse.setCode(String.valueOf(code));
        retResponse.setMsg(msg);
        retResponse.setData(data);
        //同步返回条件：不包含数据下发业务，且业务ID为空
        if (sendinfoRecord.getServiceId() == null) {
            redisUtils.set(key, retResponse, TimeOutConstants.REDIS_TIMEOUT);
            log.info("<相机响应处理> 写入redis成功，key：{}，value：{}", key, retResponse);
        }else{
            /**
             * 回调返回结果，成功时通知TaskCenter
             */
            Integer serviceId = sendinfoRecord.getServiceId().intValue();
            if (code.equals(CodeEnum.成功.getCode())) {
                taskCenterService.notify(cmdType, serviceId);
            } else {
                if (msg == null) {
                    msg = "未知错误";
                }
                taskCenterService.notify(cmdType, serviceId, msg);
            }
        }
    }
    /**
     * 添加请求记录
     * @param parkId
     * @param cmdType
     * @param serviceId
     * @param messageId
     * @param t
     */
    private <T> void addRequestRecord(Long parkId, Integer cmdType, Long serviceId, String messageId, T t) {
        SendInfoRecord<T> sendinfoRecord = new SendInfoRecord<>();
        sendinfoRecord.setMessageId(messageId);
        sendinfoRecord.setParkId(parkId);
        sendinfoRecord.setParams(t);
        sendinfoRecord.setServiceId(serviceId);
        sendinfoRecord.setServiceType(cmdType);
        sendinfoRecord.setOperType(SendOperTypeEnum.请求.getOperType());
        redisUtils.set(RedisKeyConstants.MQ_RECORD_PREFIX + messageId, sendinfoRecord, TimeOutConstants.REDIS_TIMEOUT);
    }

}
