package com.icetech.park.service.handle;

import cn.hutool.core.io.FileUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.icetech.cloudcenter.api.NotifyService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.DingZhiFuncConstants;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.cloudcenter.domain.request.p2c.LcdHintRequest;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.impl.TaskCenterServiceImpl;
import com.icetech.third.domain.entity.third.SendInfoRecord;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.enumeration.SendOperTypeEnum;
import com.icetech.cloudcenter.domain.request.NotifyRequest;
import com.icetech.cloudcenter.domain.request.p2c.LedsoundConfigRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.ParkConnectedDeviceVo;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.factory.SendServiceFactory;
import com.icetech.third.service.third.MqPushService;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.constants.RedisMsg;
import com.icetech.common.constants.ServiceEnum;
import com.icetech.common.constants.TimeOutConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.WebSocketMessage;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.ReflectUtils;
import com.icetech.common.utils.Slf4jUtils;
import com.icetech.common.utils.SpringUtils;
import com.icetech.common.utils.UUIDTools;
import com.icetech.oss.OssService;
import com.icetech.redis.handle.IceProxyMsgHandle;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 端云架构的发送消息
 */
@Component
@Slf4j
public class P2cDownHandle {
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private TaskCenterServiceImpl taskCenterService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private OssService ossService;
    @Resource
    private StringRedisTemplate stringRedisTemplate;
    @Autowired
    private IceProxyMsgHandle iceProxyMsgHandle;
    @Autowired
    private MqPushService mqPushService;
    //关闭连接命令
    private static final String CMD_CLOSE = "CLOSE";

    /**
     * 同步返回的服务
     */
    private static final List<String> SYNC_CMD_LIST = new ArrayList<>();
    static {
        SYNC_CMD_LIST.add(P2cDownCmdEnum.远程开关闸.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.名单数据清空.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.屏显信息.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.出场数据同步.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.入场数据同步.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.在场车辆查询.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.在场车辆清除.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.名单数据查询.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.LCD屏显语音信息.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.批量下发.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.升级指令下发.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.车队模式下发.getCmd());
        SYNC_CMD_LIST.add(P2cDownCmdEnum.图片抓拍.getCmd());

    }

    /**
     *
     */
    private static final List<Integer> ALL_RET_EXCLUDE_LIST = new ArrayList<>();
    static {
        ALL_RET_EXCLUDE_LIST.add(P2cDownCmdEnum.通道权限下发.getCmdType());
        ALL_RET_EXCLUDE_LIST.add(P2cDownCmdEnum.车场权限下发.getCmdType());
        ALL_RET_EXCLUDE_LIST.add(P2cDownCmdEnum.遥控器对应关系.getCmdType());
        ALL_RET_EXCLUDE_LIST.add(P2cDownCmdEnum.计费规则.getCmdType());
        ALL_RET_EXCLUDE_LIST.add(P2cDownCmdEnum.名单数据清空.getCmdType());
    }

    /**
     *
     * @param parkCode
     * @param message
     * @return
     */
    public <T> String[] send2Park(String parkCode, Message<T> message){
        List<ParkConnectedDeviceVo> list = getParkConnectedDevices(parkCode, null);
        return list.stream().map(vo -> send(parkCode, vo.getDeviceNo(), message))
                .toArray(String[]::new);
    }

    /**
     * 已连接相机列表
     * @param parkCode
     * @return
     */
    private List<ParkConnectedDeviceVo> getParkConnectedDevices(String parkCode, P2cDownHandle.ConnectedDeviceFilter filter) {
        List<ParkConnectedDeviceVo> parkConnectList = cacheHandle.getParkConnectList(parkCode);
        if (CollectionUtils.isEmpty(parkConnectList)){
            log.info("车场没有已连接的相机, parkCode[{}]", parkCode);
            return new ArrayList<>();
        }
        if (filter != null) {
            parkConnectList = parkConnectList.stream()
                    .filter(vo -> !vo.getDeviceNo().equals(filter.excludeSerialNumber))
                    .filter(vo -> vo.getInandoutType().equals(filter.inandoutType))
                    .collect(Collectors.toList());
        }
        if (parkConnectList.size() == 0) {
            log.info("车场未找到符合条件的相机, parkCode[{}], filter[{}]", parkCode, filter);
        }
        return parkConnectList;
    }

    /**
     * 给某一个设备发送消息
     * @param parkCode
     * @param serialNumber
     * @param message
     * @return
     */
    public <T> String send(String parkCode, String serialNumber, Message<T> message){
        return send(parkCode, serialNumber, message, null);
    }
    public <T> String send(String parkCode, String serialNumber, Message<T> message, String topic){
        return send(parkCode, serialNumber, message, topic, null);
    }
    public <T> String send(String parkCode, String serialNumber, Message<T> message, String topic, String extraInfo){

        Integer reqServiceType = message.getReqServiceType();
        T payload = message.getPayload();

        String clientName = parkCode + "_" + serialNumber;
        P2cBaseRequest<T> p2cBaseRequest = new P2cBaseRequest<>();
        p2cBaseRequest.setCmd(P2cDownCmdEnum.getCmd(reqServiceType));
        String messageId = UUIDTools.getUuid();
        p2cBaseRequest.setMessageId(messageId);
        TokenDeviceVo tokenInfo = cacheHandle.getDeviceInfo(serialNumber);
        log.info("serialNumber[{}] tokenInfo[{}]" ,serialNumber, tokenInfo);
        if(tokenInfo == null){
            return null;
        }
        p2cBaseRequest.setToken(tokenInfo.getToken());
        p2cBaseRequest.setBizContent(payload);
        /*
         * 定制开始
         */
        //天町大厦定制
        if (DingZhiFuncConstants.DZ003_PARKS.contains(parkCode)) {
            if (DingZhiFuncConstants.DZ003_DEVICES.contains(serialNumber)) {
                if (!P2cDownCmdEnum.剩余空车位.getCmdType().equals(reqServiceType)
                        && !P2cDownCmdEnum.自定义语音屏显配置.getCmdType().equals(reqServiceType)) {
                    log.info("天町大厦定制业务，跳过业务数据的下发，序列号：{}", serialNumber);
                    return messageId;
                }else if (P2cDownCmdEnum.自定义语音屏显配置.getCmdType().equals(reqServiceType)){
                    LedsoundConfigRequest ledsoundConfigRequest = (LedsoundConfigRequest)payload;
                    List<LedsoundConfigRequest.LedConfig> ledConfigs = ledsoundConfigRequest.getLedConfig();
                    for (LedsoundConfigRequest.LedConfig ledConfig : ledConfigs) {
                        Integer showScene = ledConfig.getShowScene();
                        if (LedsoundConfigRequest.SceneEnum.入口空闲显示.scene == showScene) {
                            ledConfig.setContent("无人值守停车场 {3}");
                        } else if (LedsoundConfigRequest.SceneEnum.出口空闲显示.scene == showScene) {
                            ledConfig.setContent("无人值守停车场 {3}");
                        }
                    }
                }
            }
        }
        // 定制结束
        String paraJson = JsonUtils.toString(p2cBaseRequest);
        boolean success;
        if (tokenInfo.getSource() == 1) {
            success = pushAll(serialNumber, p2cBaseRequest);
            log.info("<端云WS下发> 广播{}，相机:[{}]，内容为:{}", success ? "成功" : "失败", clientName, paraJson);
        } else {//阿里IOT
            success = false;
            log.warn("<端云下发> 未知下发通道[{}]，相机:[{}]，内容为:{}", tokenInfo.getSource(), clientName, paraJson);
        }
        if (success) {
            addRequestRecord(message, parkCode, messageId, payload, serialNumber, topic, extraInfo);
            if (P2cDownCmdEnum.屏显信息.getCmdType().equals(reqServiceType)) {
                mqPushService.pushP2cLedScreen(message.getParkId(), serialNumber, (HintRequest) message.getPayload());
            } else if (P2cDownCmdEnum.LCD屏显语音信息.getCmdType().equals(reqServiceType)) {
                mqPushService.pushP2cLcdScreen(message.getParkId(), serialNumber, (LcdHintRequest) message.getPayload());
            }
        }
        return success ? messageId : null;
    }

    /**
     * 响应处理
     *
     * @param p2cBaseResponse
     * @param parkId
     * @param serviceType
     * @return
     */
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, Integer serviceType) {
        String messageId = p2cBaseResponse.getMessageId();
        SendInfoRecord<Object> sendinfoRecord = redisUtils.get(RedisKeyConstants.MQ_RECORD_PREFIX + messageId, SendInfoRecord.class);
        if (sendinfoRecord == null) {
            sendinfoRecord = new SendInfoRecord<>();
            sendinfoRecord.setServiceType(serviceType);
            sendinfoRecord.setMessageId(messageId);
            sendinfoRecord.setParkId(parkId);
        }
        //serviceType以下发时的为准，主要是解决通道权限下发和车场权限下发的cmd一样的问题
        serviceType = sendinfoRecord.getServiceType();

        Integer code = p2cBaseResponse.getCode();
        String cmd = p2cBaseResponse.getCmd();
        String msg = p2cBaseResponse.getMsg();
        cmd = cmd.substring(0, cmd.indexOf(AbstractService.CMD_SUFFIX));
        String data = p2cBaseResponse.getData();
        String key = RedisConstants.RESP_MSG_PROFILE + messageId;
        //同步返回条件：不包含数据下发业务，且业务ID为空
        if (SYNC_CMD_LIST.contains(cmd) && sendinfoRecord.getServiceId() == null) {
            ObjectResponse<String> retResponse = new ObjectResponse<>();
            retResponse.setCode(String.valueOf(code));
            retResponse.setMsg(msg);
            if (code.equals(CodeEnum.成功.getCode())) {//如果相机处理成功
                if (StringUtils.isNotEmpty(p2cBaseResponse.getResultPath())){
                    //从oss 获取配置
                    File oss2File = ossService.getOSS2File(p2cBaseResponse.getResultPath());
                    //读取文件内容
                    data = FileUtil.readString(oss2File, "utf-8");
                }
            }
            retResponse.setData(data);
            redisUtils.set(key, retResponse, TimeOutConstants.REDIS_TIMEOUT);
            try {
                NotifyService<Object> notifyService = SendServiceFactory.getP2cBean(cmd, NotifyService.class);
                sendinfoRecord.setParams(JSONObject.parseObject(JsonUtils.toString(sendinfoRecord.getParams()),
                        ReflectUtils.getInterfaceGenericTypes(NotifyService.class, notifyService)[0]));
                notifyService.notify(messageId, retResponse, sendinfoRecord);
            } catch (Exception e) {
                log.warn("{} Bean error", cmd, e);
            }
            int valueLen = retResponse.toString().length();
            log.info("[相机响应处理] 写入redis成功,key[{}],value[{}]", key, valueLen > 1024 ? retResponse.toString().substring(0, 1024)  + "..." : retResponse);
        }else{
            Long serviceId = sendinfoRecord.getServiceId();
            if (serviceId != null){
                NotifyRequest notifyRequest = new NotifyRequest();
                notifyRequest.setTargetService(ServiceEnum.Data.getType());
                notifyRequest.setOne(sendinfoRecord.getTarget());
                notifyRequest.setServiceId(serviceId);
                notifyRequest.setServiceType(serviceType);
                notifyRequest.setTaskId(sendinfoRecord.getTaskId());
                notifyRequest.setRecordId(sendinfoRecord.getRecordId());
                // 回调Task的条件：
                // 1、所有设备都返回成功时；
                // 2、响应结果失败
                if (code.equals(CodeEnum.成功.getCode()) || code == 405) {
                    notifyRequest.setSuccess(true);
                    SendRequest sendRequest = new SendRequest();
                    sendRequest.setParkId(parkId);
                    sendRequest.setServiceId(serviceId);
                    sendRequest.setServiceType(serviceType);
                    sendRequest.setTaskId(sendinfoRecord.getTaskId());
                    sendRequest.setRecordId(sendinfoRecord.getRecordId());

                    if (ALL_RET_EXCLUDE_LIST.contains(serviceType)){
                        taskCenterService.notify(notifyRequest);
                    }else{
                        //是否所有设备都响应成功
                        boolean isAllRet = cacheHandle.retSendDevice(sendRequest, sendinfoRecord.getTarget());
                        if (isAllRet) {
                            log.info("准备通知TaskCenter：{}", notifyRequest);
                            taskCenterService.notify(notifyRequest);
                        }
                    }

                } else {
                    notifyRequest.setSuccess(false);
                    if (msg == null){
                        msg = "未知错误";
                    }
                    notifyRequest.setCause(msg);
                    taskCenterService.notify(notifyRequest);
                }
            }
        }
    }
    /**
     * 添加请求记录
     * @param message
     * @param messageId
     * @param t
     * @param serialNumber
     */
    private <T> void addRequestRecord(Message<T> message, String parkCode, String messageId, Object t,
                                  String serialNumber, String topic, String extraInfo) {
        SendInfoRecord<Object> sendinfoRecord = new SendInfoRecord<>();
        sendinfoRecord.setMessageId(messageId);
        sendinfoRecord.setParkCode(parkCode);
        sendinfoRecord.setParkId(message.getParkId());
        sendinfoRecord.setParams(t);
        sendinfoRecord.setServiceId(message.getServiceId());
        sendinfoRecord.setServiceType(message.getServiceType());
        sendinfoRecord.setTarget(serialNumber);
        sendinfoRecord.setOperType(SendOperTypeEnum.请求.getOperType());
        sendinfoRecord.setTaskId(message.getTaskId());
        sendinfoRecord.setRecordId(message.getRecordId());
        sendinfoRecord.setTopic(topic);
        sendinfoRecord.setEnv(SpringUtils.getActiveProfile());
        sendinfoRecord.setExtraInfo(extraInfo);
        redisUtils.set(RedisKeyConstants.MQ_RECORD_PREFIX + messageId, sendinfoRecord, TimeOutConstants.REDIS_TIMEOUT);
    }

    /**
     * 发送给不包括当前相机的本车场其他相机
     * @param parkCode
     * @param serialNumber
     * @param message
     * @return
     */
    public <T> String[] send2ParkOtherDevice(String parkCode, String serialNumber, Message<T> message) {
        P2cDownHandle.ConnectedDeviceFilter filter = new ConnectedDeviceFilter();
        filter.setExcludeSerialNumber(serialNumber);
        return sendArray(parkCode, filter, message);
    }

    /**
     * 下发给其他出口
     * @param parkCode
     * @param serialNumber
     * @param message
     * @return
     */
    public <T> String[] send2ParkOtherExit(String parkCode, String serialNumber, Message<T> message) {
        P2cDownHandle.ConnectedDeviceFilter filter = new ConnectedDeviceFilter();
        filter.setExcludeSerialNumber(serialNumber);
        filter.setInandoutType(2);
        return sendArray(parkCode, filter, message);
    }

    private <T> String[] sendArray(String parkCode, P2cDownHandle.ConnectedDeviceFilter filter, Message<T> message) {
        List<ParkConnectedDeviceVo> list = getParkConnectedDevices(parkCode, filter);
        return list.stream().map(vo -> send(parkCode, vo.getDeviceNo(), message))
                .toArray(String[] :: new);
    }

    public <T> String[] send2Exit(String parkCode, Message<T> message) {
        P2cDownHandle.ConnectedDeviceFilter filter = new ConnectedDeviceFilter();
        filter.setInandoutType(2);
        return sendArray(parkCode, filter, message);
    }

    public <T> String[] send2Enter(String parkCode, Message<T> message) {
        P2cDownHandle.ConnectedDeviceFilter filter = new ConnectedDeviceFilter();
        filter.setInandoutType(1);
        return sendArray(parkCode, filter, message);
    }
    /**
     * 广播消息到各个订阅者
     * @param sn                sn
     * @param p2cBaseRequest    请求参数
     */
    public boolean pushAll(String sn, P2cBaseRequest<?> p2cBaseRequest) {
        return pushAll(sn, p2cBaseRequest, "p2c");
    }

    /**
     * 广播消息到各个订阅者
     * @param sn                sn
     * @param p2cBaseRequest    请求参数
     */
    public boolean pushAll(String sn, P2cBaseRequest<?> p2cBaseRequest, String module) {
        try {
            WebSocketMessage message = WebSocketMessage.buildMessage(StringUtils.isEmpty(module) ? "p2c" : module, sn, JSON.toJSONString(p2cBaseRequest))
                            .setProtocolType(1).setTrace(Slf4jUtils.getTraceId());
            return iceProxyMsgHandle.sendSocketMsg(message).getData();
        } catch (Exception e) {
            log.error("[redis广播发送] clientName[{}]广播失败", sn, e);
            return false;
        }
    }

    /**
     * 广播消息到各个订阅者
     * @param clientName
     * @param content
     */
    public boolean pushAllOld(String clientName, String content){
        try{
            content = clientName + content;
            stringRedisTemplate.convertAndSend(RedisMsg.TOPIC, content);
        }catch (Exception e){
            log.error("[redis广播发送] clientName[{}]广播失败", clientName, e);
            return false;
        }
        return true;
    }

    /**
     * 关闭连接
     * @param sn
     * @return
     */
    public boolean closeAll(String sn){
        try{
            WebSocketMessage message = WebSocketMessage.buildClose("p2c", sn)
                    .setProtocolType(1)
                    .setTrace(Slf4jUtils.getTraceId());
            log.warn("<redis广播关闭> clientName：{} ", sn);
            return iceProxyMsgHandle.sendSocketMsg(message).getData();
        }catch (Exception e){
            log.error("<redis广播关闭> clientName：{} 广播失败", sn, e);
            return false;
        }
    }

    /**
     * 关闭连接
     * @param clientName
     * @return
     */
    public boolean closeAllOld(String clientName){
        try{
            log.warn("<redis广播关闭> clientName：{} ", clientName);
            stringRedisTemplate.convertAndSend(RedisMsg.TOPIC, CMD_CLOSE  + ":" + clientName);
        }catch (Exception e){
            log.error("<redis广播关闭> clientName：{} 广播失败", clientName, e);
            return false;
        }
        return true;
    }

    @Getter
    @Setter
    @ToString
    public static class ConnectedDeviceFilter{
        /**
         * 不包含的相机序列号
         */
        public String excludeSerialNumber;
        /**
         * 出入口类型
         */
        public Integer inandoutType;
    }
}
