package com.icetech.park.service.handle;

import com.alibaba.fastjson.JSONObject;
import com.icetech.cloudcenter.api.NotifyService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.FullCloudDownCmdEnum;
import com.icetech.cloudcenter.domain.enumeration.SendOperTypeEnum;
import com.icetech.cloudcenter.domain.request.NotifyRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.constants.ServiceEnum;
import com.icetech.common.constants.TimeOutConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.ReflectUtils;
import com.icetech.common.utils.SpringUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.fullcloud.api.DownMsgService;
import com.icetech.park.domain.dto.full.RequestDTO;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.factory.SendServiceFactory;
import com.icetech.park.service.impl.TaskCenterServiceImpl;
import com.icetech.third.domain.entity.third.SendInfoRecord;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * 纯云架构的发送消息
 */
@Component
@Slf4j
public class FullCloudDownHandle {
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private TaskCenterServiceImpl taskCenterService;
    @Autowired
    private DownMsgService downMsgService;

    /**
     * 同步返回的服务
     */
    private static final List<String> SYNC_CMD_LIST = new ArrayList<>();
    static {
        SYNC_CMD_LIST.add(FullCloudDownCmdEnum.远程开关闸.getCmd());
        SYNC_CMD_LIST.add(FullCloudDownCmdEnum.白名单数据查询.getCmd());
        SYNC_CMD_LIST.add(FullCloudDownCmdEnum.图片抓拍.getCmd());
        SYNC_CMD_LIST.add(FullCloudDownCmdEnum.屏显信息下发.getCmd());
    }

    private static final List<Integer> ALL_RET_EXCLUDE_LIST = new ArrayList<>();
    static {
        ALL_RET_EXCLUDE_LIST.add(FullCloudDownCmdEnum.名单数据清空.getCmdType());
    }

    /**
     * 给某一个设备发送消息
     * @param parkCode 车场编号
     * @param serialNumber 相机序列号
     * @param message 消息模板
     * @return messageId
     */
    public <T> String send(String parkCode, String serialNumber, Message<T> message){
        return send(parkCode, serialNumber, message, null);
    }
    public <T> String send(String parkCode, String serialNumber, Message<T> message, String topic){
        return send(parkCode, serialNumber, message, topic, null);
    }
    public <T> String send(String parkCode, String serialNumber, Message<T> message, String topic, String extraInfo){

        Integer reqServiceType = message.getReqServiceType();
        T payload = message.getPayload();

        String clientName = parkCode + "_" + serialNumber;
        RequestDTO requestDTO = new RequestDTO();
        requestDTO.setType(FullCloudDownCmdEnum.getCmd(reqServiceType));
        requestDTO.setBody(payload);

        TokenDeviceVo tokenInfo = cacheHandle.getDeviceInfo(serialNumber);
        log.info("serialNumber[{}] tokenInfo[{}]" ,serialNumber, tokenInfo);
        if(tokenInfo == null){
            return null;
        }

        ObjectResponse<String> objectResponse;
        if (StringUtils.isNotBlank(message.getMessageId())) {
            objectResponse = downMsgService.sendDownMessage(requestDTO, serialNumber, message.getMessageId());
        } else {
            objectResponse = downMsgService.sendDownMessage(requestDTO, serialNumber);
        }

        boolean success = ObjectResponse.isSuccess(objectResponse);
        String messageId = success ? objectResponse.getData() : null;
        log.info("[纯云MQTT] 下发{}，相机:[{}]，内容为:{}, 返回[{}]", success ? "成功" : "失败", clientName, JsonUtils.toString(requestDTO), messageId);

        if (success) {
            addRequestRecord(message, parkCode, messageId, payload, serialNumber, topic, extraInfo);
        }
        return messageId;
    }

    /**
     * 响应处理
     *
     * @param p2cBaseResponse 相机响应结果
     * @param parkId 车场ID
     * @param serviceType 业务类型
     * @return void
     */
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, Integer serviceType) {
        String messageId = p2cBaseResponse.getMessageId();
        SendInfoRecord<Object> sendinfoRecord = redisUtils.get(RedisKeyConstants.MQ_RECORD_PREFIX + messageId, SendInfoRecord.class);
        if (sendinfoRecord == null) {
            sendinfoRecord = new SendInfoRecord<>();
            sendinfoRecord.setServiceType(serviceType);
            sendinfoRecord.setMessageId(messageId);
            sendinfoRecord.setParkId(parkId);
        }
        //继续下发下一个消息，只有两条消息都成功才算成功
        if (sendinfoRecord.getNextPayLoad() != null) {
            Message<Object> message = new Message<>();
            BeanUtils.copyProperties(sendinfoRecord, message);
            message.setPayload(sendinfoRecord.getNextPayLoad());
            message.setNextPayLoad(null);
            message.setMessageId(sendinfoRecord.getMessageId());
            send(sendinfoRecord.getParkCode(), sendinfoRecord.getTarget(), message, sendinfoRecord.getTopic(), sendinfoRecord.getExtraInfo());
            return;
        }
        //serviceType以下发时的为准，主要是解决通道权限下发和车场权限下发的cmd一样的问题
        serviceType = sendinfoRecord.getServiceType();

        Integer code = p2cBaseResponse.getCode();
        String cmd = p2cBaseResponse.getCmd();
        String msg = p2cBaseResponse.getMsg();
        String data = p2cBaseResponse.getData();
        String key = RedisConstants.RESP_MSG_PROFILE + messageId;
        //同步返回条件：不包含数据下发业务，且业务ID为空
        if (SYNC_CMD_LIST.contains(cmd) && sendinfoRecord.getServiceId() == null) {
            ObjectResponse<String> retResponse = new ObjectResponse<>();
            retResponse.setCode(String.valueOf(code));
            retResponse.setMsg(msg);
            retResponse.setData(data);
            redisUtils.set(key, retResponse, TimeOutConstants.REDIS_TIMEOUT);
            try {
                NotifyService<Object> notifyService = SendServiceFactory.getFullCloudBean(cmd, NotifyService.class);
                sendinfoRecord.setParams(JSONObject.parseObject(JsonUtils.toString(sendinfoRecord.getParams()),
                        ReflectUtils.getInterfaceGenericTypes(NotifyService.class, notifyService)[0]));
                notifyService.notify(messageId, retResponse, sendinfoRecord);
            } catch (ResponseBodyException | ClassCastException e) {
            } catch (Exception e) {
                log.warn("{} Bean error", cmd, e);
            }
            int valueLen = retResponse.toString().length();
            log.info("[相机响应处理] 写入redis成功,key[{}],value[{}]", key, valueLen > 1024 ? retResponse.toString().substring(0, 1024)  + "..." : retResponse);
        } else {
            Long serviceId = sendinfoRecord.getServiceId();
            if (serviceId != null){
                NotifyRequest notifyRequest = new NotifyRequest();
                notifyRequest.setTargetService(ServiceEnum.Data.getType());
                notifyRequest.setOne(sendinfoRecord.getTarget());
                notifyRequest.setServiceId(serviceId);
                notifyRequest.setServiceType(serviceType);
                notifyRequest.setTaskId(sendinfoRecord.getTaskId());
                notifyRequest.setRecordId(sendinfoRecord.getRecordId());
                // 回调Task的条件：
                // 1、所有设备都返回成功时；
                // 2、响应结果失败
                if (code.equals(CodeEnum.成功.getCode()) || code == 405) {
                    notifyRequest.setSuccess(true);
                    SendRequest sendRequest = new SendRequest();
                    sendRequest.setParkId(parkId);
                    sendRequest.setServiceId(serviceId);
                    sendRequest.setServiceType(serviceType);
                    sendRequest.setTaskId(sendinfoRecord.getTaskId());
                    sendRequest.setRecordId(sendinfoRecord.getRecordId());

                    if (ALL_RET_EXCLUDE_LIST.contains(serviceType)){
                        taskCenterService.notify(notifyRequest);
                    }else{
                        //是否所有设备都响应成功
                        boolean isAllRet = cacheHandle.retSendDevice(sendRequest, sendinfoRecord.getTarget());
                        if (isAllRet) {
                            log.info("准备通知TaskCenter：{}", notifyRequest);
                            taskCenterService.notify(notifyRequest);
                        }
                    }

                } else {
                    notifyRequest.setSuccess(false);
                    if (msg == null){
                        msg = "未知错误";
                    }
                    notifyRequest.setCause(msg);
                    taskCenterService.notify(notifyRequest);
                }
            }
        }
    }
    /**
     * 添加请求记录
     * @param message 消息模板
     * @param messageId 消息ID
     * @param t 参数
     * @param serialNumber 序列号
     */
    private <T> void addRequestRecord(Message<T> message, String parkCode, String messageId, Object t,
                                  String serialNumber, String topic, String extraInfo) {
        SendInfoRecord<Object> sendinfoRecord = new SendInfoRecord<>();
        sendinfoRecord.setMessageId(messageId);
        sendinfoRecord.setParkCode(parkCode);
        sendinfoRecord.setParkId(message.getParkId());
        sendinfoRecord.setParams(t);
        sendinfoRecord.setServiceId(message.getServiceId());
        sendinfoRecord.setServiceType(message.getServiceType());
        sendinfoRecord.setReqServiceType(message.getReqServiceType());
        sendinfoRecord.setTarget(serialNumber);
        sendinfoRecord.setOperType(SendOperTypeEnum.请求.getOperType());
        sendinfoRecord.setTaskId(message.getTaskId());
        sendinfoRecord.setRecordId(message.getRecordId());
        sendinfoRecord.setTopic(topic);
        sendinfoRecord.setEnv(SpringUtils.getActiveProfile());
        sendinfoRecord.setExtraInfo(extraInfo);
        sendinfoRecord.setNextPayLoad(message.getNextPayLoad());
        redisUtils.set(RedisKeyConstants.MQ_RECORD_PREFIX + messageId, sendinfoRecord, TimeOutConstants.REDIS_TIMEOUT);
    }

}
