package com.icetech.park.service.handle;

import com.icetech.common.utils.DateTools;
import com.icetech.cloudcenter.domain.response.UpgradeProgressResponse;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.dao.FirmwareUpgradeDao;
import com.icetech.park.domain.entity.FirmwareUpgrade;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Map;

/**
 * 设备升级处理
 * @author fangct
 */
@Service
@Slf4j
public class DeviceUpgradeHandle {

    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private FirmwareUpgradeDao firmwareUpgradeDao;

    public UpgradeProgressResponse progress(String taskId) {
        FirmwareUpgrade firmwareUpgrade = firmwareUpgradeDao.selectByTaskId(taskId);
        if (firmwareUpgrade == null) {
            return null;
        }
        UpgradeProgressResponse upgradeProgressResponse = new UpgradeProgressResponse();

        String key = RedisConstants.UPGRADE_TASK_HASH + taskId;
        Long sendTime;
        if (redisUtils.exists(key)){
            Map<String, Object> objectObjectMap = redisUtils.hGetAll(key, Object.class);
            log.info("[升级任务进度查询] redis的情况[{}]", objectObjectMap);
            sendTime = ((Integer) objectObjectMap.get("sendTime")).longValue();
            upgradeProgressResponse.setTaskStep((Integer) objectObjectMap.get("step"));
            Integer stepStatus = (Integer) objectObjectMap.get("stepStatus");
            //不是进行中的步骤，才有结束时间
            if (stepStatus != 1){
                upgradeProgressResponse.setResultTime((String) objectObjectMap.get("updateTime"));
            }
            upgradeProgressResponse.setStepStatus(stepStatus);
        }else{
            sendTime = firmwareUpgrade.getActionTime().getTime()/1000;
            upgradeProgressResponse.setTaskStep(3);
            //默认为进行中
            upgradeProgressResponse.setStepStatus(1);
            Integer status = firmwareUpgrade.getStatus();
            if (status == 3){
                upgradeProgressResponse.setStepStatus(2);
            }
            if (status == 4){
                upgradeProgressResponse.setStepStatus(3);
            }
            //待执行的任务没有时间
            if (upgradeProgressResponse.getStepStatus() == 1){
                upgradeProgressResponse.setResultTime(null);
            }else{
                if (status == 3){
                    upgradeProgressResponse.setResultTime(DateTools.getFormat(firmwareUpgrade.getSuccessTime()));
                }else{
                    upgradeProgressResponse.setResultTime(DateTools.getFormat(firmwareUpgrade.getUpdateTime()));
                }
            }
        }
        //是否超时的处理
        Integer taskStep = upgradeProgressResponse.getTaskStep();
        Integer stepStatus = upgradeProgressResponse.getStepStatus();
        //如果是进行中的任务，再判断是否超时
        if (stepStatus == 1 && validTimeout(sendTime, taskStep)){
            firmwareUpgrade.setStatus(FirmwareUpgrade.StatusEnum.升级失败.status);
            firmwareUpgrade.setReason("设备响应超时");
            firmwareUpgrade.setUpdateTime(new Date());
            firmwareUpgradeDao.updateById(firmwareUpgrade);

            String resultTime = DateTools.Date();
            if (redisUtils.exists(key)){
                redisUtils.hPut(key, "stepStatus", 3);
                redisUtils.hPut(key, "updateTime", resultTime);
            }
            upgradeProgressResponse.setStepStatus(3);
            upgradeProgressResponse.setResultTime(resultTime);
        }

        return upgradeProgressResponse;
    }

    private boolean validTimeout(Long sendTime, Integer taskStep) {
        long now = DateTools.unixTimestamp();
        if (taskStep == 1){
            //第一步固件地址下发的超时时间为1分钟
            if (now - sendTime > 60){
                return true;
            }
        }else {
            //第二三步固件地址下发的超时时间为10分钟
            if (now - sendTime > 600){
                return true;
            }
        }
        return false;
    }

    public void versionModify(String sn, String newVer) {
        //参数校验
        if (sn == null || newVer == null){
            return;
        }
        //查询执行中的任务
        FirmwareUpgrade firmwareUpgrade = firmwareUpgradeDao.selectTaskBySnAndStatus(sn, FirmwareUpgrade.StatusEnum.升级中.status);
        if (firmwareUpgrade == null){
            return;
        }
        String targetVer = firmwareUpgrade.getTargetVer();
        String key = RedisConstants.UPGRADE_TASK_HASH + firmwareUpgrade.getTaskId();
        if (newVer.equals(targetVer)){
            log.info("[设备固件版本号变更] 升级成功,taskId[{}]sn[{}]当前版本[{}]",firmwareUpgrade.getTaskId(), sn, newVer);
            firmwareUpgrade.setStatus(FirmwareUpgrade.StatusEnum.升级成功.status);
            firmwareUpgrade.setSuccessTime(new Date());

            if (redisUtils.exists(key)){
                redisUtils.hPut(key, "step", 3);
                redisUtils.hPut(key, "stepStatus", 2);
                redisUtils.hPut(key, "updateTime", DateTools.Date());
            }
        }else{
            log.warn("[设备固件版本号变更] 升级失败,新版本号与升级版本号不一致,taskId[{}]sn[{}]当前版本[{}]目标升级版本[{}]",firmwareUpgrade.getTaskId(), sn, newVer, targetVer);
            firmwareUpgrade.setStatus(FirmwareUpgrade.StatusEnum.升级失败.status);
            firmwareUpgrade.setReason("新版本号与升级版本号不一致");

            if (redisUtils.exists(key)){
                redisUtils.hPut(key, "step", 3);
                redisUtils.hPut(key, "stepStatus", 3);
                redisUtils.hPut(key, "updateTime", DateTools.Date());
            }
        }
        firmwareUpgrade.setUpdateTime(new Date());
        firmwareUpgradeDao.updateById(firmwareUpgrade);
    }
}
