package com.icetech.park.service.handle;

import cn.hutool.core.util.StrUtil;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.DateTools;
import com.icetech.cloudcenter.domain.request.p2c.VipInfoRequest;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.dao.BatchsendTaskSubDao;
import com.icetech.park.domain.entity.BatchsendTaskSub;
import com.icetech.cloudcenter.domain.request.p2c.BizBatchDownRequest;
import com.icetech.cloudcenter.domain.vo.BatchSendRepeatVO;
import com.icetech.cloudcenter.domain.vo.BatchSendVO;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.upload.IDataUploadService;
import com.icetech.park.handle.CacheHandle;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadPoolExecutor;

import static com.icetech.cloudcenter.domain.constants.BatchDownMsgConstants.*;

/**
 * @author fangct
 */
@Service
@Slf4j
public class BatchDownBizHandle<T> {

    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private BatchsendTaskSubDao batchsendTaskSubDao;
    @Autowired
    @Qualifier("uploadMonthCards")
    private IDataUploadService<T> uploadMonthCards;
    @Autowired
    @Qualifier("uploadVips")
    private IDataUploadService<T> uploadVips;
    @Autowired
    @Qualifier("uploadBlacklist")
    private IDataUploadService<T> uploadBlacklist;
    @Autowired
    private P2cDownHandle p2CDownHandle;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;
    //数据集合缓存过期时间
    int DATA_LIST_EXPIRE = 60 * 60;
    //旧版本VIP下发的表头
    String[] oldVIPHeaders = {"id","vipId","vipTypeName","plateNum","phone","startDate","endDate","discountType","discountNumber", "billtypecode"};

    public void batchDown(BatchSendVO vo, List<T> dataList, List<Object> idList, String[] headers, int cmdType) {
        String parkCode = vo.getParkCode();
        Long parkId = vo.getParkId();
        String taskId = vo.getTaskId();
        List<BatchSendVO.SubTaskInfo> subTaskInfos = vo.getSubTaskInfos();
        int size = dataList.size();

        VersionFile versionFile = new VersionFile();
        int dataType = 0;
        if (P2cDownCmdEnum.月卡.getCmdType().equals(cmdType)){
            versionFile = new VersionFile(uploadMonthCards, parkCode, taskId, cmdType);
            dataType = 1;
        }else if (P2cDownCmdEnum.黑名单下发.getCmdType().equals(cmdType)){
            versionFile = new VersionFile(uploadBlacklist, parkCode, taskId, cmdType);
            dataType = 2;
        }else if (P2cDownCmdEnum.VIP车辆.getCmdType().equals(cmdType)){
            versionFile = new VersionFile(uploadVips, parkCode, taskId, cmdType);
            dataType = 3;
        }
        log.info("[数据批量下发业务] 文件上传完成, 参数[{}]", vo);

        BizBatchDownRequest bizBatchDownRequest = new BizBatchDownRequest();
        bizBatchDownRequest.setDataType(dataType);
        bizBatchDownRequest.setIsClear(1);
        bizBatchDownRequest.setTotalNum(size);
        for (int i = 0; i < subTaskInfos.size(); i++){
            BatchSendVO.SubTaskInfo subTaskInfo = subTaskInfos.get(i);

            Integer subTaskId = subTaskInfo.getSubTaskId();
            bizBatchDownRequest.setSubTaskId(String.valueOf(subTaskId));

            //缓存cardId列表，为匹配失败的id集合
            redisUtils.lRightPushAll(RedisConstants.BATCH_DOWN_DATA_LIST + subTaskId + ":" + cmdType, idList);
            redisUtils.expire(RedisConstants.BATCH_DOWN_DATA_LIST + subTaskId + ":" + cmdType, DATA_LIST_EXPIRE);

            //缓存子任务详情
            String subTaskKey = RedisConstants.BATCH_DOWN_SUBTASK_HASH + subTaskId;
            redisUtils.hPut(subTaskKey, "totalNum", size);

            String sn = subTaskInfo.getSn();
            if (StrUtil.isBlank(sn)){
                redisUtils.hPut(subTaskKey, "status", 3);
                redisUtils.hPut(subTaskKey, "failNum", size);
                redisUtils.hPut(subTaskKey, "reason", NO_CAMERA);

                finishErrorTask(size, subTaskId, NO_CAMERA);
            }else{

                TokenDeviceVo tokenDeviceVo = cacheHandle.getDeviceInfo(sn);
                String version = tokenDeviceVo == null ? null : tokenDeviceVo.getVersion();
                //批量下发后的版本不再初始化下发任何数据
                if (version != null && P2cVersionEnum.getIndex(version) < P2cVersionEnum.版本6.getIndex()){
                    String error =  "相机版本过低，无法下发";
                    redisUtils.hPut(subTaskKey, "status", 3);
                    redisUtils.hPut(subTaskKey, "failNum", size);
                    redisUtils.hPut(subTaskKey, "reason", error);
                    finishErrorTask(size, subTaskId, error);
                    continue;
                }
                //下发消息，并异步处理响应结果
                if (P2cDownCmdEnum.VIP车辆.getCmdType().equals(cmdType) && P2cVersionEnum.getIndex(version) < P2cVersionEnum.版本9.getIndex()) {
                    List<VipInfoRequest> dl = (List<VipInfoRequest>) dataList;
                    dl.removeIf(vip -> !Integer.valueOf(1).equals(vip.getDiscountType()));
                    size = dl.size();
                    redisUtils.hPut(subTaskKey, "totalNum", size);
                    if (size == 0) {
                        String error =  "相机版本过低，非全免VIP无需下发";
                        redisUtils.hPut(subTaskKey, "status", 3);
                        redisUtils.hPut(subTaskKey, "failNum", size);
                        redisUtils.hPut(subTaskKey, "reason", error);
                        finishErrorTask(size, subTaskId, error);
                        continue;
                    }
                    bizBatchDownRequest.setBizDataPath(versionFile.getFilePathByVersion(dataList, cmdType + "_old", oldVIPHeaders));
                }else{
                    bizBatchDownRequest.setBizDataPath(versionFile.getFilePathByVersion(dataList, String.valueOf(cmdType), headers));
                }
                Message message = new Message(parkId, P2cDownCmdEnum.批量下发.getCmdType(), bizBatchDownRequest);

                String messageId = p2CDownHandle.send(parkCode, sn, message);
                redisUtils.set(RedisConstants.BATCH_DOWN_OPEN_FLAG + parkId + ":" + cmdType, 1, 600L);
                Integer finalSize = size;
                asyncExecutor.execute(ThreadUtils.wrapTrace(() -> {
                    if (messageId == null) {
                        redisUtils.hPut(subTaskKey, "status", 3);
                        redisUtils.hPut(subTaskKey, "failNum", finalSize);
                        redisUtils.hPut(subTaskKey, "reason", DEVICE_OFF_LINE);
                        finishErrorTask(finalSize, subTaskId, DEVICE_OFF_LINE);
                    } else {
                        ObjectResponse<String> response = cacheHandle.getResponseFromRedis(messageId, 4000L);
                        if (ObjectResponse.isSuccess(response)) {
                            redisUtils.hPut(subTaskKey, "status", 2);
                        } else {
                            String msg = BatchDownUtils.convertMsgAndGetNewMsg(null, response == null ? DEVICE_TIMEOUT : response.getMsg());
                            redisUtils.hPut(subTaskKey, "status", 3);
                            redisUtils.hPut(subTaskKey, "failNum", finalSize);
                            redisUtils.hPut(subTaskKey, "reason", msg);
                            finishErrorTask(finalSize, subTaskId, msg);
                        }
                    }
                }));
            }
        }
    }


    public void repeatBatch(BatchSendRepeatVO vo, List<T> dataList, List<Object> idList, String[] headers, int cmdType) {
        String parkCode = vo.getParkCode();
        Long parkId = vo.getParkId();
        String taskId = vo.getTaskId();
        String sn = vo.getSn();
        int size = dataList.size();

        VersionFile versionFile = new VersionFile();
        int dataType = 0;
        if (P2cDownCmdEnum.月卡.getCmdType().equals(cmdType)){
            versionFile = new VersionFile(uploadMonthCards, parkCode, taskId, cmdType);
            dataType = 1;
        }else if (P2cDownCmdEnum.黑名单下发.getCmdType().equals(cmdType)){
            versionFile = new VersionFile(uploadBlacklist, parkCode, taskId, cmdType);
            dataType = 2;
        }else if (P2cDownCmdEnum.VIP车辆.getCmdType().equals(cmdType)){
            versionFile = new VersionFile(uploadVips, parkCode, taskId, cmdType);
            dataType = 3;
        }
        log.info("[数据批量下发业务-重发] 文件上传完成, 参数[{}]", vo);

        BizBatchDownRequest bizBatchDownRequest = new BizBatchDownRequest();
        bizBatchDownRequest.setDataType(dataType);
        bizBatchDownRequest.setIsClear(dataList.size() == vo.getOldTotalNum() ? 1 : 0);
        bizBatchDownRequest.setTotalNum(size);

        Integer subTaskId = vo.getSubTaskId();
        bizBatchDownRequest.setSubTaskId(String.valueOf(subTaskId));

        //缓存Id列表，为匹配失败的id集合
        String dataListKey = RedisConstants.BATCH_DOWN_DATA_LIST + subTaskId + ":" + cmdType;
        redisUtils.remove(dataListKey);
        redisUtils.lRightPushAll(dataListKey, idList);
        redisUtils.expire(dataListKey, DATA_LIST_EXPIRE);

        //缓存子任务详情
        String subTaskKey = RedisConstants.BATCH_DOWN_SUBTASK_HASH + subTaskId;
        redisUtils.expire(subTaskKey, 60 * 60);
        redisUtils.hPut(subTaskKey, "startTime", DateTools.unixTimestamp());
        if (StrUtil.isBlank(sn)){
            redisUtils.hPut(subTaskKey, "status", 3);
            redisUtils.hPut(subTaskKey, "failNum", size);
            redisUtils.hPut(subTaskKey, "reason", NO_CAMERA);
            finishErrorTask(size, subTaskId, NO_CAMERA);
        }else {

            TokenDeviceVo tokenDeviceVo = cacheHandle.getDeviceInfo(sn);
            String version = tokenDeviceVo == null ? null : tokenDeviceVo.getVersion();
            //批量下发后的版本不再初始化下发任何数据
            if (version != null && P2cVersionEnum.getIndex(version) < P2cVersionEnum.版本6.getIndex()) {
                String error = "相机版本过低，无法下发";
                redisUtils.hPut(subTaskKey, "status", 3);
                redisUtils.hPut(subTaskKey, "failNum", size);
                redisUtils.hPut(subTaskKey, "reason", error);
                finishErrorTask(size, subTaskId, error);
                return;
            }
            //下发消息，并异步处理响应结果
            if (P2cDownCmdEnum.VIP车辆.getCmdType().equals(cmdType) && P2cVersionEnum.getIndex(version) < P2cVersionEnum.版本9.getIndex()) {
                bizBatchDownRequest.setBizDataPath(versionFile.getRepeatFilePathByVersion(dataList, cmdType + "_old", oldVIPHeaders));
            }else{
                bizBatchDownRequest.setBizDataPath(versionFile.getRepeatFilePathByVersion(dataList, String.valueOf(cmdType), headers));
            }
            Message message = new Message(parkId, P2cDownCmdEnum.批量下发.getCmdType(), bizBatchDownRequest);
            String messageId = p2CDownHandle.send(parkCode, sn, message);
            redisUtils.set(RedisConstants.BATCH_DOWN_OPEN_FLAG + parkId + ":" + cmdType, 1, 600L);
            asyncExecutor.execute(ThreadUtils.wrapTrace(() -> {
                String reason = redisUtils.hGet(subTaskKey, "reason", String.class);
                if (messageId == null) {
                    String error = BatchDownUtils.convertMsgAndGetNewMsg(reason, DEVICE_OFF_LINE);
                    redisUtils.hPut(subTaskKey, "status", 3);
                    redisUtils.hPut(subTaskKey, "failNum", size);
                    redisUtils.hPut(subTaskKey, "reason", error);
                    finishErrorTask(size, subTaskId, error);
                } else {
                    ObjectResponse<String> response = cacheHandle.getResponseFromRedis(messageId, 4000L);
                    if (ObjectResponse.isSuccess(response)) {
                        redisUtils.hPut(subTaskKey, "status", 2);
                    } else {
                        String error = BatchDownUtils.convertMsgAndGetNewMsg(reason, response == null ? DEVICE_TIMEOUT : response.getMsg());
                        redisUtils.hPut(subTaskKey, "status", 3);
                        redisUtils.hPut(subTaskKey, "failNum", size);
                        redisUtils.hPut(subTaskKey, "reason", error);
                        finishErrorTask(size, subTaskId, error);
                    }
                }
            }));
        }
    }

    private void finishErrorTask(int size, Integer subTaskId, String reason) {

        BatchsendTaskSub batchsendTaskSub = new BatchsendTaskSub();
        batchsendTaskSub.setId(subTaskId);
        batchsendTaskSub.setStatus(3);
        batchsendTaskSub.setTotalNum(size);
        batchsendTaskSub.setFailNum(size);
        batchsendTaskSub.setReason(reason);
        batchsendTaskSubDao.updateById(batchsendTaskSub);
    }

    /**
     * 不同版本不同的文件
     */
    class VersionFile {
        public Map<String, Object> para = new ConcurrentHashMap<>();
        public Map<String, Object> repeatPara = new ConcurrentHashMap<>();

        private IDataUploadService dataUploadService;
        private String parkCode;
        private int cmdType;
        private String taskId;
        private int index = 0;

        public VersionFile(){

        }
        public VersionFile(IDataUploadService dataUploadService, String parkCode, String taskId, int cmdType){
            this.dataUploadService = dataUploadService;
            this.parkCode = parkCode;
            this.cmdType = cmdType;
            this.taskId = taskId;
        }
        public String getFilePathByVersion(List<T> dataList, String typeFlag, String[] header) {
            if (para.get(typeFlag) != null){
                return (String) para.get(typeFlag);
            }

            String date = DateTools.getFormat(DateTools.DF_, new Date());
            String[] ymd = date.split("-");
            String filePath = new StringBuffer(parkCode)
                    .append("/biz_data/")
                    .append(ymd[0] + ymd[1])
                    .append("/" + taskId + "_" + cmdType + "_" + dataList.size()
                            + "_" + (index++) + RandomUtils.nextInt(10000, 99999) + ".csv")
                    .toString();
            dataUploadService.upload(dataList, header, filePath);
            para.put(typeFlag, filePath);
            return filePath;
        }
        public String getRepeatFilePathByVersion(List<T> dataList, String typeFlag, String[] header) {
            if (repeatPara.get(typeFlag) != null){
                return (String) repeatPara.get(typeFlag);
            }

            String date = DateTools.getFormat(DateTools.DF_, new Date());
            String[] ymd = date.split("-");
            String filePath = new StringBuffer(parkCode)
                    .append("/biz_data/")
                    .append(ymd[0] + ymd[1])
                    .append("/" + taskId + "_" + cmdType + "_" + dataList.size() + "_repeat" + ".csv")
                    .toString();
            dataUploadService.upload(dataList, header, filePath);
            repeatPara.put(typeFlag, filePath);
            return filePath;
        }
    }
}
