package com.icetech.park.service.handle;

import com.alibaba.fastjson.TypeReference;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.vo.p2c.ParkConnectedDeviceVo;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * 通用缓存管理
 * @author fangct
 */
@Slf4j
public class BaseCacheHandle {

    @Autowired
    private RedisUtils redisUtils;
    /**
     * 间隔查询时间，50ms
     */
    private static final int INTERVAL = 50;
    /**
     * 缓存有效期1天6小时
     */
    private static final int EXPIRE_TIME = 3600 * 30;

    /**
     * WS连接关闭后，清除缓存
     *
     * @param sn sn
     * @return 成功/失败
     */
    public boolean closeForClearCache(String sn) {
        TokenDeviceVo tokenInfo = getDeviceInfo(sn);
        if (tokenInfo != null) {
            return closeForClearCache(tokenInfo.getParkCode(), sn, tokenInfo.getInandoutCode());
        } else {
            return false;
        }
    }

    /**
     * WS连接关闭后，清除缓存，定时心跳检测后执行
     *
     * @return 成功/失败
     */
    public boolean closeForClearCache(String parkCode, String serialNumber, String inandoutCode) {
        //车场已连接设备,存放通道信息
        String key1 = getSnKey(serialNumber);
        if (redisUtils.exists(key1)) {
            redisUtils.remove(key1);
        }
        String key2 = getChannelSnKey(parkCode, inandoutCode);
        if (redisUtils.exists(key2)) {
            redisUtils.remove(key2);
        }
        String key = getParkConnectedListKey(parkCode);
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key, 0, -1, ParkConnectedDeviceVo.class);
        for (ParkConnectedDeviceVo vo : list) {
            if (vo.getInandoutCode() == null || vo.getDeviceNo() == null) {
                redisUtils.lRemove(key, 0, vo);
            }
            if (vo.getDeviceNo().equals(serialNumber)) {
                redisUtils.lRemove(key, 0, vo);
            }
        }
        log.info("缓存清除成功，serialNumber[{}]", serialNumber);
        return true;
    }

    protected String getDeviceProfile() {
        return "null";
    }
    protected String getSnKey(String sn) {
        return getDeviceProfile() + RedisConstants.COMMON_SN_PROFILE + sn;
    }
    protected String getChannelSnKey(String parkCode, String channelCode) {
        return getDeviceProfile() + RedisConstants.COMMON_CHANNEL_SN_PROFILE + parkCode + "_" + channelCode;
    }
    protected String getParkConnectedListKey(String parkCode) {
        return getDeviceProfile() + RedisConstants.COMMON_PARK_CONNECTED_LIST_PROFILE + parkCode;
    }
    /**
     * 缓存SN对应的信息
     *
     * @param sn sn
     * @param vo 设备连接
     */
    public void cacheDeviceInfo(String sn, TokenDeviceVo vo) {
        //设备相关信息缓存
        String key = getSnKey(sn);
        redisUtils.set(key, vo, EXPIRE_TIME);
    }

    /**
     * 更新SN对应的信息
     *
     * @param sn sn
     * @param vo 设备连接
     */
    public void updateDeviceInfo(String sn, TokenDeviceVo vo) {
        //设备相关信息缓存
        String key = getSnKey(sn);
        redisUtils.set(key, vo, EXPIRE_TIME);
    }

    /**
     * 延长设备在线的缓存
     *
     * @param sn sn
     */
    public void expireDeviceInfo(String sn) {
        //设备相关信息缓存
        String key = getSnKey(sn);
        redisUtils.expire(key, EXPIRE_TIME);
    }
    public boolean expireChannelSn(String parkCode, String channelCode){
        String key = getChannelSnKey(parkCode, channelCode);
        //有效期3分钟
        redisUtils.expire(key, 180L);
        return true;
    }

    /**
     * 获取设备信息
     *
     * @param sn sn
     * @return 设备连接
     */
    public TokenDeviceVo getDeviceInfo(String sn) {
        String key = getSnKey(sn);
        return redisUtils.get(key, TokenDeviceVo.class);
    }
    /**
     * 获取data信息
     * @param messageId 消息ID
     * @param timeOut 超时时间
     * @return 结果
     */
    public <T> T getDataFromRedis(String messageId, long timeOut, Class<T> clazz){
        long currentTimeMillis = System.currentTimeMillis();
        long lastTime = currentTimeMillis + timeOut;
        int n = 1;
        //查询redis中是否返回
        while (lastTime > currentTimeMillis) {
            T data = redisUtils.get(messageId, clazz);
            if (data != null) {
                log.info("第{}次从redis中读取到了key：{}响应的信息：{}", n, messageId, data);
                return data;
            }
            try {
                Thread.sleep(INTERVAL);
            } catch (InterruptedException e) {
                log.warn(String.valueOf(e.getMessage()), e);
            }
            currentTimeMillis = System.currentTimeMillis();
            n ++;
        }
        log.info("时限内未查询到key：{}响应的信息！", messageId);
        return null;
    }
    /**
     * 获取data信息
     *
     * @param messageId 消息ID
     * @param timeOut 超时时间
     * @return 结果
     */
    public ObjectResponse<String> getResponseFromRedis(String messageId, Long timeOut) {
        String key = RedisConstants.RESP_MSG_PROFILE + messageId;
        long currentTimeMillis = System.currentTimeMillis();
        long lastTime = currentTimeMillis + timeOut;
        int n = 1;
        //查询redis中是否返回
        while (lastTime > currentTimeMillis) {
            if (redisUtils.exists(key)) {
                ObjectResponse<String> data = redisUtils.get(key, new TypeReference<ObjectResponse<String>>() {
                });
                log.info("第{}次从redis中读取到了key：{}响应的信息：{}", n, key, data);
                return data;
            }
            try {
                Thread.sleep(INTERVAL);
            } catch (InterruptedException e) {
                log.warn(String.valueOf(e.getMessage()), e);
            }
            currentTimeMillis = System.currentTimeMillis();
            n++;
        }
        log.info("时限内未查询到key：{}响应的信息！", key);
        return null;
    }

    /**
     * 根据车场编号和通道编号获取设备序列号
     *
     * @param parkCode 车场编号
     * @param channelCode 通道编号
     * @return sn
     */
    public String getSerialNumber(String parkCode, String channelCode) {
        return redisUtils.get(getChannelSnKey(parkCode, channelCode), String.class);
    }

    /**
     * 保存当前通道一个相机的设备序列号
     *
     * @param parkCode 车场编号
     * @param channelCode 通道编号
     * @return 成功/失败
     */
    public boolean setChannelSn(String parkCode, String channelCode, String sn) {
        String key = getChannelSnKey(parkCode, channelCode);
        try {
            return redisUtils.set(key, sn, EXPIRE_TIME);
        } catch (Exception e) {
            log.error("<缓存操作-通道相机序列号> redis写入异常，key：{}，value：{}", key, sn);
            return false;
        }
    }

    /**
     * 更新心跳时间
     *
     * @param sn sn
     * @param connTime 连接时间
     * @return 无
     */
    public void updateLastConnTime(String sn, Long connTime) {
        //更新心跳时间
        redisUtils.hPut(getDeviceProfile() + RedisConstants.COMMON_HEARTBEAT_TIME, sn, connTime);
    }

    /**
     * 添加车场的连接设备信息到列表
     * @param parkCode 车场编号
     * @param parkConnectedDeviceVo 设备连接
     */
    public void addParkConnectList(String parkCode, ParkConnectedDeviceVo parkConnectedDeviceVo){
        String key = getParkConnectedListKey(parkCode);
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key, 0, -1, ParkConnectedDeviceVo.class);
        if (list.isEmpty()) {
            list = new ArrayList<>();
        }
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            // 检查是否有重复，如果有，则先移除旧的
            if (vo.getDeviceNo().equals(parkConnectedDeviceVo.getDeviceNo())) {
                redisUtils.lRemove(key, 0, vo);
                iterator.remove();
                continue;
            }
        }
        list.add(parkConnectedDeviceVo);
        log.info("[注册设备] 车场[{}]当前在线设备[{}]台，分别是[{}]", parkCode, list.size(), list);
        redisUtils.lRightPush(key, parkConnectedDeviceVo);
    }

    /**
     * 获取车场在线的相机列表
     * @param parkCode 车场编号
     * @return 连接设备列表
     */
    public List<ParkConnectedDeviceVo> getParkConnectList(String parkCode){
        String key = getParkConnectedListKey(parkCode);
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key, 0, -1, ParkConnectedDeviceVo.class);
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            TokenDeviceVo deviceInfo = getDeviceInfo(vo.getDeviceNo());
            if (deviceInfo == null || !parkCode.equals(deviceInfo.getParkCode())) {
                redisUtils.lRemove(key, 0, vo);
                iterator.remove();
                log.info("[获取最新在线列表] parkCode[{}], 移除不在线设备[{}]", parkCode, vo);
            }
        }
        log.info("[获取最新在线列表] parkCode[{}], 设备列表[{}]", parkCode, list);
        return list;
    }
}
