package com.icetech.park.service.down.pnc.impl;

import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.api.fee.ChargeService;
import com.icetech.cloudcenter.api.month.VipCarService;
import com.icetech.cloudcenter.api.month.VipRegionService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.basics.domain.entity.park.ParkChargeconfig;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.fee.domain.entity.vip.VipRegion;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.cloudcenter.domain.request.pnc.RegionBillContent;
import com.icetech.cloudcenter.domain.request.pnc.VipCarTypeRequest;
import com.icetech.cloudcenter.domain.response.VipTypeDto;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;

/**
 * VIP车辆类型下发
 */
@Slf4j
@Service
public class VipCarTypeServiceImpl extends AbstractService implements DownSendService {
    @Resource
    private PncDownHandle downHandle;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private VipCarService vipCarService;

    @Autowired
    private VipRegionService vipRegionService;

    @Autowired
    private ParkService parkService;

    @Autowired
    private ChargeService chargeService;

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                String.valueOf(sendRequest.getParkId()), String.class);
        if(version == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getDefaultMessage(TextConstant.TWO, "找不到车场版本"));
        }
        if(PncVersionEnum.getIndex(version) < PncVersionEnum.版本5.getIndex()) {
            log.info("获取到车场[{}]版本: [{}] -> [{}]", sendRequest.getParkId(), version,
                    PncVersionEnum.getIndex(version));
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getVersionMessage(version, PncVersionEnum.版本5.getVersion()));
        }

        Long vipTypeId = sendRequest.getServiceId();
        try {
            int type = Integer.parseInt(sendRequest.getInclude());
            VipCarTypeRequest request = null;
            // 删除时只传输车辆类型ID
            if (type == 3) {
                request = new VipCarTypeRequest();
                request.setVipCarTypeId(vipTypeId.intValue());
                request.setOperTime(System.currentTimeMillis());
            } else {
                ObjectResponse<VipTypeDto> vipTypeResp = vipCarService.getVipTypeById(vipTypeId.intValue());
                if (!ObjectResponse.isSuccess(vipTypeResp)) {
                    return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "获取VIP类型信息失败"));
                }
                request = buildRequest(vipTypeResp.getData());
            }
            request.setOperType(type);
            List<VipRegion> vipRegionList = vipRegionService.getVipRegionsListByVipTypeId(vipTypeId.intValue());
            if (CollectionUtils.isNotEmpty(vipRegionList)) {
                VipRegion vipRegion = vipRegionList.stream()
                        .filter(v -> v.getRegionId() == 0)
                        .findFirst().orElse(null);
                // 不是全车场
                if (vipRegion == null) {
                    request.setRegionRange(2);
                    List<RegionBillContent> contents = vipRegionList.stream()
                            .filter(Objects::nonNull)
                            .map(v -> RegionBillContent.builder()
                                    .regionCode(getRegionCode(v.getRegionId()))
                                    .regionBillTypeCode(getParkChargeConfigCode(v.getBillId()))
                                    .build())
                            .collect(Collectors.toList());
                    request.setRegionBillContent(contents);
                    request.setBilltypeCode(contents.stream()
                            .filter(Objects::nonNull)
                            .map(RegionBillContent::getRegionBillTypeCode)
                            .filter(StringUtils::isNotEmpty)
                            .findFirst()
                            .orElse(""));
                }else {
                    request.setRegionRange(1);
                    request.setBilltypeCode(getParkChargeConfigCode(vipRegion.getBillId()));
                }
            }
            String messageId = downHandle.signAndSend(sendRequest.getParkId(), DownServiceEnum.VIP车辆类型.getServiceName(), request, vipTypeId);
            if (messageId == null) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
            }
            return ObjectResponse.success();
        } catch (ResponseBodyException e) {
            log.warn("下发VIP车辆类型失败: {}, {}", e.getErrCode(), e.getMessage(), e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "下发VIP车辆类型失败"));
        } catch (Exception e) {
            log.error("下发VIP车辆类型失败", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "获取VIP车辆信息失败"));
        }
    }

    public String getRegionCode(Long id) {
        ObjectResponse<ParkRegion> regionResponse = parkService.getParkRegionById(id);
        if (ObjectResponse.isSuccess(regionResponse)) {
            ParkRegion parkRegion = regionResponse.getData();
            return parkRegion.getRegionCode();
        }
        return "";
    }

    public String getParkChargeConfigCode(Integer id) {
        ParkChargeconfig chargeConfig =  chargeService.getParkChargeConfigById(id);
        if (chargeConfig == null || StringUtils.isEmpty(chargeConfig.getBilltypecode())) {
            return "";
        }
        return chargeConfig.getBilltypecode();
    }

    private VipCarTypeRequest buildRequest(VipTypeDto vipType) {
        VipCarTypeRequest request = new VipCarTypeRequest();
        request.setOperTime(vipType.getUpdateTime().getTime() / 1000);
        request.setVipCarTypeId(vipType.getId());
        request.setVipCarTypeName(vipType.getName());
        request.setDiscountType(vipType.getType());
        request.setDiscountNumber(vipType.getAmount());
        request.setBilltypeCode(vipType.getBilltypecode());
        return request;
    }
}
