package com.icetech.park.service.down.pnc.impl;

import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.api.month.VipCarService;
import com.icetech.fee.dao.vip.VipRecordDao;
import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.fee.domain.vo.VipInfoDetail;
import com.icetech.fee.domain.entity.vip.VipPlate;
import com.icetech.fee.domain.entity.vip.VipRecord;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.cloudcenter.domain.request.pnc.BatchDelBizRequest;
import com.icetech.cloudcenter.domain.request.pnc.VipCarInfoRequest;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.AssertTools;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;

/**
 * VIP车辆信息下发
 */
@Slf4j
@Service
public class VipCarInfoServiceImpl extends AbstractService implements DownSendService {
    @Resource
    private PncDownHandle downHandle;
    @Resource
    private VipCarService vipCarService;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private VipRecordDao vipRecordDao;

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                String.valueOf(sendRequest.getParkId()), String.class);
        if(version == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getDefaultMessage(TextConstant.TWO, "找不到车场版本"));
        }
        if (PncVersionEnum.getIndex(version) < PncVersionEnum.版本5.getIndex()) {
            log.info("获取到车场[{}]版本: [{}] -> [{}]", sendRequest.getParkId(), version,
                    PncVersionEnum.getIndex(version));
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getVersionMessage(version, PncVersionEnum.版本5.getVersion()));
        }

        Long recordId = sendRequest.getServiceId();
        try {
            ObjectResponse<VipRecord> vipRecordResp = vipCarService.getOperatorRecordById(recordId);
            ObjectResponse.notError(vipRecordResp);
            VipRecord vipRecord = vipRecordResp.getData();
            VipCarInfoRequest request;
            if (vipRecord.getOperType() == 3) {
                request = new VipCarInfoRequest();
                request.setVipInfoId(vipRecord.getVipId());
            } else {
                ObjectResponse<VipInfoDetail> vipInfoResp = vipCarService.getVipInfoDetailById(vipRecord.getVipId());
                ObjectResponse.notError(vipInfoResp);
                VipInfoDetail vipInfo = vipInfoResp.getData();
                ObjectResponse<List<VipPlate>> vipPlatesResp = vipCarService.getVipPlatesByInfoId(vipInfo.getId());
                ObjectResponse.notError(vipPlatesResp);
                String plateNums = vipPlatesResp.getData().stream()
                        .map(VipPlate::getPlateNum)
                        .collect(Collectors.joining("/")).replace("、", "/");
                request = buildRequest(vipRecord, vipInfo);
                request.setPlateNums(plateNums);
            }

            request.setOperType(vipRecord.getOperType());
            String messageId = downHandle.signAndSend(vipRecord.getParkId().longValue(),
                    DownServiceEnum.VIP车辆信息.getServiceName(), request, recordId);
            if (messageId == null) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
            }
            return ObjectResponse.success();
        } catch (ResponseBodyException e) {
            log.warn("下发VIP车辆信息失败: {}, {}", e.getErrCode(), e.getMessage(), e);
            return ObjectResponse.failed(ERROR_410,  TextConstant.getDefaultMessage(TWO, "下发VIP车辆信息失败"));
        } catch (Exception e) {
            log.error("下发VIP车辆信息失败", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "获取VIP车辆信息失败"));
        }
    }

    @Override
    public ObjectResponse batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode){
        List<Long> ids = sendRequestList.stream().map(SendRequest::getServiceId).collect(Collectors.toList());
        List<VipRecord> list = vipRecordDao.getRecordListByIds(ids);
        if (CollectionUtils.isEmpty(list)) {
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }
        List<String> vipIdList = list.stream().map(VipRecord :: getVipId)
                .map(String :: valueOf).collect(Collectors.toList());
        BatchDelBizRequest batchDelBizRequest = new BatchDelBizRequest();
        batchDelBizRequest.setServiceType(3);
        batchDelBizRequest.setIdList(vipIdList);
        batchDelBizRequest.setServiceIds(ids);
        String messageId = downHandle.signAndSend(parkId, DownServiceEnum.批量删除业务数据.getServiceName(), batchDelBizRequest);
        AssertTools.notNull(messageId, CodeConstants.ERROR, "下发失败");
        return ObjectResponse.success();
    }

    private VipCarInfoRequest buildRequest(VipRecord vipRecord, VipInfoDetail vipInfo) {
        VipCarInfoRequest request = new VipCarInfoRequest();
        request.setOperTime(vipRecord.getOperTime().getTime() / 1000);
        request.setVipCarTypeId(vipInfo.getTypeId());
        request.setVipCarTypeName(vipInfo.getVipTypeName());
        request.setVipInfoId(vipInfo.getId());
        request.setStartDate(DateFormatUtils.format(vipInfo.getStartTime(), VipCarInfoRequest.DATE_PATTERN));
        request.setEndDate(DateFormatUtils.format(vipInfo.getEndTime(), VipCarInfoRequest.DATE_PATTERN));
        request.setOwner(vipInfo.getOwner() == null ? "" : vipInfo.getOwner());
        request.setPhone(vipInfo.getPhone() == null ? "" : vipInfo.getPhone());

        return request;
    }
}
