package com.icetech.park.service.down.pnc.impl;

import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.api.store.StoreCardService;
import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.fee.domain.entity.storecard.StoreCard;
import com.icetech.fee.domain.entity.storecard.StoreCardRecord;
import com.icetech.fee.domain.entity.storecard.StoreCardUsed;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.cloudcenter.domain.request.pnc.StoreCardRequest;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;

/**
 * 储值卡车辆信息下发
 */
@Slf4j
@Service
public class StoredCardServiceImpl extends AbstractService implements DownSendService {
    @Resource
    private PncDownHandle downHandle;
    @Resource
    private StoreCardService storeCardService;
    @Resource
    private RedisUtils redisUtils;

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                String.valueOf(sendRequest.getParkId()), String.class);
        if(version == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getDefaultMessage(TextConstant.TWO, "找不到车场版本"));
        }
        if(PncVersionEnum.getIndex(version) < PncVersionEnum.版本5.getIndex()) {
            log.info("获取到车场[{}]版本: [{}] -> [{}]", sendRequest.getParkId(), version,
                    PncVersionEnum.getIndex(version));
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getVersionMessage(version, PncVersionEnum.版本5.getVersion()));
        }

        Long serviceId = sendRequest.getServiceId();
        try {
            StoreCardRequest request = null;
            // include等于"use"时代表使用储值卡, 取储值卡使用记录表数据
            if("use".equals(sendRequest.getInclude())) {
                ObjectResponse<StoreCardUsed> usedResp = storeCardService.getStoreCardUsedById(serviceId);
                if (!ObjectResponse.isSuccess(usedResp)) {
                    return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "储值卡使用记录未找到"));
                }
                StoreCardUsed cardUsed = usedResp.getData();
                request = new StoreCardRequest();
                request.setOperType(4);
                request.setCardId(cardUsed.getCardId());
                request.setBalance(cardUsed.getCardAfterAmount().toString());
                request.setOperTime(cardUsed.getPayTime().getTime() / 1000);
            } else {
                ObjectResponse<StoreCardRecord> recordResp = storeCardService.getStoreCardRecordById(serviceId);
                if (!ObjectResponse.isSuccess(recordResp)) {
                    return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "储值卡操作记录未找到"));
                }
                StoreCardRecord record = recordResp.getData();
                int operaType = convertOperaType(record.getCardOperType());
                if (operaType == 3) {
                    request = new StoreCardRequest();
                } else {
                    ObjectResponse<StoreCard> storeCardResp = storeCardService.getStoreCardDetail(record.getCardId().intValue());
                    if (!ObjectResponse.isSuccess(recordResp)) {
                        return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "储值卡信息未找到"));
                    }
                    StoreCard storeCard = storeCardResp.getData();
                    request = buildRequest(storeCard);
                }
                request.setOperType(operaType);
                request.setCardId(record.getCardId());
                request.setPlateNums(record.getPlateNum().replace(",", "/"));
                request.setOperTime(record.getUpdateTime().getTime() / 1000);
            }

            String messageId = downHandle.signAndSend(sendRequest.getParkId(), DownServiceEnum.储值卡车辆信息.getServiceName(), request, serviceId);
            if (messageId == null) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
            }
            return ObjectResponse.success();
        } catch (ResponseBodyException e) {
            log.warn("下发储值卡操作记录失败: {}, {}", e.getErrCode(), e.getMessage(), e);
            return ObjectResponse.failed(ERROR_410,  TextConstant.getUnEstablishConnection());
        } catch (Exception e) {
            log.error("下发储值卡操作记录失败", e);
            return ObjectResponse.failed(ERROR_410, "下发储值卡操作记录失败");
        }
    }

    private StoreCardRequest buildRequest(StoreCard storeCard) {
        StoreCardRequest request = new StoreCardRequest();
        request.setCardNum(storeCard.getCardsNum());
        request.setOwner(storeCard.getCardOwner());
        request.setPhone(storeCard.getPhone());
        request.setAddress(storeCard.getAddress());
        request.setStartDate(DateFormatUtils.format(storeCard.getStartTime(), StoreCardRequest.DATE_PATTERN));
        request.setEndDate(DateFormatUtils.format(storeCard.getEndTime(), StoreCardRequest.DATE_PATTERN));
        request.setBalance(storeCard.getBalance().toString());
        return request;
    }

    private int convertOperaType(int operaType) {
        // 充值及编辑统一按照编辑下发
        if (operaType == 3) {
            return 2;
        }
        if (operaType == 4) {
            return 3;
        }

        return operaType;
    }
}
