package com.icetech.park.service.down.pnc.impl;

import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.api.OrderEnterService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.park.ParkVisitService;
import com.icetech.park.dao.park.ParkVisitDao;
import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.domain.constants.ParkVisitMoreType;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.park.domain.entity.park.ParkVisit;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.request.OrderEnterRequest;
import com.icetech.cloudcenter.domain.request.pnc.BatchDelBizRequest;
import com.icetech.cloudcenter.domain.response.ParkVisitDto;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.park.service.handle.PublicHandle;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.AssertTools;
import com.icetech.common.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;

@Service
@Slf4j
public class OrderEnterServiceImpl implements OrderEnterService , DownSendService {

    @Autowired
    private PncDownHandle downHandle;
    @Autowired
    private PublicHandle publicHandle;
    @Autowired
    private ParkVisitService parkVisitService;
    @Autowired
    private ParkService parkService;

    @Autowired
    private ParkVisitDao parkVisitDao;

    @Override
    public ObjectResponse orderEnter(OrderEnterRequest orderEnterRequest) {
        try {
            String parkCode = orderEnterRequest.getParkCode();
            orderEnterRequest.setParkCode(null);
            String messageId = downHandle.signAndSend(parkCode, DownServiceEnum.车辆预约.getServiceName(),
                    orderEnterRequest);
            AssertTools.notNull(messageId, ERROR_410, "订单入场失败");
            return ObjectResponse.success();
        }catch (ResponseBodyException responseBodyException){
            return ObjectResponse.failed(responseBodyException.getErrCode());
        }catch (Exception e){
            log.warn(String.valueOf(e.getMessage()), e);
            return ObjectResponse.failed(CodeConstants.ERROR);
        }

    }

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        if (StringUtils.isNotEmpty(sendRequest.getInclude()) && TWO.equals(sendRequest.getInclude())) {
            return batchSend(Collections.singletonList(sendRequest),sendRequest.getParkId(), null);
        }
        try {
            ObjectResponse<ParkVisitDto> objectResponse = parkVisitService.selectVisitByid(sendRequest.getServiceId());
            if (!ObjectResponse.isSuccess(objectResponse)) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO,"未找到预约车信息"));
            }
            if(ObjectResponse.isSuccess(objectResponse)){
                ParkVisitDto parkVisitDto = objectResponse.getData();
                String[] split = parkVisitDto.getPlateNums().split(",");
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                ObjectResponse<ParkConfig> parkConfig = parkService.getParkConfig(parkVisitDto.getParkCode());
                if (!ObjectResponse.isSuccess(parkConfig)) {
                    return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO,"车场配置未找到"));
                }
                //是否收费
                Integer visitIscharge = 0;
                if(ObjectResponse.isSuccess(parkConfig)){
                    visitIscharge = parkConfig.getData().getVisitIscharge();
                }
                for (int i = 0; i < split.length; i++) {
                    OrderEnterRequest orderEnterRequest = new OrderEnterRequest();
                    orderEnterRequest.setVisitId(parkVisitDto.getId());
                    orderEnterRequest.setParkCode(null);
                    orderEnterRequest.setIsFree(visitIscharge == 1 ? 0 : 1);
                    orderEnterRequest.setPlateNum(split[i]);
                    orderEnterRequest.setStartTime(simpleDateFormat.parse(parkVisitDto.getStartTime()).getTime()/1000);
                    orderEnterRequest.setEndTime(simpleDateFormat.parse(parkVisitDto.getEndTime()).getTime()/1000);
                    orderEnterRequest.setIsAllowManyTimes(parkVisitDto.getInoutMore());
                    if (ParkVisitMoreType.CUSTOM.getType().equals(parkVisitDto.getInoutMore())){
                        orderEnterRequest.setAllowVisitNum(parkVisitDto.getInoutNum());
                    }
                    System.out.println("请求数据"+JsonUtils.toString(orderEnterRequest));
                    String messageId = downHandle.signAndSend(parkVisitDto.getParkCode(), DownServiceEnum.车辆预约.getServiceName(),
                            orderEnterRequest,sendRequest.getServiceId());
                    if (StringUtils.isEmpty(messageId)) {
                        return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());

                    }
                }
            }
        }catch (Exception e){
            log.warn(String.valueOf(e.getMessage()), e);
            return ObjectResponse.failed(ERROR_410, "车辆预约下发失败");
        }
        return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
    }

    @Override
    public ObjectResponse batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode) {
        List<Long> ids = sendRequestList.stream()
                .map(SendRequest::getServiceId)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        List<ParkVisit> lists = parkVisitDao.findAllByIds(ids, Boolean.TRUE);
        if (CollectionUtils.isEmpty(lists)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "未找到访客车辆信息"));
        }
        BatchDelBizRequest batchDelBizRequest = new BatchDelBizRequest();
        batchDelBizRequest.setServiceType(5);
        batchDelBizRequest.setServiceIds(ids);
        batchDelBizRequest.setIdList(lists.stream()
                .map(a -> String.valueOf(a.getId()))
                .collect(Collectors.toList()));
        String messageId = downHandle.signAndSend(parkId, DownServiceEnum.批量删除业务数据.getServiceName(), batchDelBizRequest);
        if (StringUtils.isEmpty(messageId)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TWO, "访客车辆批量下发失败"));
        }
        return ObjectResponse.success();
    }

}
