package com.icetech.park.service.down.pnc.impl;

import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.request.pnc.IssuedCardRequest;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.park.domain.entity.more.MoreMonthInfo;
import com.icetech.park.domain.entity.more.MoreMonthPark;
import com.icetech.park.domain.entity.more.MoreMonthPlate;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.park.service.more.MoreMonthInfoService;
import com.icetech.park.service.more.MoreMonthParkService;
import com.icetech.park.service.more.MoreMonthPlateService;
import com.icetech.third.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;

/**
 * 多车场月卡下发
 */
@Service
@Slf4j
public class MoreMonthCardServiceImpl implements DownSendService {

    @Autowired
    private PncDownHandle downHandle;

    @Autowired
    private MoreMonthInfoService moreMonthInfoService;

    @Autowired
    private MoreMonthParkService moreMonthParkService;

    @Autowired
    private MoreMonthPlateService moreMonthPlateService;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    /**
     * 多车场月卡下发key前缀
     */
    public static final String MORE_MONTH_CARD_PREFIX = "more:month:card:";

    /**
     * 最多发送10次，大约需要24小时41分钟
     */
    private static final long EXPIRE_TIME_SEND_DEVICES_ = 24 * 3600 + 41 * 60;

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        Long serviceId = sendRequest.getServiceId();
        // 1.获取多车场月卡信息
        MoreMonthInfo moreMonthInfo = moreMonthInfoService.getById(serviceId);
        if (moreMonthInfo == null) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.TWO, "多车场月卡信息不存在"));
        }
        if (StringUtils.isEmpty(sendRequest.getInclude())) {
            if (moreMonthInfo.getCardStatus() == 3) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.TWO, "多车场月卡信息不存在"));
            }
        }
        // 2.获取多车场月卡关联车场信息
        List<MoreMonthPark> monthParks = moreMonthParkService.getMonthParkByMoreMonthId(serviceId);
        if (CollectionUtils.isEmpty(monthParks)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.TWO, "多车场月卡没有关联车场"));
        }
        // 3.获取多车场月卡关联车牌信息
        List<MoreMonthPlate> moreMonthPlates = moreMonthPlateService.getMoreMonthPlateByMoreMonthId(serviceId);
        if (CollectionUtils.isEmpty(moreMonthPlates)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.TWO, "多车场月卡没有关联车牌号"));
        }
        String plateNums = moreMonthPlates.stream()
                .map(MoreMonthPlate::getPlateNum)
                .filter(Objects::nonNull)
                .collect(Collectors.joining(","));
        // 4.下发信息组装
        IssuedCardRequest request = new IssuedCardRequest();
        request.setProdName("多车场月卡");
        request.setProdDuration(1);
        request.setCardId(String.valueOf(serviceId));
        request.setPlateNum(plateNums);
        request.setPhone(moreMonthInfo.getPhone());
        request.setCardOwner(moreMonthInfo.getCardOwner());
        request.setCardProperty(moreMonthInfo.getCardProperty());
        request.setStartDate(DateFormatUtils.format(moreMonthInfo.getStartTime(), "yyyy-MM-dd"));
        request.setEndDate(DateFormatUtils.format(moreMonthInfo.getEndTime(), "yyyy-MM-dd"));
        request.setCardType(1);
        request.setCardOperType(1);
        request.setPlotCount(moreMonthInfo.getPlotCount());
        request.setPayMoney("0.00");
        request.setOperAccount(moreMonthInfo.getOperAccount());
        request.setOperTime(moreMonthInfo.getCreateTime().getTime() / 1000);
        // 5.下发删除逻辑处理
        if (StringUtils.isNotEmpty(sendRequest.getInclude())) {
            request.setCardOperType(4);
        }
        log.info("[端网云]多车场月卡下发参数[{}]", JsonUtils.toJson(request));
        // 6.设置缓存
        List<String> parkIds = monthParks.stream()
                .map(MoreMonthPark::getParkId)
                .filter(Objects::nonNull)
                .map(String::valueOf)
                .collect(Collectors.toList());
        String key = MORE_MONTH_CARD_PREFIX + sendRequest.getTaskId();
        if (!stringRedisTemplate.hasKey(key)) {
            stringRedisTemplate.opsForSet().add(key, parkIds.toArray(new String[0]));
            stringRedisTemplate.expire(key, EXPIRE_TIME_SEND_DEVICES_, TimeUnit.SECONDS);
        }
        // 7.多车场下发
        for (MoreMonthPark monthPark : monthParks) {
            String messageId = downHandle.signAndSend(monthPark.getParkId(), DownServiceEnum.多车场月卡.getServiceName(), request, serviceId, null, sendRequest.getTaskId());
            if (StringUtils.isEmpty(messageId)) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
            }
        }
        return ObjectResponse.success();
    }
}
