package com.icetech.park.service.down.pnc.impl;

import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.api.lcd.LedService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.domain.entity.led.LedConfig;
import com.icetech.park.domain.entity.led.LedShow;
import com.icetech.park.domain.entity.led.LedSound;
import com.icetech.park.domain.entity.led.LedTips;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.cloudcenter.domain.request.pnc.LedConfigRequest;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.NumberUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;
import static com.icetech.cloudcenter.domain.request.pnc.LedConfigRequest.DisplayConfig;
import static com.icetech.cloudcenter.domain.request.pnc.LedConfigRequest.SoundConfig;
import static com.icetech.cloudcenter.domain.request.pnc.LedConfigRequest.SpecialDisplayConfig;

/**
 * LED自定义语音屏显下发
 */
@Slf4j
@Service
public class LedsoundConfigServiceImpl extends AbstractService implements DownSendService {
    @Resource
    private PncDownHandle downHandle;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private LedService ledService;
    @Resource
    private ParkService parkService;
    @Resource
    private ParkRegionDao parkRegionDao;

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                String.valueOf(sendRequest.getParkId()), String.class);
        if(version == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getDefaultMessage(TextConstant.TWO,"找不到车场版本"));
        }
        if(PncVersionEnum.getIndex(version) < PncVersionEnum.版本5.getIndex()) {
            log.info("获取到车场[{}]版本: [{}] -> [{}]", sendRequest.getParkId(), version,
                    PncVersionEnum.getIndex(version));
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getVersionMessage(version, PncVersionEnum.版本5.getVersion()));
        }

        Long channelId = sendRequest.getServiceId();
        try {
            ObjectResponse<ParkInoutdevice> channelResp = parkService.getInoutDeviceById(channelId);
            ObjectResponse.notError(channelResp);
            ParkInoutdevice channel = channelResp.getData();
            LedConfigRequest request = buildRequest(channel);
            List<DisplayConfig> displayConfigs = buildDisplayConfigs(channel);
            request.setLedConfig(displayConfigs);
            List<SpecialDisplayConfig> specialDisplayConfigs = buildSpecialDisplayConfig(channelId);
            request.setLedSpecial(specialDisplayConfigs);
            List<SoundConfig> soundConfigs = buildSoundConfig(channelId, version);
            request.setSoundConfig(soundConfigs);

            String messageId = downHandle.signAndSend(sendRequest.getParkId(), DownServiceEnum.LED语音屏显.getServiceName(), request, channelId);
            if (messageId == null) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
            }
            return ObjectResponse.success();
        } catch (ResponseBodyException e) {
            log.warn("下发LED屏显语音失败: {}, {}", e.getErrCode(), e.getMessage(), e);
            return ObjectResponse.failed(ERROR_410,TextConstant.getDefaultMessage(TWO,"下发LED屏显语音失败"));
        } catch (Exception e) {
            log.error("下发LED屏显语音失败", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
        }
    }

    private LedConfigRequest buildRequest(ParkInoutdevice channel) {
        ObjectResponse<LedConfig> soundConfigResp = ledService.getLedConfigByChannel(channel.getId());
        LedConfig ledConfig = soundConfigResp.getData();
        if (ledConfig == null) ledConfig = ledService.getLedConfigByChannel(null).getData();

        LedConfigRequest request = new LedConfigRequest();
        request.setInandoutCode(channel.getInandoutCode());
        if (NumberUtils.toPrimitive(channel.getLedcardType()) == 5 && channel.getDisplayTerminal() == 2) {
            request.setIsSupportTTS(1);
        } else {
            if (channel.getTtsType() == 1) {
                request.setIsSupportTTS(1);
            } else {
                request.setIsSupportTTS(2);
            }
        }
        request.setCtrlCardModel(channel.getLedcardType());
        request.setLedLineNum(ledConfig.getLedLineNum());
        request.setLedRemainDaysMc(ledConfig.getLedRemainDaysMc());
        request.setLedExpireDaysMc(ledConfig.getLedExpireDaysMc());
        request.setLedBlackCar(ledConfig.getLedBlackCar());
        request.setLedRestoreDefaultTime(ledConfig.getLedRestoreDefaultTime());
        request.setLedQrcodeRule(ledConfig.getLedQrcodeRule());
        request.setVolumeValue(ledConfig.getVolumeValue());
        request.setQuietHoursSwitch(ledConfig.getQuietHoursSwitch());
        request.setQuietStartTime(ledConfig.formatQuietStartTime());
        request.setQuietEndTime(ledConfig.formatQuietEndTime());
        request.setQuietVolumeValue(ledConfig.getQuietVolumeValue());
        request.setLimitType(ledConfig.getLimitType());
        request.setLimitDriveNum4(ledConfig.getLimitDriveNum4());
        request.setLimitDriveNum1(ledConfig.getLimitDriveNum1());
        request.setLimitDriveNum3(ledConfig.getLimitDriveNum3());
        request.setLimitDriveNum2(ledConfig.getLimitDriveNum2());
        request.setLimitDriveNum5(ledConfig.getLimitDriveNum5());
        if (ledConfig.getRegionFreeSpace() == null
                || Integer.valueOf(0).equals(ledConfig.getRegionFreeSpace())) {
            request.setRegionFreeSpace(String.valueOf(0));
        } else {
            ParkRegion parkRegion = parkRegionDao.selectById(ledConfig.getRegionFreeSpace());
            if (parkRegion != null) {
                request.setRegionFreeSpace(parkRegion.getRegionCode());
            } else {
                request.setRegionFreeSpace(String.valueOf(0));
            }
        }
        return request;
    }

    private List<DisplayConfig> buildDisplayConfigs(ParkInoutdevice channel) {
        Map<String, List<LedShow>> ledMap = ledService.getLedShowMapByChannel(channel.getId());
        if (ledMap.isEmpty()) return null;
        int channelType = NumberUtils.toPrimitive(channel.getInandoutType());
        int otherChannelType = -1;
        String[] otherDisplayTypes = null;
        if (channelType == 1) {
            otherChannelType = 2;
            otherDisplayTypes = new String[] {String.valueOf(LedShow.DisplayTypeEnum.出口空闲显示.type), String.valueOf(LedShow.DisplayTypeEnum.出场显示.type)};
        } else {
            otherChannelType = 1;
            otherDisplayTypes = new String[] {String.valueOf(LedShow.DisplayTypeEnum.入口空闲显示.type), String.valueOf(LedShow.DisplayTypeEnum.入场显示.type)};
        }
        ObjectResponse<List<ParkInoutdevice>> otherChannels = parkService.getChannelByType(channel.getParkId(), otherChannelType);
        Long otherChannelId = Optional.ofNullable(otherChannels)
                .map(ObjectResponse::getData).filter(CollectionUtils::isNotEmpty)
                .map(list -> list.get(0)).map(ParkInoutdevice::getId).orElse(null);
        Map<String, List<LedShow>> otherLedMap = ledService.getLedShowMapByChannel(otherChannelId);
        for (int i = 0; i < otherDisplayTypes.length; i++) {
            String otherDisplayType = otherDisplayTypes[i];
            ledMap.put(otherDisplayType, otherLedMap.get(otherDisplayType));
        }

        return ledMap.values().stream().filter(Objects::nonNull)
                .flatMap(Collection::stream).filter(Objects::nonNull)
                .map(led -> {
                    DisplayConfig displayConfig = new DisplayConfig();
                    displayConfig.setDisplayType(led.getDisplayType());
                    displayConfig.setRowNum(led.getRowNum());
                    displayConfig.setLedColor(led.getLedColor());
                    displayConfig.setFontSize(null);
                    displayConfig.setAlignmentMode(null);
                    displayConfig.setDynamicContent(led.getDynamicContent());
                    displayConfig.setCustomContent(led.getCustomContent());
                    return displayConfig;
                }).collect(Collectors.toList());
    }

    private List<SpecialDisplayConfig> buildSpecialDisplayConfig(long channelId) {
        ObjectResponse<LedTips> specialLedResp = ledService.getLedTipsByChannel(channelId);
        LedTips specialLed = specialLedResp.getData();
        if (specialLed == null) specialLed = ledService.getLedTipsByChannel(null).getData();
        SpecialDisplayConfig displayConfig = new SpecialDisplayConfig();
        displayConfig.setEnterTmpSpecial(specialLed.getEnterTmpSpecial());
        displayConfig.setEnterMonthVip(specialLed.getEnterMonthVip());
        displayConfig.setEnterNotAllowTmpRun(specialLed.getEnterNotAllowTmpRun());
        displayConfig.setEnterMultiParkCar(specialLed.getEnterMultiParkCar());
        displayConfig.setEnterBlackCar(specialLed.getEnterBlackCar());
        displayConfig.setEnterMonthExpire(specialLed.getEnterMonthExpire());
        displayConfig.setEnterNoParkingSpace(specialLed.getEnterNoParkingSpace());
        displayConfig.setEnterNotAllowNocarRun(specialLed.getEnterNotAllowNocarRun());
        displayConfig.setExitPay(specialLed.getExitPay());
        displayConfig.setExitOpenGate(specialLed.getExitOpenGate());
        displayConfig.setExitNoEnterinfo(specialLed.getExitNoEnterinfo());
        displayConfig.setExitNoCarplate(specialLed.getExitNoCarplate());
        displayConfig.setExitNotAllowRun(specialLed.getExitNotAllowRun());
        return Collections.singletonList(displayConfig);
    }

    private List<SoundConfig> buildSoundConfig(long channelId, String version) {
        Map<String, LedSound> soundMap = ledService.getLedSoundMapByChannel(channelId);
        if (MapUtils.isEmpty(soundMap)) {
            soundMap = ledService.getLedSoundMapByChannel(null);
        }

        return soundMap.values().stream().filter(Objects::nonNull)
                .filter(ledSound -> PncVersionEnum.getIndex(version) >= PncVersionEnum.版本2500.getIndex()
                        || (LedSound.SoundTypeEnum.入口欠费补缴.type != ledSound.getSoundType()
                            && LedSound.SoundTypeEnum.出口欠费补缴.type != ledSound.getSoundType()
                            && LedSound.SoundTypeEnum.入口虚假车牌.type != ledSound.getSoundType()
                            && LedSound.SoundTypeEnum.出口虚假车牌.type != ledSound.getSoundType()))
                .map(sound -> {
                    SoundConfig soundConfig = new SoundConfig();
                    soundConfig.setSoundType(sound.getSoundType());
                    soundConfig.setDynamicContent(sound.getDynamicContent());
                    soundConfig.setCustomContent(sound.getCustomContent());
                    return soundConfig;
        }).collect(Collectors.toList());
    }

}
